import Button from 'carbon-react/esm/components/button';
import Dialog from 'carbon-react/esm/components/dialog';
import DialogFullScreen from 'carbon-react/esm/components/dialog-full-screen';
import Form from 'carbon-react/esm/components/form';
import Loader from 'carbon-react/esm/components/loader';
import { isEqual, isFunction } from 'lodash';
import React from 'react';
import { connect } from 'react-redux';
import * as xtremRedux from '../../../redux';
import { CollectionValue } from '../../../service/collection-data-service';
import { CollectionFieldTypes } from '../../../service/collection-data-types';
import { mergeGraphQLFilters } from '../../../service/filter-service';
import { buildSearchBoxFilter, convertFilterDecoratorToGraphQLFilter } from '../../../service/graphql-query-builder';
import { fetchReferenceFieldData } from '../../../service/graphql-service';
import { removeEdges } from '../../../service/graphql-utils';
import { localize } from '../../../service/i18n-service';
import { showToast } from '../../../service/toast-service';
import { ContextType } from '../../../types';
import { xtremConsole } from '../../../utils/console';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { getFieldTitle } from '../../field/carbon-helpers';
import { getReferenceOrderBy } from '../../field/reference/reference-utils';
import { AsyncTableComponent } from '../../field/table/async-table-component';
import { text } from '../../nested-fields';
import { CLASS_NAME_LOOKUP_DIALOG } from './lookup-dialog-utils';
import { LookupCreateNewItem } from './lookup-create-new-item';
import { splitValueToMergedValue } from '../../../utils/transformers';
export class LookupDialog extends React.Component {
    constructor(props) {
        super(props);
        this.onClose = () => {
            this.props.onCloseLookupDialog();
            this.props.closeLookupDialog();
        };
        this.onRowClick = (selected) => {
            if (!this.props.isMultiSelect) {
                this.props.onSelectionFinished([selected]);
            }
        };
        this.onFinishSelection = () => {
            const { value } = this.state;
            if (value) {
                this.props.onSelectionFinished(this.state.selectedItems.map(r => value.getRawRecord({
                    id: r,
                    cleanMetadata: true,
                    temporaryRecords: this.getTemporaryRecords(),
                })));
            }
            else {
                this.props.onSelectionFinished([]);
            }
        };
        this.loadTableData = async (elementId, appliedSearchText = this.state.appliedSearchText) => {
            if (this.props.value) {
                return this.props.value;
            }
            try {
                const fieldProperties = {
                    ...this.props.fieldProperties,
                    filter: convertFilterDecoratorToGraphQLFilter(this.props.screenDefinition, this.props.fieldProperties?.filter, this.props.recordContext),
                };
                let filter;
                if (appliedSearchText) {
                    filter = buildSearchBoxFilter(fieldProperties, this.props.nodeTypes, this.props.locale, CollectionFieldTypes.LOOKUP_DIALOG, appliedSearchText);
                    if (fieldProperties.filter && !isFunction(fieldProperties.filter)) {
                        filter = mergeGraphQLFilters([filter, fieldProperties.filter]);
                    }
                }
                else if (fieldProperties.filter && !isFunction(fieldProperties.filter)) {
                    filter = fieldProperties.filter;
                }
                const orderBy = getReferenceOrderBy(fieldProperties);
                const result = await fetchReferenceFieldData({
                    fieldProperties,
                    screenId: this.props.screenId,
                    elementId,
                    valueField: this.props.valueField || fieldProperties.valueField,
                    filter,
                    orderBy,
                    parentElementId: this.props.parentElementId,
                    recordContext: this.props.recordContext,
                    contextNode: this.props.contextNode,
                    pageSize: 10,
                    level: this.props.level,
                });
                const columns = fieldProperties.columns || [];
                return new CollectionValue({
                    bind: fieldProperties.bind,
                    screenId: this.props.screenId,
                    elementId: this.props.fieldId,
                    isTransient: !!fieldProperties.isTransient,
                    hasNextPage: result.query?.pageInfo?.hasNextPage || false,
                    orderBy: [orderBy],
                    columnDefinitions: [columns],
                    nodeTypes: this.props.nodeTypes,
                    nodes: [fieldProperties.node],
                    filter: [filter],
                    initialValues: removeEdges(result, true, true)?.data ?? [],
                    fieldType: CollectionFieldTypes.LOOKUP_DIALOG,
                    parentElementId: this.props.parentElementId,
                    recordContext: this.props.recordContext,
                    contextNode: this.props.contextNode,
                    referenceLookupContextLevel: this.props.level,
                });
            }
            catch (err) {
                // eslint-disable-next-line no-console
                xtremConsole.warn(err);
                showToast(localize('@sage/xtrem-ui/lookup-dialog-failed-fetch', 'Failed to fetch options'), {
                    type: 'warning',
                });
                this.props.onSelectionFinished(undefined);
                return null;
            }
        };
        this.setFieldProperties = (_elementId, fieldProperties) => {
            const newSelection = [...(fieldProperties.selectedRecords || [])].sort();
            const currentSelection = [...this.state.selectedItems].sort();
            if (!isEqual(currentSelection, newSelection)) {
                this.setState({ selectedItems: newSelection });
            }
        };
        this.getColumns = () => {
            return this.props.fieldProperties.columns
                ? [...this.props.fieldProperties.columns].map((column) => {
                    const properties = { ...column.properties };
                    properties.isReadOnly = true;
                    properties.size = 'small';
                    return { ...column, properties };
                })
                : [text({ bind: this.props.fieldProperties.valueField })];
        };
        this.onKeyDown = (event) => {
            if (this.props.isMultiSelect && this.state.value && event.key === 'Enter') {
                this.onFinishSelection();
            }
        };
        this.getDialogTitle = () => resolveByValue({
            screenId: this.props.screenId,
            propertyValue: this.props.fieldProperties.lookupDialogTitle,
            skipHexFormat: true,
            rowValue: splitValueToMergedValue(this.props.recordContext || {}),
        }) ||
            getFieldTitle(this.props.screenId, this.props.fieldProperties, this.props.recordContext) ||
            localize('@sage/xtrem-ui/lookup-dialog-dialog-title', 'Selection');
        this.getTemporaryRecords = () => {
            return resolveByValue({
                propertyValue: this.props.fieldProperties.additionalLookupRecords,
                screenId: this.props.screenId,
                rowValue: null,
                fieldValue: null,
                skipHexFormat: true,
            });
        };
        this.state = {
            value: props.value || null,
            appliedSearchText: props.searchText || '',
            selectedItems: props.selectedRecordId
                ? props.selectedRecordId instanceof Array
                    ? [...props.selectedRecordId]
                    : [props.selectedRecordId]
                : [],
        };
    }
    async componentDidMount() {
        if (this.props.isOpen) {
            try {
                await this.populateDialog();
            }
            catch (err) {
                // Intentionally left empty.
            }
        }
    }
    async componentDidUpdate(prevProps) {
        const isClosing = prevProps.isOpen && !this.props.isOpen;
        const isOpening = !prevProps.isOpen && this.props.isOpen;
        if (isClosing) {
            this.props.onCloseLookupDialog();
            // eslint-disable-next-line react/no-did-update-set-state
            this.setState({ value: null });
        }
        if (isOpening) {
            await this.populateDialog();
        }
    }
    // Quick fix until https://github.com/Sage/carbon/pull/4323 is released and we are upgraded to it
    componentDidCatch(err) {
        xtremConsole.warn(err);
    }
    componentWillUnmount() {
        this.props.onCloseLookupDialog();
    }
    async populateDialog() {
        const appliedSearchText = this.props.searchText || '';
        this.setState({
            appliedSearchText: this.props.searchText || '',
            selectedItems: this.props.selectedRecordId
                ? this.props.selectedRecordId instanceof Array
                    ? [...this.props.selectedRecordId]
                    : [this.props.selectedRecordId]
                : [],
        });
        this.props.onOpenLookupDialog({
            onClose: this.onClose,
            elementId: this.props.parentElementId || this.props.fieldId,
            screenId: this.props.screenId,
            dialogTitle: this.getDialogTitle(),
            nestedField: this.props.parentElementId ? this.props.fieldId : undefined,
        });
        // If a starting value is not provided, we call the server and create a collection value instance
        const value = await this.loadTableData(this.props.fieldId, appliedSearchText);
        if (!value) {
            return;
        }
        this.setState({ value });
        if (value.getData({ cleanMetadata: false, temporaryRecords: this.getTemporaryRecords() }).length === 0) {
            setTimeout(() => {
                const closeButton = document.querySelector('button[data-element="close"]');
                closeButton?.focus();
            }, 500);
        }
    }
    renderContent() {
        const isDeviceLessThanM = (this.props.browser && this.props.browser.lessThan.m) || false;
        const columns = this.getColumns();
        const data = this.state.value.getData({ cleanMetadata: false, temporaryRecords: this.getTemporaryRecords() });
        const hasData = data.length > 0;
        const fieldProps = {
            ...this.props.fieldProperties,
            canSelect: false,
            columns: [...columns],
            hasSearchBoxMobile: true,
            helperText: undefined,
            isHelperTextHidden: true,
            isTransient: this.props.fieldProperties.isTransient,
            isHidden: false,
            isTitleHidden: true,
            pageSize: 10,
            cardView: isDeviceLessThanM,
            orderBy: getReferenceOrderBy(this.props.fieldProperties),
            ...(hasData && { selectedRecords: [this.props.selectedRecordId ?? data[0]._id] }),
        };
        delete fieldProps.title;
        if (this.props.isMultiSelect) {
            fieldProps.canSelect = true;
            fieldProps.selectedRecords = hasData ? this.state.selectedItems : [];
        }
        return (React.createElement(Form, { stickyFooter: !!this.props.isMultiSelect, buttonAlignment: "right", leftSideButtons: this.props.isMultiSelect && (React.createElement(Button, { onClick: this.onClose, "data-testid": "e-lookup-dialog-button-cancel" }, localize('@sage/xtrem-ui/cancel', 'Cancel'))), rightSideButtons: this.props.isMultiSelect && (React.createElement(Button, { onClick: this.onFinishSelection, "data-testid": "e-lookup-dialog-button-select", buttonType: "primary" }, localize('@sage/xtrem-ui/lookup-dialog-confirm-select', 'Select'))) },
            this.props.isLinkCreateNewText && (React.createElement(LookupCreateNewItem, { node: fieldProps.node, createTunnelLinkText: this.props.createTunnelLinkText, onCreateNewItemLinkClick: this.props.onCreateNewItemLinkClick, screenId: this.props.screenId, isDeviceLessThanM: isDeviceLessThanM })),
            React.createElement(AsyncTableComponent, { accessBindings: this.props.screenDefinition.accessBindings || {}, additionalLookupRecords: this.getTemporaryRecords, browser: this.props.browser, contextType: ContextType.dialog, dataTypes: this.props.dataTypes, elementId: this.props.fieldId, enableMobileLoadMore: true, enumTypes: this.props.enumTypes, fieldProperties: fieldProps, isUsingInfiniteScroll: true, item: { $bind: this.props.fieldId }, locale: this.props.locale, lookupSelectionMode: this.props.isMultiSelect ? 'multiple' : 'single', nodeTypes: this.props.nodeTypes, numberOfVisibleRows: 10, onRowClick: this.onRowClick, recordContext: this.props.recordContext, screenId: this.props.screenId, searchText: this.props.searchText, setFieldProperties: this.setFieldProperties, tableViews: {}, validationErrors: [], value: this.state.value })));
    }
    render() {
        const isDeviceLessThanM = (this.props.browser && this.props.browser.lessThan.m) || false;
        const dialogProps = {
            open: this.props.isOpen,
            children: (React.createElement("div", { className: CLASS_NAME_LOOKUP_DIALOG, "data-testid": "e-lookup-dialog" },
                React.createElement("div", { className: "e-dialog-content" },
                    React.createElement("div", { className: "e-dialog-body", "data-testid": "e-dialog-body", onKeyDown: this.onKeyDown },
                        !this.state.value && (React.createElement("div", { className: "e-lookup-dialog-loader" },
                            React.createElement(Loader, { size: "large" }))),
                        this.state.value && this.renderContent())))),
            onCancel: this.onClose,
            showCloseIcon: true,
            size: 'large',
            title: this.getDialogTitle(),
            disableAutoFocus: true,
        };
        return isDeviceLessThanM ? React.createElement(DialogFullScreen, { ...dialogProps }) : React.createElement(Dialog, { ...dialogProps });
    }
}
const extendedMapStateToProps = (state, props) => {
    const screenDefinition = state.screenDefinitions[props.screenId];
    // If the parent of reference-lookup-dialog is a nested field the field properties will come as a prop.
    const fieldProperties = props.fieldProperties || screenDefinition.metadata.uiComponentProperties[props.fieldId];
    return {
        ...props,
        browser: state.browser,
        dataTypes: state.dataTypes,
        enumTypes: state.enumTypes,
        fieldProperties,
        locale: state.applicationContext?.locale || 'base',
        nodeTypes: state.nodeTypes,
        onCloseLookupDialog: xtremRedux.actions.actionStub,
        onOpenLookupDialog: xtremRedux.actions.actionStub,
        screenDefinition,
    };
};
const mapDispatchToProps = (dispatch) => ({
    onOpenLookupDialog: (dialogDetails) => {
        dispatch(xtremRedux.actions.openLookupDialog(dialogDetails));
    },
    onCloseLookupDialog: () => {
        dispatch(xtremRedux.actions.closeLookupDialog());
    },
});
// Explicit cast so that LookupDialog can receive a generic from its consumer
export default connect(extendedMapStateToProps, mapDispatchToProps)(LookupDialog);
//# sourceMappingURL=lookup-dialog-component.js.map