import * as tokens from '@sage/design-tokens/js/base/common';
import Icon from 'carbon-react/esm/components/icon';
import IconButton from 'carbon-react/esm/components/icon-button';
import Typography from 'carbon-react/esm/components/typography';
import { camelCase } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import { selectSidebarNextRecord, selectSidebarPreviousRecord } from '../../redux/actions';
import { localize } from '../../service/i18n-service';
import { calculateActionMenu } from '../../utils/action-menu-utils';
import { triggerHandledEvent } from '../../utils/events';
import { resolveByValue } from '../../utils/resolve-value-utils';
import { getPageDefinitionFromState, getSidebarTableProperties } from '../../utils/state-utils';
import { cleanMetadataFromRecord, splitValueToMergedValue } from '../../utils/transformers';
import { withCollectionValueItemSubscription } from '../connected-collection';
import NavigationArrows from '../ui/navigation-arrows/navigation-arrows';
import XtremActionPopover from '../ui/xtrem-action-popover';
export function TableSidebarHeader({ accessBindings, elementId, hasNextRecord, hasPreviousRecord, headerDropdownActions, headerQuickActions, isLoading, onError, onNextRecord, onPreviousRecord: onPrevRecord, onTelemetryEvent, recordValue, screenDefinition, screenId, sidebarGetTitle, tableTitle, }) {
    if (!screenDefinition) {
        return null;
    }
    const rowValue = splitValueToMergedValue(recordValue);
    const title = isLoading
        ? localize('@sage/xtrem-ui/table-loadingOoo', 'Loading...')
        : resolveByValue({
            propertyValue: sidebarGetTitle,
            fieldValue: recordValue?._id,
            rowValue: recordValue,
            screenId,
            skipHexFormat: true,
        }) ||
            tableTitle ||
            elementId;
    const resolvedPopoverItems = calculateActionMenu({
        accessBindings,
        actions: headerDropdownActions,
        actionType: 'table-sidebar-header-dropdown-action',
        onTriggerMenuItem: (context, onClick, onActionError) => {
            if (onTelemetryEvent) {
                onTelemetryEvent(`tableSidebarHeaderDropdownActionTriggered-${context.uniqueId}`, {
                    screenId,
                    elementId,
                    id: context.id,
                    uniqueId: context.uniqueId,
                });
            }
            triggerHandledEvent(screenId, elementId, { onClick, onError: onActionError || onError }, recordValue?._id, cleanMetadataFromRecord(splitValueToMergedValue(recordValue)));
        },
        rowValue,
        screenId,
    });
    const resolvedHeaderQuickActions = calculateActionMenu({
        accessBindings,
        actions: headerQuickActions,
        actionType: 'table-sidebar-header-quick-action',
        onTriggerMenuItem: (context, onClick, onActionError) => {
            if (onTelemetryEvent) {
                onTelemetryEvent(`tableSidebarHeaderQuickActionTriggered-${context.uniqueId}`, {
                    screenId,
                    elementId,
                    id: context.id,
                    uniqueId: context.uniqueId,
                });
            }
            triggerHandledEvent(screenId, elementId, { onClick, onError: onActionError || onError }, recordValue?._id, cleanMetadataFromRecord(splitValueToMergedValue(recordValue)));
        },
        rowValue,
        screenId,
    });
    return (React.createElement(Typography, { variant: "h3", mb: "16px", className: "e-sidebar-header", "data-testid": "e-sidebar-header" },
        React.createElement(NavigationArrows, { hasNextRecord: !isLoading && hasNextRecord, hasPreviousRecord: !isLoading && hasPreviousRecord, onNextRecord: onNextRecord, onPreviousRecord: onPrevRecord }),
        React.createElement("div", { className: "e-sidebar-title" }, title),
        !isLoading && resolvedHeaderQuickActions?.length > 0 && (React.createElement("div", { className: "e-sidebar-quick-action-container", "data-testid": "e-sidebar-quick-action-container" }, resolvedHeaderQuickActions.map(a => {
            if (!a.icon) {
                throw new Error('Icon is missing from sidebar quick action definition.');
            }
            if (a.isHidden) {
                return null;
            }
            return (React.createElement(IconButton, { key: a.icon, onAction: () => a.onClick(), disabled: a.isDisabled, "aria-label": a.title, ml: "16px", "data-testid": `e-sidebar-quick-action-label-${camelCase(a.title)}` },
                React.createElement(Icon, { type: a.icon, tooltipMessage: a.title, color: a.isDestructive
                        ? tokens.colorsSemanticNegative500
                        : tokens.colorsUtilityMajor400 })));
        }))),
        !isLoading && resolvedPopoverItems?.length > 0 && (React.createElement("div", { className: "e-sidebar-dropdown-action-container", "data-testid": "e-sidebar-dropdown-action-container" },
            React.createElement(XtremActionPopover, { items: resolvedPopoverItems, noIconSupport: false, color: tokens.colorsUtilityMajor450, isOverSidebar: true })))));
}
export const mapDispatchToProps = (dispatch) => ({
    onNextRecord: () => dispatch(selectSidebarNextRecord()),
    onPreviousRecord: () => dispatch(selectSidebarPreviousRecord()),
});
export const mapStateToProps = (state) => {
    let screenDefinition;
    let accessBindings = {};
    let tableProperties;
    let value;
    let tableTitle;
    let sidebarGetTitle;
    if (state.sidebar?.screenId) {
        screenDefinition = getPageDefinitionFromState(state.sidebar?.screenId, state);
        tableProperties = getSidebarTableProperties({
            elementId: state.sidebar.elementId,
            pageDefinition: screenDefinition,
            level: state.sidebar.level,
        });
        value = screenDefinition.values[state.sidebar.elementId];
        sidebarGetTitle = tableProperties.sidebar?.title;
        accessBindings = screenDefinition.accessBindings;
        tableTitle = resolveByValue({
            propertyValue: tableProperties?.title,
            screenId: state.sidebar?.screenId,
            rowValue: null,
            fieldValue: null,
            skipHexFormat: true,
        });
    }
    return {
        accessBindings,
        elementId: state.sidebar?.elementId,
        hasNextRecord: state.sidebar?.hasNextRecord,
        hasPreviousRecord: state.sidebar?.hasPreviousRecord,
        headerDropdownActions: tableProperties?.sidebar?.headerDropdownActions,
        headerQuickActions: tableProperties?.sidebar?.headerQuickActions,
        isLoading: state.sidebar?.isLoading,
        isUncommitted: true,
        level: state.sidebar?.level,
        onError: tableProperties?.onError,
        onTelemetryEvent: state.applicationContext?.onTelemetryEvent,
        recordId: state.sidebar?.recordId || undefined,
        screenDefinition,
        screenId: state.sidebar?.screenId,
        sidebarGetTitle,
        tableTitle,
        value,
    };
};
export const ConnectedTableSidebarHeader = connect(mapStateToProps, mapDispatchToProps)(withCollectionValueItemSubscription(TableSidebarHeader));
//# sourceMappingURL=table-sidebar-header.js.map