import * as React from 'react';
import Dialog from 'carbon-react/esm/components/dialog';
import Button from 'carbon-react/esm/components/button';
import Form from 'carbon-react/esm/components/form';
import { WorkflowNodeSelectorComponent } from './workflow-node-selector-component';
import { StepSequence, StepSequenceItem } from 'carbon-react/esm/components/step-sequence';
import Typography from 'carbon-react/esm/components/typography';
import { useSelector } from 'react-redux';
import { ScreenComponent } from '../../ui/screen-component';
import { useWorkflowNodeVariables } from './workflow-component-utils';
import BusinessActions from '../../container/footer/business-actions';
import { ContextType } from '../../../types';
import { getArtifactDescription } from '../../../utils/transformers';
import { getPageTitlesFromPageDefinition } from '../../../utils/page-utils';
import { localize } from '../../../service/i18n-service';
export function AddWorkflowNodeDialog({ isOpen, onClose, onConfirm, previousNodeId, screenId, elementId, filterType, }) {
    const workflowNodes = useSelector(s => s.workflowNodes);
    const inputVariables = useWorkflowNodeVariables(previousNodeId, screenId, elementId, true);
    const [selectedNodeType, setSelectedNodeType] = React.useState(null);
    const [step, setStep] = React.useState(1);
    const selectedWorkflowType = workflowNodes?.find(n => n.key === selectedNodeType);
    const configPageArtifactDescription = selectedWorkflowType
        ? getArtifactDescription(selectedWorkflowType.configurationPage)
        : null;
    const configPageDefinition = useSelector(s => {
        if (!configPageArtifactDescription) {
            return null;
        }
        return s.screenDefinitions[configPageArtifactDescription.name] || null;
    });
    const onWorkflowConfigPageFinish = React.useCallback((values) => {
        if (selectedNodeType) {
            onConfirm({ selectedNodeType, values: { ...values, outputVariables: values.outputVariables || [] } });
        }
    }, [onConfirm, selectedNodeType]);
    const onPositiveButtonClick = React.useCallback(() => {
        if (step === 1) {
            setStep(2);
        }
    }, [step]);
    const onNegativeButtonClick = React.useCallback(() => {
        if (step === 1) {
            onClose();
        }
        if (step === 2) {
            setStep(1);
        }
    }, [onClose, step]);
    React.useEffect(() => {
        setStep(1);
        setSelectedNodeType(null);
    }, [isOpen]);
    const dialogTitle = React.useMemo(() => {
        if (!configPageDefinition || step === 1) {
            return localize('@sage/xtrem-ui/workflow-component-wizard-title', 'Add a new workflow node');
        }
        const { title } = getPageTitlesFromPageDefinition(configPageDefinition, 'en-US');
        return title;
    }, [configPageDefinition, step]);
    const hiddenCompleteLabel = localize('@sage/xtrem-ui/step-sequence-item-aria-complete', 'Complete');
    const hiddenCurrentLabel = localize('@sage/xtrem-ui/step-sequence-item-aria-current', 'Current');
    return (React.createElement(Dialog, { open: isOpen, onCancel: onClose, title: React.createElement("div", null,
            React.createElement(Typography, { variant: "h1" }, dialogTitle),
            React.createElement(StepSequence, { mb: "16px", mt: "16px" },
                React.createElement(StepSequenceItem, { "aria-label": localize('@sage/xtrem-ui/step-sequence-item-aria-count', 'Step {{0}} of {{1}}', [1, 2]), hiddenCompleteLabel: hiddenCompleteLabel, hiddenCurrentLabel: hiddenCurrentLabel, indicator: "1", status: step === 1 ? 'current' : 'complete' }, localize('@sage/xtrem-ui/workflow-component-wizard-step-selection', 'Step selection')),
                React.createElement(StepSequenceItem, { "aria-label": localize('@sage/xtrem-ui/step-sequence-item-aria-count', 'Step {{0}} of {{1}}', [2, 2]), hiddenCompleteLabel: hiddenCompleteLabel, hiddenCurrentLabel: hiddenCurrentLabel, indicator: "2", status: step === 1 ? 'incomplete' : 'current' }, localize('@sage/xtrem-ui/workflow-component-wizard-step-configuration', 'Configuration')))) },
        React.createElement(Form, { stickyFooter: true, height: "500px", leftSideButtons: React.createElement(Button, { onClick: onNegativeButtonClick }, step === 1
                ? localize('@sage/xtrem-ui/cancel', 'Cancel')
                : localize('@sage/xtrem-ui/wizard-previous', 'Previous')), saveButton: configPageDefinition && step === 2 ? (React.createElement(BusinessActions, { contextType: ContextType.dialog, defaultButtonType: "primary", screenId: configPageDefinition.metadata.screenId, businessActions: configPageDefinition.page.$.businessActions })) : (React.createElement(Button, { buttonType: "primary", type: "button", onClick: onPositiveButtonClick, disabled: !selectedNodeType || step === 2 }, "Next")) },
            step === 1 && (React.createElement("div", { className: "e-workflow-add-dialog" },
                React.createElement(WorkflowNodeSelectorComponent, { onChange: setSelectedNodeType, value: selectedNodeType, filterType: filterType }))),
            step === 2 && selectedWorkflowType && (React.createElement(ScreenComponent, { screenPath: selectedWorkflowType.configurationPage, queryParameters: { inputVariables }, onFinish: onWorkflowConfigPageFinish })))));
}
//# sourceMappingURL=add-workflow-node-dialog.js.map