import { debounce, set } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import { localize } from '../../../service/i18n-service';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { triggerFieldEvent, triggerNestedFieldEvent } from '../../../utils/events';
import { formatNumericValue, parseLocalizedNumberStringToNumber } from '../../../utils/formatters';
import { CarbonWrapper } from '../carbon-wrapper';
import { mapDispatchToProps, mapStateToProps } from '../field-base-component';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { isKeyPressedNumeric } from '../../../utils/ag-grid/ag-grid-cell-editor-utils';
import Textbox from 'carbon-react/esm/components/textbox';
import { splitValueToMergedValue } from '../../../utils/transformers';
import { getCommonCarbonComponentProperties } from '../carbon-helpers';
import { deepMerge } from '@sage/xtrem-shared';
import { convertDeepBindToPathNotNull } from '../../../utils/nested-field-utils';
export function NumericComponent(props) {
    const { screenId, fieldProperties, value, handlersArguments, locale, elementId, parentElementId, columnDefinition, isNested, setFieldValue, validate, } = props;
    const { scale } = fieldProperties;
    const sep = React.useMemo(() => localize('@sage/xtrem-ui/number-format-separator', '.', {}, locale), [locale]);
    const resolvedScale = React.useMemo(() => resolveByValue({
        screenId,
        propertyValue: scale,
        skipHexFormat: true,
        rowValue: handlersArguments?.rowValue ? splitValueToMergedValue(handlersArguments?.rowValue) : null,
        fieldValue: null,
    }) || 0, [handlersArguments?.rowValue, scale, screenId]);
    const [internalValue, setInternalValue] = React.useState(formatNumericValue({
        screenId,
        value,
        scale: resolvedScale,
        rowValue: handlersArguments?.rowValue,
    }));
    const changeEventHandler = React.useCallback((newValue) => {
        if (isNested && handlersArguments?.onChange && parentElementId) {
            const rowValue = newValue !== undefined
                ? deepMerge(handlersArguments.rowValue, set({}, convertDeepBindToPathNotNull(columnDefinition?.properties?.bind || fieldProperties?.bind || elementId), newValue))
                : handlersArguments?.rowValue;
            triggerNestedFieldEvent(screenId, parentElementId || elementId, fieldProperties, 'onChange', rowValue?._id, rowValue);
        }
        else {
            triggerFieldEvent(screenId, elementId, 'onChange');
        }
    }, [
        isNested,
        handlersArguments?.onChange,
        handlersArguments?.rowValue,
        parentElementId,
        columnDefinition?.properties?.bind,
        fieldProperties,
        elementId,
        screenId,
    ]);
    const clickEventHandler = React.useCallback(() => {
        if (isNested && handlersArguments?.onClick && parentElementId) {
            triggerNestedFieldEvent(screenId, parentElementId, fieldProperties, 'onClick', ...handlersArguments?.onClick);
        }
        else {
            triggerFieldEvent(screenId, elementId, 'onClick');
        }
    }, [screenId, elementId, parentElementId, isNested, handlersArguments, fieldProperties]);
    const isUnchangedValue = React.useCallback(() => {
        const numericValue = parseLocalizedNumberStringToNumber(internalValue, sep);
        // Do not trigger the change event if the value has not changed.
        if ((value === undefined && numericValue === undefined) ||
            (value === null && numericValue === null) ||
            value === numericValue) {
            return true;
        }
        // Do not trigger the change event if the value has not changed when taking the scale into account.
        if (value !== undefined &&
            value !== null &&
            numericValue !== undefined &&
            numericValue !== null &&
            resolvedScale !== undefined &&
            resolvedScale !== null &&
            value.toFixed(resolvedScale) === numericValue.toFixed(resolvedScale)) {
            return true;
        }
        return false;
    }, [internalValue, resolvedScale, sep, value]);
    const setFormattedValue = React.useCallback(() => {
        const formattedValue = formatNumericValue({
            screenId,
            value,
            scale: resolvedScale,
            rowValue: handlersArguments?.rowValue,
        });
        setInternalValue(formattedValue);
    }, [handlersArguments?.rowValue, resolvedScale, screenId, value]);
    const onInputValueChanged = React.useMemo(() => debounce(async (searchText) => {
        await triggerFieldEvent(screenId, elementId, 'onInputValueChange', searchText);
    }, 150), [screenId, elementId]);
    const onBlur = React.useCallback(() => {
        const numericValue = parseLocalizedNumberStringToNumber(internalValue, sep);
        if (isUnchangedValue())
            return;
        handleChange(elementId, numericValue, setFieldValue, validate, changeEventHandler);
    }, [changeEventHandler, elementId, internalValue, isUnchangedValue, sep, setFieldValue, validate]);
    const onChange = React.useCallback((ev) => {
        setInternalValue(ev.target.value);
        if (isUnchangedValue())
            return;
        onInputValueChanged(ev.target.value);
    }, [isUnchangedValue, onInputValueChanged]);
    const onKeyDown = React.useCallback((ev) => {
        const { key, altKey, metaKey, ctrlKey } = ev;
        if (key === 'Enter') {
            onBlur();
            return;
        }
        if (key === 'Escape') {
            setFormattedValue();
            return;
        }
        if (key === sep && resolvedScale === 0) {
            // If the user tries to use the separator character in an integer field.
            ev.preventDefault();
            ev.stopPropagation();
            return;
        }
        if (isKeyPressedNumeric(ev) ||
            [
                'ArrowLeft',
                'ArrowRight',
                'Tab',
                'Control',
                'Meta',
                'Shift',
                'Delete',
                'Backspace',
                'End',
                'Home',
                'PageUp',
                'PageDown',
            ].indexOf(key) !== -1 ||
            altKey ||
            metaKey ||
            ctrlKey) {
            return;
        }
        ev.preventDefault();
        ev.stopPropagation();
    }, [onBlur, sep, resolvedScale, setFormattedValue]);
    React.useEffect(() => {
        // Update the internal value when the value changes
        setFormattedValue();
    }, [setFormattedValue]);
    return (React.createElement(CarbonWrapper, { ...props, className: `e-numeric-field${resolvedScale ? ' e-decimal-field' : ''}`, componentName: "numeric", handlersArguments: handlersArguments, noReadOnlySupport: true, value: internalValue, readOnlyDisplayValue: internalValue },
        React.createElement(Textbox, { ...getCommonCarbonComponentProperties(props), onKeyDown: onKeyDown, onBlur: onBlur, onChange: onChange, onClick: clickEventHandler, value: internalValue, "data-testid": "e-ui-decimal-input", inputIcon: fieldProperties.icon })));
}
export const ConnectedNumericComponent = connect(mapStateToProps(), mapDispatchToProps())(NumericComponent);
export default ConnectedNumericComponent;
//# sourceMappingURL=numeric-component.js.map