import React from 'react';
import { connect, useSelector } from 'react-redux';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { mapDispatchToProps, mapStateToProps } from '../field-base-utils';
import { NodeBrowserTree } from '../../ui/node-browser-tree/node-browser-tree';
import { getComponentClass, getDataTestIdAttribute } from '../../../utils/dom';
import { FieldLabel, HelperText } from '../carbon-utility-components';
import { getLabelTitle, isFieldDisabled, isFieldReadOnly } from '../carbon-helpers';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { triggerFieldEvent } from '../../../utils/events';
import { useFocus } from '../../../utils/hooks/effects/use-focus';
import { getScreenElement } from '../../../service/screen-base-definition';
export const NodeBrowserTreeComponent = React.memo(props => {
    const screenDefinition = useSelector(state => state.screenDefinitions[props.screenId]);
    const { fieldProperties, setFieldValue, validate, value, elementId, isInFocus } = props;
    const { filter, node, fetchItems } = fieldProperties;
    const componentRef = React.useRef(null);
    useFocus(componentRef, isInFocus, 'input');
    const changeEventHandler = React.useCallback(() => triggerFieldEvent(props.screenId, props.elementId, 'onChange'), [props.screenId, props.elementId]);
    const changeClickHandler = React.useCallback(() => triggerFieldEvent(props.screenId, props.elementId, 'onClick'), [props.screenId, props.elementId]);
    const onCheckedItemsUpdated = React.useCallback((checkedItems) => {
        handleChange(elementId, checkedItems, setFieldValue, validate, changeEventHandler);
    }, [elementId, setFieldValue, validate, changeEventHandler]);
    const isReadOnly = isFieldReadOnly(props.screenId, props.fieldProperties, props.value, null); // Not available as a nested fields
    const isDisabled = isFieldDisabled(props.screenId, props.fieldProperties, props.value, null); // Not available as a nested fields
    const title = getLabelTitle(props.screenId, props.fieldProperties, null); // Not available as a nested fields
    const { isTitleHidden } = props.fieldProperties;
    const boundFetchItems = React.useCallback((property) => {
        if (fetchItems) {
            return fetchItems?.apply(getScreenElement(screenDefinition), [property]);
        }
        throw new Error('fetchItems is not defined');
    }, [fetchItems, screenDefinition]);
    const infoMessage = React.useMemo(() => resolveByValue({
        screenId: props.screenId,
        fieldValue: props.value,
        propertyValue: props.fieldProperties.infoMessage,
        rowValue: null,
        skipHexFormat: true,
    }), [props.screenId, props.value, props.fieldProperties.infoMessage]);
    const warningMessage = React.useMemo(() => resolveByValue({
        screenId: props.screenId,
        fieldValue: props.value,
        propertyValue: props.fieldProperties.warningMessage,
        rowValue: null,
        skipHexFormat: true,
    }), [props.screenId, props.value, props.fieldProperties.warningMessage]);
    return (React.createElement("div", { "data-testid": getDataTestIdAttribute('node-browser-tree', title, props.elementId), className: getComponentClass(props, 'e-node-browser-tree'), onClick: changeClickHandler, ref: componentRef },
        !isTitleHidden && (React.createElement(FieldLabel, { label: title, errorMessage: props.validationErrors?.[0].message, infoMessage: infoMessage, warningMessage: warningMessage })),
        React.createElement(NodeBrowserTree, { locale: props.locale, filter: filter, checkedItems: value || {}, node: node, fetchItems: fetchItems ? boundFetchItems : undefined, onCheckedItemsUpdated: onCheckedItemsUpdated, isDisabled: isDisabled, isReadOnly: isReadOnly }),
        React.createElement(HelperText, { helperText: props.fieldProperties.helperText })));
});
export const ConnectedNodeBrowserTree = connect(mapStateToProps(), mapDispatchToProps())(NodeBrowserTreeComponent);
export default ConnectedNodeBrowserTree;
NodeBrowserTreeComponent.displayName = 'NodeBrowserTreeComponent';
//# sourceMappingURL=node-browser-tree-component.js.map