import React, { useEffect, useState } from 'react';
import { MobileTable } from '../../ui/mobile-table/mobile-table-component';
import { cleanMetadataFromRecord } from '../../../utils/transformers';
import { getCardDefinitionFromColumns } from '../../../utils/table-component-utils';
import { convertDeepBindToPath } from '../../../utils/nested-field-utils';
import Button from 'carbon-react/esm/components/button';
import { triggerFieldEvent } from '../../../utils/events';
import { getFieldTitle, isFieldTitleHidden } from '../carbon-helpers';
import ButtonMinor from 'carbon-react/esm/components/button-minor';
import { localize } from '../../../service/i18n-service';
import { motion, AnimatePresence } from 'framer-motion';
export function MobileNestedGridComponent(props) {
    const [isFetchingNextPage, setIsFetchingNextPage] = useState(false);
    const [isLoading, setIsLoading] = useState(false);
    const [hasMorePages] = useState(false);
    const [valuesDisplayed, setValuesDisplayed] = useState([]);
    const [pageNumber, setPageNumber] = useState(0);
    const [currentLevel, setCurrentLevel] = useState(0);
    const [parentByLevel, setParentByLevel] = useState([]);
    const [direction, setDirection] = useState(1);
    const [isAnimationComplete, setIsAnimationComplete] = useState(true);
    const isTitleHidden = isFieldTitleHidden(props.screenId, props.fieldProperties, null);
    const currentLevelObject = props.fieldProperties.levels[currentLevel];
    useEffect(() => {
        setIsFetchingNextPage(true);
        setIsLoading(true);
        fetchRows({
            value: props.value,
            fieldProperties: props.fieldProperties,
            pageNumber,
            currentLevel,
            parentId: currentLevel > 0 ? parentByLevel[currentLevel - 1]?._id : undefined,
            pageSize: 10,
        }).then(newData => {
            setIsFetchingNextPage(false);
            setIsLoading(false);
            setValuesDisplayed(newData);
        });
    }, [props.value, props.fieldProperties, pageNumber, currentLevel, parentByLevel]);
    const onFetchNextPage = () => {
        setIsFetchingNextPage(true);
        fetchRows({
            value: props.value,
            fieldProperties: props.fieldProperties,
            pageNumber: pageNumber + 1,
            currentLevel,
            pageSize: 10,
            cursor: valuesDisplayed[valuesDisplayed.length - 1].cursor,
        }).then(newData => {
            setIsFetchingNextPage(false);
            setPageNumber(pageNumber + 1);
            setValuesDisplayed([...valuesDisplayed, ...newData]);
        });
    };
    const onRowSelected = (row) => {
        return triggerFieldEvent(props.screenId, props.elementId, 'onRowSelected', row._id, cleanMetadataFromRecord(row), row.__level ?? 0);
    };
    const onBackButtonClick = () => {
        setCurrentLevel(currentLevel - 1);
        setDirection(-1);
        if (currentLevel < 1) {
            setParentByLevel([]);
        }
        else {
            setParentByLevel(parentByLevel.slice(0, currentLevel - 1));
        }
    };
    const onRowClick = (item) => () => {
        setCurrentLevel(currentLevel + 1);
        setParentByLevel([...parentByLevel, item]);
        setDirection(1);
        if (item._isNewRecord) {
            if (typeof props.onRowClick === 'function') {
                props.onRowClick(item._id, currentLevel);
            }
        }
        else if (typeof props.onRowClick === 'function') {
            props.onRowClick(item._id, currentLevel);
        }
    };
    const renderGridHeader = () => {
        const nestedFields = getCardDefinitionFromColumns({
            columns: currentLevelObject.columns,
            mobileCard: props.fieldProperties.mobileCard,
            screenId: props.screenId,
            isGreaterThanSmall: false,
        });
        const titleBind = convertDeepBindToPath(nestedFields.title?.properties?.bind);
        return (React.createElement("div", { "data-testid": "e-mobile-nested-grid-header", className: "e-nested-grid-mobile-header" },
            React.createElement("div", { className: "e-nested-grid-mobile-header-left-container" },
                currentLevel === 0 && props.fieldProperties.title && !isTitleHidden && (React.createElement("span", { "data-testid": "e-mobile-nested-grid-header-title", className: "e-nested-grid-mobile-header-text" }, getFieldTitle(props.screenId, props.fieldProperties, null))),
                currentLevel > 0 && (React.createElement(ButtonMinor, { "data-testid": "e-mobile-nested-grid-header-back-button", iconType: "arrow_left", onClick: onBackButtonClick, buttonType: "tertiary", size: "large" })),
                currentLevel > 0 && titleBind && (React.createElement("span", { "data-testid": "e-mobile-nested-grid-header-child-title", className: "e-nested-grid-mobile-header-text" }, parentByLevel[currentLevel - 1][titleBind]))),
            React.createElement("div", { className: "e-mobile-nested-grid-header-right-container" },
                currentLevelObject.canAddNewLine && currentLevelObject.sidebar && (React.createElement(Button, { onClick: () => props.openSidebar({
                        screenId: props.screenId,
                        elementId: props.elementId,
                        level: currentLevel,
                        parentId: parentByLevel[currentLevel - 1]?._id,
                    }), buttonType: "primary" }, localize('@sage/xtrem-ui/add-item-in-line', 'Add line'))),
                React.createElement("span", { "data-testid": "e-mobile-grid-header-level", className: "e-nested-grid-mobile-header-text" },
                    currentLevel + 1,
                    "/",
                    props.fieldProperties.levels.length))));
    };
    const uiMobileProps = {
        elementId: props.elementId,
        fieldProperties: props.fieldProperties,
        columns: currentLevelObject.columns,
        groupByField: null,
        hasMorePages,
        isDisabled: props.isDisabled,
        isGreaterThanSmall: false,
        isFetchingNextPage,
        isNavigationPanel: false,
        onFetchNextPage,
        onRowSelected,
        dropdownActions: currentLevelObject.dropdownActions,
        screenId: props.screenId,
        valuesDisplayed,
        onRowClick: props.onRowClick || currentLevel < props.fieldProperties.levels.length - 1 ? onRowClick : undefined,
        isLoading,
    };
    const animationVariants = {
        enter: (incDirection) => {
            return {
                x: incDirection > 0 ? 1000 : -1000,
                opacity: 0,
            };
        },
        center: {
            zIndex: 1,
            x: 0,
            opacity: 1,
        },
        exit: (incDirection) => {
            return {
                zIndex: 0,
                x: incDirection < 0 ? 1000 : -1000,
                opacity: 0,
            };
        },
    };
    return (React.createElement("div", { "data-testid": "e-mobile-nested-grid-component" },
        renderGridHeader(),
        React.createElement(AnimatePresence, { initial: false, custom: direction },
            React.createElement(motion.div, { key: currentLevel, custom: direction, variants: animationVariants, initial: "enter", animate: "center", exit: "exit", transition: {
                    x: { type: 'spring', stiffness: 300, damping: 30 },
                    opacity: { duration: 0.2 },
                }, onAnimationStart: () => setIsAnimationComplete(false), onAnimationComplete: () => setIsAnimationComplete(true) }, isAnimationComplete && React.createElement(MobileTable, { ...uiMobileProps })))));
}
async function fetchRows({ value, fieldProperties, pageNumber, currentLevel, pageSize, cursor, parentId, }) {
    return value.getPageWithCurrentQueryArguments({
        tableFieldProperties: {
            ...fieldProperties,
            columns: fieldProperties.levels[currentLevel].columns,
        },
        pageSize,
        pageNumber,
        level: currentLevel,
        parentId,
        cleanMetadata: false,
        cursor,
    });
}
//# sourceMappingURL=mobile-nested-grid-component.js.map