import { debounce, noop } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import uid from 'uid';
import * as xtremRedux from '../../../redux';
import { fetchReferenceFieldSuggestions } from '../../../service/graphql-service';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { triggerFieldEvent } from '../../../utils/events';
import { splitValueToMergedValue } from '../../../utils/transformers';
import { text } from '../../nested-fields';
import LookupDialog from '../../ui/lookup-dialog/lookup-dialog-component';
import { Select } from '../../ui/select/select-component';
import { getCommonCarbonComponentProperties, getLabelTitle } from '../carbon-helpers';
import { CarbonWrapper } from '../carbon-wrapper';
import { EditableFieldBaseComponent, mapDispatchToProps, mapStateToProps } from '../field-base-component';
import { getReferenceSelectedRecord, hasLookupIcon, nodesToSelectItems } from '../reference/reference-utils';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { useCombobox } from 'downshift';
export class MultiReferenceComponent extends EditableFieldBaseComponent {
    constructor(props) {
        super(props);
        this.lookupButtonRef = React.createRef();
        this.onInputValueChanged = debounce(async (searchText, type) => {
            if (type !== useCombobox.stateChangeTypes.FunctionSelectItem) {
                await triggerFieldEvent(this.props.screenId, this.props.elementId, 'onInputValueChange', searchText);
            }
        }, 150);
        this.onMobileInputChange = async (searchText, type) => {
            this.mobileAutoSelect(type);
            await this.onInputValueChanged(searchText, type);
        };
        this.onSelectedItemsChange = (selectedRecords) => {
            handleChange(this.props.elementId, selectedRecords, this.props.setFieldValue, this.props.validate, this.triggerChangeListener);
        };
        this.openLookupDialog = () => {
            if (this.props.setFieldProperties) {
                this.props.setFieldProperties(this.props.elementId, {
                    ...this.props.fieldProperties,
                    isMultiReferenceDialogOpen: true,
                });
                this.props.removeNonNestedErrors(this.props.elementId);
            }
            const triggerDialogOpen = () => triggerFieldEvent(this.props.screenId, this.props.elementId, 'onOpenMultiReferenceDialog');
            if (this.props.parentElementId) {
                this.setState({ isNestedLookupDialogOpen: true }, () => triggerDialogOpen);
            }
            else {
                triggerDialogOpen();
            }
        };
        this.closeLookupDialog = (cb = noop) => {
            triggerFieldEvent(this.props.screenId, this.props.elementId, 'onCloseMultiReferenceDialog');
            if (this.props.setFieldProperties) {
                this.props.setFieldProperties(this.props.elementId, {
                    ...this.props.fieldProperties,
                    isMultiReferenceDialogOpen: false,
                });
            }
            const stateUpdate = {};
            if (this.props.parentElementId && this.state.isNestedLookupDialogOpen) {
                stateUpdate.isNestedLookupDialogOpen = false;
            }
            this.setState(stateUpdate, () => {
                cb();
                setTimeout(() => {
                    this.lookupButtonRef.current?.focus();
                }, 500);
            });
        };
        this.onLookupDialogSelectionFinished = (selectedRecords) => {
            this.closeLookupDialog(() => {
                this.onSelectedItemsChange(selectedRecords);
            });
        };
        this.hasLookupIcon = () => hasLookupIcon({
            screenId: this.props.screenId,
            fieldProperties: this.props.fieldProperties,
            value: this.props.value,
            rowValue: this.props.handlersArguments?.rowValue
                ? splitValueToMergedValue(this.props.handlersArguments?.rowValue)
                : this.props.handlersArguments?.rowValue,
        });
        this.getBorderColor = () => {
            // INFO: In case the border of nested multi-reference components should not be rendered,
            //       this function can return 'transparent' as follows:
            //       return this.props.contextType === ContextType.pod ? 'transparent' : undefined;
            return undefined;
        };
        this.isLookupDialogOpen = () => (!!this.props.fieldProperties.isMultiReferenceDialogOpen || !!this.state.isNestedLookupDialogOpen) &&
            !!this.props.fieldProperties.columns;
        this.isLookupDialogOpenMobile = () => !!this.props.fieldProperties.isMultiReferenceDialogOpen && !this.isReadOnly() && !this.isDisabled();
        this.isReadOnlyAndHasPicture = () => !!this.props.fieldProperties.imageField && (this.isReadOnly() || this.isDisabled());
        this.mobileAutoSelect = debounce(async (type) => {
            const isOrganicChange = this.selectInputRef.current?.value !== '' && type !== useCombobox.stateChangeTypes.FunctionSelectItem;
            if (isOrganicChange) {
                this.openLookupDialog();
            }
        }, 200);
        this.renderMobile = (className) => {
            const carbonProps = getCommonCarbonComponentProperties(this.props);
            const { value, fieldProperties } = this.props;
            const helperText = this.props.fieldProperties.helperText;
            const wrapperStyle = helperText && !this.props.fieldProperties.isHelperTextHidden
                ? { alignItems: 'center' }
                : { alignItems: 'flex-end' };
            const selectedRecords = (value ?? [])
                .map(v => getReferenceSelectedRecord({
                value: v,
                fieldProperties,
            }))
                .filter(Boolean);
            // This will be used by the CarbonWrapper in case the field is nested
            const initialInputValue = this.props.isNested ? selectedRecords.map(s => s.value).join(', ') : '';
            return (React.createElement(CarbonWrapper, { ...this.props, className: className, componentName: "multi-reference", componentRef: this.componentRef, handlersArguments: this.props.handlersArguments, helperText: helperText, noReadOnlySupport: true, value: initialInputValue },
                React.createElement("div", { className: "e-multi-reference-field", style: wrapperStyle },
                    React.createElement(Select, { ...carbonProps, borderColor: this.getBorderColor(), disabled: this.isDisabled(), fullWidth: this.props.fieldProperties.isFullWidth, getItems: this.getItems, hasLookupIcon: true, helperText: helperText, icon: this.props.fieldProperties.icon, isSortedAlphabetically: true, 
                        // this maps to the select's input text, which will always initially be empty
                        // the internal pills will come from 'selectedRecords'
                        initialInputValue: "", inputId: carbonProps.id, onInputChange: this.onMobileInputChange, isDropdownDisabled: true, isMultiSelect: true, label: getLabelTitle(this.props.screenId, this.props.fieldProperties, this.props.handlersArguments?.rowValue), lookupButtonRef: this.lookupButtonRef, lookupIconId: `e-multi-reference-field-lookup-icon-${this.state.id}`, minLookupCharacters: this.getMinLookupCharacters(), onInputFocus: this.props.onFocus, onLookupIconClick: this.openLookupDialog, onSelectedItemsChange: this.onSelectedItemsChange, placeholder: this.props.fieldProperties.placeholder, preventSelectionOnBlur: Boolean(this.props.fieldProperties.isMultiReferenceDialogOpen) ||
                            Boolean(this.state.isNestedLookupDialogOpen), readOnly: this.isReadOnly(), ref: this.selectInputRef, selectedItems: selectedRecords, testId: `e-multi-reference-field-lookup-search-text-${this.state.id}`, screenId: this.props.screenId, elementId: this.props.elementId, isSoundDisabled: resolveByValue({
                            screenId: this.props.screenId,
                            propertyValue: this.props.fieldProperties.isSoundDisabled,
                            skipHexFormat: true,
                            fieldValue: null,
                            rowValue: null,
                        }) })),
                React.createElement(LookupDialog, { isOpen: this.isLookupDialogOpenMobile(), fieldProperties: this.props.fieldProperties, screenId: this.props.screenId, onSelectionFinished: this.onLookupDialogSelectionFinished, fieldId: this.props.elementId, searchText: this.selectInputRef.current?.value ?? '', closeLookupDialog: this.closeLookupDialog, isMultiSelect: true, selectedRecordId: this.props.value?.map(r => r._id) })));
        };
        this.getItems = async (filterValue) => {
            const nodes = await fetchReferenceFieldSuggestions({
                fieldProperties: this.props.fieldProperties,
                screenId: this.props.screenId,
                fieldId: this.props.elementId,
                filterValue,
                parentElementId: this.props.parentElementId,
                recordContext: this.props.handlersArguments?.rowValue,
                contextNode: this.props.contextNode,
                level: this.props.level,
            });
            return nodesToSelectItems({
                nodes,
                fieldProperties: this.props.fieldProperties,
            });
        };
        this.getMinLookupCharacters = () => this.props.fieldProperties.minLookupCharacters ?? 3;
        this.renderDesktop = (className) => {
            const selectedRecords = (this.props.value ?? [])
                .map(v => getReferenceSelectedRecord({
                value: v,
                fieldProperties: this.props.fieldProperties,
            }))
                .filter(Boolean);
            const helperText = this.props.fieldProperties.helperText;
            const carbonProps = getCommonCarbonComponentProperties(this.props);
            // This will be used by the CarbonWrapper in case the field is nested
            const initialInputValue = this.props.isNested ? selectedRecords.map(s => s.value).join(', ') : '';
            return (React.createElement(React.Fragment, null,
                React.createElement(CarbonWrapper, { ...this.props, className: className, componentName: "multi-reference", componentRef: this.componentRef, helperText: this.props.fieldProperties.helperText, noReadOnlySupport: true, value: initialInputValue },
                    React.createElement("div", { onClick: this.getClickHandler(), className: "e-multi-reference-field-body" },
                        React.createElement(Select, { disabled: this.isDisabled(), error: this.props.validationErrors?.[0]?.message, info: carbonProps.info, warning: carbonProps.warning, fullWidth: this.props.fieldProperties.isFullWidth, getItems: this.getItems, hasLookupIcon: this.hasLookupIcon(), onInputChange: this.onInputValueChanged, helperText: helperText, icon: this.props.fieldProperties.icon, 
                            // this maps to the select's input text, which will always initially be empty
                            // the internal pills will come from 'selectedRecords'
                            initialInputValue: "", inputId: carbonProps.id, isSortedAlphabetically: true, isDropdownDisabled: this.props.fieldProperties.isDropdownDisabled, isMultiSelect: true, label: !this.props.fieldProperties.isTitleHidden
                                ? getLabelTitle(this.props.screenId, this.props.fieldProperties, this.props.handlersArguments?.rowValue)
                                : undefined, lookupIconId: `e-multi-reference-field-lookup-icon-${this.state.id}`, lookupButtonRef: this.lookupButtonRef, minLookupCharacters: this.getMinLookupCharacters(), onSelectedItemsChange: this.onSelectedItemsChange, onInputFocus: this.props.onFocus, onLookupIconClick: this.openLookupDialog, placeholder: this.props.fieldProperties.placeholder, readOnly: this.isReadOnly(), selectedItems: selectedRecords, size: this.props.fieldProperties.size, testId: `e-multi-reference-field-lookup-input-${this.state.id}`, screenId: this.props.screenId, elementId: this.props.elementId, isSoundDisabled: resolveByValue({
                                screenId: this.props.screenId,
                                propertyValue: this.props.fieldProperties.isSoundDisabled,
                                skipHexFormat: true,
                                fieldValue: null,
                                rowValue: null,
                            }) }))),
                React.createElement(LookupDialog, { isOpen: this.isLookupDialogOpen(), fieldProperties: this.props.fieldProperties, screenId: this.props.screenId, onSelectionFinished: this.onLookupDialogSelectionFinished, fieldId: this.props.elementId, closeLookupDialog: this.closeLookupDialog, parentElementId: this.props.parentElementId, recordContext: this.props.recordContext, contextNode: this.props.contextNode, isMultiSelect: true, selectedRecordId: this.props.value?.map(r => r._id) })));
        };
        this.isRelatedTargetEqualTo = (e, input) => e.relatedTarget?.id === `${input}-${this.state.id}`;
        this.state = {
            id: uid(16),
        };
        this.selectInputRef = React.createRef();
    }
    componentDidMount() {
        // On mobile if there are no columns by default we add "valueField" & "helperTextField"
        if (!this.props.fieldProperties.columns &&
            !this.isReadOnly() &&
            !(this.props.browser?.greaterThan.s ?? true) &&
            this.props.setFieldProperties) {
            this.props.setFieldProperties(this.props.elementId, {
                ...this.props.fieldProperties,
                columns: [this.props.fieldProperties.valueField, this.props.fieldProperties.helperTextField]
                    .filter(c => !!c)
                    .map(c => {
                    // Hacky as query generation works with object as well as strings
                    const bind = typeof c === 'object' ? c : String(c);
                    // If we remove the "as" we get a type error from TS
                    // NOSONAR
                    return text({ bind: bind });
                }),
            });
        }
    }
    componentDidUpdate(prevProps) {
        if (!!prevProps.fieldProperties.isMultiReferenceDialogOpen &&
            !this.props.fieldProperties.isMultiReferenceDialogOpen &&
            this.componentRef.current) {
            const input = this.getFocusableElement(this.componentRef.current);
            if (input && document.activeElement !== input) {
                input.focus();
            }
        }
    }
    render() {
        let className = 'e-multi-reference-field';
        if (this.hasLookupIcon()) {
            className += ' e-multi-reference-field-lookup';
        }
        if (this.isReadOnlyAndHasPicture()) {
            className += ' e-multi-reference-inline-picture';
        }
        if (!this.isReadOnly() && !(this.props.browser?.greaterThan.s ?? true)) {
            return this.renderMobile(className);
        }
        return this.renderDesktop(className);
    }
}
const extendedMapDispatchToProps = (dispatch, props) => {
    const defaultMapDispatchToProps = mapDispatchToProps()(dispatch, props);
    return {
        ...defaultMapDispatchToProps,
        setFieldProperties: (elementId, value) => {
            dispatch(xtremRedux.actions.setFieldProperties(props.screenId, elementId, value));
        },
    };
};
export const ConnectedMultiReferenceComponent = connect(mapStateToProps, extendedMapDispatchToProps)(MultiReferenceComponent);
export default ConnectedMultiReferenceComponent;
//# sourceMappingURL=multi-reference-component.js.map