import { omitBy } from 'lodash';
import React from 'react';
import uid from 'uid';
import * as xtremRedux from '../../../redux';
import { fetchReferenceFieldSuggestions } from '../../../service/graphql-service';
import { localize } from '../../../service/i18n-service';
import { getInitialCellEditorState, setDefaultAgGridInputStyles, } from '../../../utils/ag-grid/ag-grid-cell-editor-utils';
import { triggerFieldEvent } from '../../../utils/events';
import { shouldRenderDropdownAbove } from '../../../utils/table-component-utils';
import { cleanMetadataAndNonPersistentIdFromRecord, splitValueToMergedValue } from '../../../utils/transformers';
import LookupDialog from '../../ui/lookup-dialog/lookup-dialog-component';
import { Select } from '../../ui/select/select-component';
import { nodeToSelectItem, nodesToSelectItems } from '../reference/reference-utils';
export default class FilterSelectCellEditor extends React.Component {
    constructor(props) {
        super(props);
        this.getItems = async (filterValue) => {
            const level = this.props.node?.data?.__level;
            const nodes = await fetchReferenceFieldSuggestions({
                fieldProperties: this.props.fieldProperties,
                screenId: this.props.screenId,
                fieldId: this.props.elementId,
                filterValue,
                parentElementId: this.props.tableElementId,
                recordContext: splitValueToMergedValue(cleanMetadataAndNonPersistentIdFromRecord(this.props.data)),
                level,
                contextNode: String(this.props.contextNode),
            });
            return nodesToSelectItems({
                nodes,
                fieldProperties: this.props.fieldProperties,
            }).map(item => omitBy(item, (_v, k) => k === '__collectionItem'));
        };
        this.onSelected = (selectedRecord, selectionType) => {
            this.props.eGridCell.id = selectedRecord?.id ?? '';
            this.props.onValueChange(selectedRecord?.value ?? null);
            if (selectionType !== '__input_blur__') {
                this.props.stopEditing(true);
            }
        };
        this.isLookupDialogOpen = () => this.state.isLookupPanelOpen && !!this.props.fieldProperties.columns;
        this.onLookupDialogSelectionFinished = ([collectionItem] = []) => {
            if (collectionItem) {
                this.onSelected(nodeToSelectItem({ collectionItem, fieldProperties: this.props.fieldProperties }));
            }
            else {
                this.onSelected();
            }
            this.closeLookupDialog();
        };
        this.closeLookupDialog = () => {
            triggerFieldEvent(this.props.screenId, this.props.elementId, 'onCloseFilterSelectDialog');
            xtremRedux.getStore().dispatch(xtremRedux.actions.setFieldProperties(this.props.screenId, this.props.elementId, {
                ...this.props.fieldProperties,
                isFilterSelectDialogOpen: false,
            }));
            this.setState({ isLookupPanelOpen: false });
            setTimeout(() => {
                this.props.api.setFocusedCell(this.props.node.rowIndex ?? 0, this.props.column, this.props.node.isRowPinned() ? 'top' : null);
            }, 50);
            this.props.stopEditing(true);
        };
        this.openLookupDialog = (event) => {
            event.stopPropagation();
            event.preventDefault();
            triggerFieldEvent(this.props.screenId, this.props.elementId, 'onOpenFilterSelectDialog');
            const filterSelectValue = this.props.data[this.props.colDef.field];
            const dispatch = xtremRedux.getStore().dispatch;
            dispatch(xtremRedux.actions.setFieldProperties(this.props.screenId, this.props.elementId, filterSelectValue));
            dispatch(xtremRedux.actions.setFieldProperties(this.props.screenId, this.props.elementId, {
                ...this.props.fieldProperties,
                isFilterSelectDialogOpen: true,
            }));
            dispatch(xtremRedux.actions.removeNonNestedErrors(this.props.screenId, this.props.elementId));
            this.setState({ isLookupPanelOpen: true });
        };
        this.getLocalizedSuffix = () => `(${localize('@sage/xtrem-ui/new', 'New')})`;
        this.getNewItem = (value) => {
            const displayedAs = `${value} ${this.getLocalizedSuffix()}`;
            return {
                id: '-1',
                value,
                displayedAs,
            };
        };
        this.selectRef = React.createRef();
        this.state = this.createInitialState(props);
    }
    componentDidMount() {
        const input = this.selectRef.current;
        if (!input) {
            return;
        }
        // try to match the styles of ag-grid
        setDefaultAgGridInputStyles(input);
        const inputDiv = input.parentNode;
        inputDiv.style.width = '100%';
        inputDiv.style.height = '100%';
        inputDiv.style.border = 'none';
        inputDiv.style.display = 'flex';
        inputDiv.style.alignItems = 'center';
        input.focus();
        if (this.state.highlightOnFocus) {
            input.select();
            this.setState({
                highlightOnFocus: false,
            });
        }
        else {
            // when we started editing, we want the caret at the end, not the start.
            // this comes into play in two scenarios: a) when user hits F2 and b)
            // when user hits a printable character, then on IE (and only IE) the caret
            // was placed after the first character, thus 'apply' would end up as 'pplea'
            const length = input.value ? input.value.length : 0;
            if (length > 0) {
                input.setSelectionRange(length, length);
            }
        }
    }
    createInitialState(props) {
        const { highlightOnFocus, value: startValue } = getInitialCellEditorState({
            eventKey: props.eventKey,
            initialValue: props.initialValue,
        });
        const selectedRecord = props.value
            ? {
                id: this.props.eGridCell.id || '',
                value: props.value,
                displayedAs: props.value,
            }
            : undefined;
        return {
            highlightOnFocus,
            id: uid(16),
            isLookupPanelOpen: false,
            selectedRecord,
            startValue,
        };
    }
    render() {
        const dataTestId = `${this.props.tableElementId}-${this.props.node.rowIndex}-${(this.props.api.getColumns() ?? []).indexOf(this.props.column) + 1}`;
        const width = this.props.eGridCell?.style?.width || '200px';
        return (React.createElement("div", { "data-testid": dataTestId, style: { width }, className: "e-filter-select-cell-editor" },
            React.createElement(Select, { allowClearOnTab: true, allowSelectOnTab: true, autoSelect: true, defaultIsOpen: true, elementId: this.props.elementId, escapeBehavior: "revert", getItems: this.getItems, getNewItem: this.getNewItem, hasLookupIcon: false, initialInputValue: this.state.startValue, isSortedAlphabetically: true, isDropdownDisabled: false, isInTable: true, lookupIconId: `e-filter-select-cell-editor-lookup-icon-${this.state.id}`, minLookupCharacters: this.props.fieldProperties.minLookupCharacters, onLookupIconClick: this.openLookupDialog, onSelected: this.onSelected, placeholder: this.props.fieldProperties.placeholder, preventSelectionOnBlur: this.isLookupDialogOpen(), ref: this.selectRef, screenId: this.props.screenId, selectedItem: this.state.selectedRecord, shouldRenderOptionsAbove: shouldRenderDropdownAbove({
                    isPhantomRow: !!this.props.data?.__phantom,
                    pageSize: this.props.api.paginationGetPageSize(),
                    rowIndex: this.props.node.rowIndex ?? 0,
                }), testId: `${dataTestId}-input`, variant: "plain" }),
            React.createElement(LookupDialog, { isOpen: this.isLookupDialogOpen(), fieldProperties: this.props.fieldProperties, screenId: this.props.screenId, onSelectionFinished: this.onLookupDialogSelectionFinished, fieldId: this.props.elementId, closeLookupDialog: this.closeLookupDialog, parentElementId: this.props.tableElementId, contextNode: String(this.props.contextNode), recordContext: splitValueToMergedValue(cleanMetadataAndNonPersistentIdFromRecord(this.props.data)) })));
    }
}
//# sourceMappingURL=filter-select-cell-editor.js.map