import Button from 'carbon-react/esm/components/button';
import * as React from 'react';
import { connect } from 'react-redux';
import * as tokens from '@sage/design-tokens/js/base/common';
import { localize } from '../../../service/i18n-service';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { getComponentClass, getDataTestIdAttribute, getNewFileReader } from '../../../utils/dom';
import { triggerFieldEvent } from '../../../utils/events';
import { useFocus } from '../../../utils/hooks/effects/use-focus';
import { Icon } from '../../ui/icon/icon-component';
import { getLabelTitle, isFieldDisabled, isFieldReadOnly } from '../carbon-helpers';
import { mapDispatchToProps, mapStateToProps } from '../field-base-utils';
import { FieldLabel, HelperText } from '../carbon-utility-components';
import { resolveByValue } from '../../../utils/resolve-value-utils';
export function FileComponent(props) {
    const hasValue = props.value && props.value.value;
    const [mimeType, setMimeType] = React.useState('data:application/octet-stream;base64');
    const inputFileRef = React.useRef(null);
    const fileComponentRef = React.useRef(null);
    useFocus(fileComponentRef, props.isInFocus, 'button');
    const isReadOnly = isFieldReadOnly(props.screenId, props.fieldProperties, props.value, null); // Not available as a nested fields
    const isDisabled = isFieldDisabled(props.screenId, props.fieldProperties, props.value, null); // Not available as a nested fields
    const changeEventHandler = React.useCallback(() => triggerFieldEvent(props.screenId, props.elementId, 'onChange'), [props.screenId, props.elementId]);
    const clickEventHandler = React.useCallback(() => triggerFieldEvent(props.screenId, props.elementId, 'onClick'), [props.screenId, props.elementId]);
    const onInputTypeFileChanged = async (event) => {
        if (event.target.files && event.target.files.length > 0) {
            const result = await readFile(event.target.files[0]);
            handleChange(props.elementId, result ? { value: result[1] } : null, props.setFieldValue, props.validate, changeEventHandler);
            if (result) {
                setMimeType(result[0]);
            }
        }
    };
    const readFile = (file) => new Promise(resolve => {
        const reader = getNewFileReader();
        reader.onload = () => {
            let components = null;
            if (reader.result) {
                const result = reader.result.toString().split(',');
                if (result.length) {
                    components = result;
                }
            }
            triggerFieldEvent.apply(null, [
                props.screenId,
                props.elementId,
                'onFileInfo',
                file.type,
                file.size,
                file.name,
            ]);
            resolve(components);
        };
        reader.readAsDataURL(file);
    });
    const onFileUploadAreaClick = () => {
        if (!isReadOnly && !isDisabled) {
            inputFileRef.current?.click();
        }
    };
    const onDeleteFile = () => handleChange(props.elementId, null, props.setFieldValue, props.validate, changeEventHandler);
    const renderFileUploadedView = () => {
        const buildHref = isDisabled || !hasValue ? undefined : `${mimeType},${props.value?.value}`;
        const deleteLabel = localize('@sage/xtrem-ui/crud-delete', 'Delete');
        return (React.createElement("div", { className: `e-file-field-file-uploaded${isDisabled ? ' e-file-field-file-uploaded-disabled' : ''}`, "data-testid": "e-file-field-file-uploaded" },
            React.createElement("div", { className: "e-file-field-details", "data-testid": "e-file-field-details" },
                React.createElement(Icon, { type: "pdf", my: 1, color: tokens.colorsUtilityMajor300 }),
                React.createElement("a", { className: "e-file-field-name", "data-testid": "e-file-field-name", href: buildHref, target: "_blank", download: props.fieldProperties.text, rel: "noreferrer" },
                    React.createElement("span", null, props.fieldProperties.text)),
                !isReadOnly && !isDisabled && (React.createElement(Button, { iconType: "delete", "aria-label": deleteLabel, iconTooltipMessage: deleteLabel, disabled: isDisabled, onClick: isDisabled || isReadOnly ? undefined : onDeleteFile })))));
    };
    const renderUploadArea = () => (React.createElement("div", { "data-testid": "e-file-field-upload-area", onClick: onFileUploadAreaClick },
        React.createElement("input", { accept: props.fieldProperties.fileTypes, className: "e-hidden", "data-testid": "e-file-field-file-input", onChange: onInputTypeFileChanged, ref: inputFileRef, type: "file" }),
        React.createElement(Button, { buttonType: "secondary", iconType: "upload", disabled: isDisabled || isReadOnly, onClick: clickEventHandler }, localize('@sage/xtrem-ui/file-component-browse-files', 'Browse Files'))));
    const title = getLabelTitle(props.screenId, props.fieldProperties, null); // Not available as a nested fields
    const { isTitleHidden } = props.fieldProperties;
    const infoMessage = resolveByValue({
        screenId: props.screenId,
        fieldValue: props.value,
        propertyValue: props.fieldProperties.infoMessage,
        rowValue: null,
        skipHexFormat: true,
    });
    const warningMessage = resolveByValue({
        screenId: props.screenId,
        fieldValue: props.value,
        propertyValue: props.fieldProperties.warningMessage,
        rowValue: null,
        skipHexFormat: true,
    });
    return (React.createElement("div", { "data-testid": getDataTestIdAttribute('file', title, props.elementId), className: getComponentClass(props, 'e-file-field'), ref: fileComponentRef },
        !isTitleHidden && (React.createElement(FieldLabel, { label: title, errorMessage: props.validationErrors?.[0].message, infoMessage: infoMessage, warningMessage: warningMessage })),
        React.createElement("div", { className: "e-file-field-content-wrapper", "data-testid": "e-file-field-content-wrapper" }, hasValue ? renderFileUploadedView() : renderUploadArea()),
        React.createElement(HelperText, { helperText: props.fieldProperties.helperText })));
}
export const ConnectedFileComponent = connect(mapStateToProps(), mapDispatchToProps())(FileComponent);
export default ConnectedFileComponent;
//# sourceMappingURL=file-component.js.map