import { Time } from '@sage/xtrem-date-time';
import { usePrevious } from '@sage/xtrem-ui-components';
import { useCallback, useEffect, useMemo, useReducer } from 'react';
import { datetimeToCalendarDate, datetimeToTime, isValidTimeZone, makeDatetime, toDatetime, } from './datetime-range-utils';
function datetimeRangeReducer(state, action) {
    switch (action.type) {
        case 'SET_START_DATE':
            return handleSetStartDate(state, action);
        case 'SET_END_DATE':
            return handleSetEndDate(state, action);
        case 'SET_START_TIME':
            return handleSetStartTime(state, action);
        case 'SET_END_TIME':
            return handleSetEndTime(state, action);
        case 'RESET':
            return handleReset(state, action);
        case 'SET_OPEN_POPOVER_ON_INPUT':
            return handleSetOpenPopover(state, action);
        default:
            return state;
    }
}
function handleSetStartDate(state, action) {
    let newStartDatetime = null;
    let newEndDatetime = state.value.end;
    if (action.date && state.startTime) {
        newStartDatetime = makeDatetime(action.date, Time.parse(state.startTime), state.timeZone);
    }
    if (action.date && state.endDate && state.endDate.compare(action.date) < 0) {
        newEndDatetime = null;
    }
    const value = {
        ...state.value,
        start: newStartDatetime,
        end: newEndDatetime,
    };
    return {
        ...state,
        startDate: action.date,
        endDate: newEndDatetime ? datetimeToCalendarDate(newEndDatetime, state.timeZone) : null,
        value,
        isDirty: true,
    };
}
function handleSetEndDate(state, action) {
    let newEndDatetime = null;
    let validationError = false;
    if (action.date && state.endTime) {
        newEndDatetime = makeDatetime(action.date, Time.parse(state.endTime), state.timeZone);
    }
    if (newEndDatetime && state.value.start && newEndDatetime.compare(state.value.start) < 0) {
        validationError = true;
    }
    const value = {
        ...state.value,
        end: newEndDatetime,
    };
    return {
        ...state,
        endDate: action.date,
        value,
        validationError,
        isDirty: true,
    };
}
function handleSetStartTime(state, action) {
    let endDate = state.endDate;
    let endTime = state.endTime;
    let newStartDatetime = null;
    let newEndDatetime = state.value.end;
    if (action.time && state.startDate) {
        newStartDatetime = makeDatetime(state.startDate, Time.parse(action.time), state.timeZone);
    }
    const validationError = (newStartDatetime && state.value.end && state.value.end.compare(newStartDatetime) < 0) || false;
    if (validationError) {
        endDate = null;
        endTime = null;
        newEndDatetime = null;
    }
    const value = {
        ...state.value,
        start: newStartDatetime,
        end: newEndDatetime,
    };
    return {
        ...state,
        startTime: action.time,
        endDate,
        endTime,
        value,
        validationError,
        isDirty: true,
    };
}
function handleSetEndTime(state, action) {
    let newEndDatetime = null;
    if (action.time && state.endDate) {
        newEndDatetime = makeDatetime(state.endDate, Time.parse(action.time), state.timeZone);
    }
    const validationError = (newEndDatetime && state.value.start && newEndDatetime.compare(state.value.start) < 0) || false;
    const value = {
        ...state.value,
        end: newEndDatetime,
    };
    return {
        ...state,
        endTime: action.time,
        value,
        validationError,
        isDirty: true,
    };
}
function handleReset(state, action) {
    const { timeZone, value } = action;
    return {
        ...state,
        value: { start: value?.start || null, end: value?.end || null },
        endDate: value?.end ? datetimeToCalendarDate(value.end, timeZone) : null,
        validationError: false,
        endTime: value?.end ? datetimeToTime(value.end, timeZone) : null,
        isDirty: false,
        startDate: value?.start ? datetimeToCalendarDate(value.start, timeZone) : null,
        startTime: value?.start ? datetimeToTime(value.start, timeZone) : null,
        timeZone,
        openInputPopover: null,
    };
}
function handleSetOpenPopover(state, action) {
    const { isOpen, inputName } = action;
    return {
        ...state,
        openInputPopover: isOpen ? inputName : null,
    };
}
export function useDatetimeRange({ fieldProperties, locale, value }) {
    const previousValue = usePrevious(value);
    const timeZone = useMemo(() => fieldProperties.timeZone && isValidTimeZone(fieldProperties.timeZone) ? fieldProperties.timeZone : 'GMT', [fieldProperties.timeZone]);
    const initialDate = useMemo(() => (fieldProperties.defaultDate ? toDatetime(fieldProperties.defaultDate, timeZone) : null), [fieldProperties.defaultDate, timeZone]);
    const initialState = {
        value: { start: value?.start || null, end: value?.end || null },
        endDate: value?.end ? datetimeToCalendarDate(value.end, timeZone) : null,
        validationError: false,
        endTime: value?.end ? datetimeToTime(value.end, timeZone) : null,
        isDirty: false,
        startDate: value?.start ? datetimeToCalendarDate(value.start, timeZone) : null,
        startTime: value?.start ? datetimeToTime(value.start, timeZone) : null,
        timeZone,
        openInputPopover: null,
    };
    const [state, dispatch] = useReducer(datetimeRangeReducer, initialState);
    const minDateValue = useMemo(() => toDatetime(fieldProperties.minDate || new Date(1970, 0, 1), timeZone, locale), [fieldProperties.minDate, locale, timeZone]);
    const maxDateValue = useMemo(() => toDatetime(fieldProperties.maxDate || new Date(2100, 11, 31), timeZone, locale), [fieldProperties.maxDate, locale, timeZone]);
    const handleStartDateChange = useCallback((date) => {
        dispatch({ type: 'SET_START_DATE', date });
    }, []);
    const handleEndDateChange = useCallback((date) => {
        dispatch({ type: 'SET_END_DATE', date });
    }, []);
    const handleStartTimeChange = useCallback((time) => {
        dispatch({ type: 'SET_START_TIME', time });
    }, []);
    const handleEndTimeChange = useCallback((time) => {
        dispatch({ type: 'SET_END_TIME', time });
    }, []);
    const handlePopoverOpenChange = useCallback((isOpen, inputName) => {
        dispatch({ type: 'SET_OPEN_POPOVER_ON_INPUT', isOpen, inputName });
    }, []);
    useEffect(() => {
        if (value !== state.value && previousValue !== value) {
            dispatch({ type: 'RESET', value, timeZone });
        }
    }, [value, timeZone, state.value, previousValue]);
    useEffect(() => {
        if (initialDate) {
            dispatch({ type: 'RESET', value: { start: initialDate, end: initialDate }, timeZone });
        }
    }, [initialDate, timeZone]);
    return {
        currentDates: state.value,
        endDate: state.endDate,
        validationError: state.validationError,
        endTime: state.endTime,
        handleEndDateChange,
        handleEndTimeChange,
        handlePopoverOpenChange,
        handleStartDateChange,
        handleStartTimeChange,
        isDirty: state.isDirty,
        maxDateValue,
        minDateValue,
        openInputPopover: state.openInputPopover,
        startDate: state.startDate,
        startTime: state.startTime,
        timeZone,
    };
}
//# sourceMappingURL=use-datetime-range.js.map