import React, { useMemo, useRef } from 'react';
import { connect } from 'react-redux';
import { CarbonWrapper } from '../carbon-wrapper';
import { mapDispatchToProps, mapStateToProps } from '../field-base-component';
import { localize } from '../../../service/i18n-service';
import { useDatetimeRange } from './use-datetime-range';
import { generateFieldId, getFieldIndicatorStatus, isFieldDisabled, isFieldReadOnly, getLabelTitle, } from '../carbon-helpers';
import { DatetimeInputComponent } from './datetime-input-component';
import { FieldLabel, HelperText } from '../carbon-utility-components';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { triggerFieldEvent } from '../../../utils/events';
import { useFocus } from '../../../utils/hooks/effects/use-focus';
export function DatetimeRangeComponent({ locale, elementId, screenId, setFieldValue, validate, isInFocus, fieldProperties, isParentDisabled, isParentHidden, contextType, validationErrors, value, ...rest }) {
    const containerRef = useRef(null);
    const startDateRef = useRef(null);
    const endDateRef = useRef(null);
    useFocus(startDateRef, isInFocus);
    const triggerChangeListener = React.useCallback(() => () => {
        triggerFieldEvent(screenId, elementId, 'onChange');
    }, [screenId, elementId]);
    const onChange = React.useCallback((newValue) => {
        handleChange(elementId, newValue, setFieldValue, validate, triggerChangeListener());
    }, [elementId, setFieldValue, triggerChangeListener, validate]);
    const { currentDates, endDate, validationError, endTime, handleEndDateChange, handleEndTimeChange, handlePopoverOpenChange, handleStartDateChange, handleStartTimeChange, isDirty, maxDateValue, minDateValue, openInputPopover, startDate, startTime, timeZone, } = useDatetimeRange({
        elementId,
        fieldProperties,
        locale,
        onChange,
        screenId,
        setFieldValue,
        validate,
        value,
    });
    const { isTitleHidden, helperText, isHelperTextHidden, isTimeZoneHidden } = fieldProperties;
    const { error, warning, info } = React.useMemo(() => getFieldIndicatorStatus({
        validationErrors,
        screenId,
        value,
        fieldProperties,
        isParentDisabled,
        isParentHidden,
    }), [fieldProperties, isParentDisabled, isParentHidden, screenId, validationErrors, value]);
    const isDisabled = useMemo(() => isFieldDisabled(screenId, fieldProperties, currentDates, null), [currentDates, fieldProperties, screenId]);
    const isReadOnly = useMemo(() => isFieldReadOnly(screenId, fieldProperties, currentDates, null), [currentDates, fieldProperties, screenId]);
    const fieldId = useMemo(() => generateFieldId({ screenId, elementId, contextType, fieldProperties, isNested: false }), [contextType, elementId, fieldProperties, screenId]);
    const resolvedTitle = useMemo(() => getLabelTitle(screenId, fieldProperties, null), [fieldProperties, screenId]);
    const startLabel = localize('@sage/xtrem-ui/date-time-range-start-date', 'Start');
    const endLabel = localize('@sage/xtrem-ui/date-time-range-end-date', 'End');
    const ariaLabelTitle = resolvedTitle || localize('@sage/xtrem-ui/datetime-range-aria-label', 'Date and time range');
    const onClick = React.useCallback((ev) => {
        if (!ev.target || !containerRef.current) {
            return;
        }
        if (!containerRef.current?.contains(ev.target) && !validationError) {
            onChange(currentDates.start && currentDates.end ? currentDates : null);
        }
    }, [currentDates, onChange, validationError]);
    React.useEffect(() => {
        if (isDirty) {
            window.addEventListener('click', onClick);
        }
        return () => {
            window.removeEventListener('click', onClick);
        };
    }, [isDirty, onClick]);
    const getValidationErrors = React.useCallback(() => {
        return validationError
            ? localize('@sage/xtrem-ui/datetime-range-end-date-error', 'You need to enter an End date later than the Start date')
            : '';
    }, [validationError]);
    return (React.createElement(CarbonWrapper, { contextType: contextType, elementId: elementId, screenId: screenId, fieldProperties: fieldProperties, isParentDisabled: isParentDisabled, isParentHidden: isParentHidden, locale: locale, ...rest, noReadOnlySupport: true, className: "e-datetime-range-field", componentName: "datetime-range" },
        !isTitleHidden && resolvedTitle && (React.createElement(FieldLabel, { label: resolvedTitle, errorMessage: error, warningMessage: warning, infoMessage: info })),
        React.createElement("div", { className: "e-datetime-range-container", ref: containerRef },
            React.createElement("div", { className: "e-date-input-wrapper" },
                React.createElement(DatetimeInputComponent, { screenId: screenId, elementId: elementId, "aria-label": `${ariaLabelTitle} - ${startLabel}`, date: startDate, fieldId: `${fieldId}-start-date`, inputRef: startDateRef, isDisabled: isDisabled, isReadOnly: isReadOnly, isTimeZoneHidden: isTimeZoneHidden, locale: locale, maxDate: maxDateValue, minDate: minDateValue, onDateChange: handleStartDateChange, onTimeChange: handleStartTimeChange, onPopperOpenChange: handlePopoverOpenChange, isPopoverOpen: openInputPopover === 'start', rangeStartDate: null, time: startTime, timeZone: timeZone, title: startLabel, type: "start" })),
            React.createElement("div", { className: "e-date-input-wrapper" },
                React.createElement(DatetimeInputComponent, { screenId: screenId, elementId: elementId, "aria-label": `${ariaLabelTitle} - ${endLabel}`, date: endDate, fieldId: `${fieldId}-end-date`, inputRef: endDateRef, isDisabled: isDisabled, isReadOnly: isReadOnly, isTimeZoneHidden: isTimeZoneHidden, locale: locale, maxDate: maxDateValue, minDate: minDateValue, onDateChange: handleEndDateChange, onTimeChange: handleEndTimeChange, onPopperOpenChange: handlePopoverOpenChange, isPopoverOpen: openInputPopover === 'end', rangeStartDate: startDate, time: endTime, timeZone: timeZone, title: endLabel, type: "end", validationError: getValidationErrors() }))),
        !isHelperTextHidden && helperText && React.createElement(HelperText, { helperText: helperText })));
}
export const ConnectedDatetimeRangeComponent = connect(mapStateToProps(), mapDispatchToProps())(DatetimeRangeComponent);
export default ConnectedDatetimeRangeComponent;
//# sourceMappingURL=datetime-range-component.js.map