import * as React from 'react';
import { connect } from 'react-redux';
import { ContextType } from '../../../types';
import { getElementAccessStatus } from '../../../utils/access-utils';
import { getDataTestIdAttribute, isHidden } from '../../../utils/dom';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { calculateContainerWidth, calculateFieldWidth, getGutterSize } from '../../../utils/responsive-utils';
import { getPageDefinitionFromState } from '../../../utils/state-utils';
import { getFieldTitle } from '../../field/carbon-helpers';
import { GridColumn, GridRow } from '@sage/xtrem-ui-components';
import { TileGridManager } from './tile-grid-manager';
import { ConnectedTileField } from './tile-field';
export function TileComponent(props) {
    const computedWidth = calculateContainerWidth(props.browser.is, props.availableColumns, props.fieldProperties.width);
    const gridGutter = getGutterSize(props.browser.is);
    const gridVerticalMargin = props.contextType === ContextType.page ? 16 : 0;
    const isTitleHidden = resolveByValue({
        screenId: props.screenId,
        fieldValue: null,
        rowValue: null,
        propertyValue: props.fieldProperties.isTitleHidden,
        skipHexFormat: true,
    });
    const isTileHidden = isHidden(props.item, props.browser) ||
        props.accessRule === 'unavailable' ||
        resolveByValue({
            screenId: props.screenId,
            propertyValue: props.fieldProperties.isHidden,
            skipHexFormat: true,
            rowValue: null,
        });
    const isDisabled = props.isParentDisabled ||
        resolveByValue({
            screenId: props.screenId,
            propertyValue: props.fieldProperties.isDisabled,
            skipHexFormat: true,
            rowValue: null,
        });
    const title = getFieldTitle(props.screenId, props.fieldProperties, null);
    const classes = ['e-container-parent', 'e-tile-parent'];
    if (isTileHidden) {
        classes.push('e-hidden');
    }
    if (props.contextType) {
        classes.push(`e-tile-context-${props.contextType}`);
    }
    return (React.createElement(GridColumn, { className: classes.join(' '), columnSpan: computedWidth },
        React.createElement("div", { className: "e-tile", "data-testid": getDataTestIdAttribute('tile', title, props.item.$containerId) },
            !isTitleHidden && title && (React.createElement("div", { className: "e-tile-header" },
                React.createElement("h3", { className: "e-tile-title", "data-testid": "e-tile-title" }, title))),
            React.createElement(GridRow, { columns: computedWidth, gutter: gridGutter, margin: 16, verticalMargin: gridVerticalMargin }, props.item.$layout?.$items
                .filter(i => !!i.$bind && props.columnInfo[i.$bind])
                .map((i) => {
                const columnInfo = props.columnInfo[i.$bind];
                const gridColumnClasses = ['e-field-grid-column'];
                if (columnInfo.isFirstInRow) {
                    gridColumnClasses.push('e-grid-column-first-in-row');
                }
                if (columnInfo.isLastInRow) {
                    gridColumnClasses.push('e-grid-column-last-in-row');
                }
                return (React.createElement(GridColumn, { key: i.$bind, className: gridColumnClasses.join(' '), columnSpan: columnInfo.columnSpan, columnStart: columnInfo.columnStart },
                    React.createElement(ConnectedTileField, { elementId: i.$bind, screenId: props.screenId, availableColumns: computedWidth, item: i, isParentDisabled: isDisabled })));
            })))));
}
TileComponent.displayName = 'TileComponent';
const mapStateToProps = (state, props) => {
    if (!props.item.$layout) {
        throw new Error(`No layout found for ${props.screenId} ${props.item.$containerId} tile.`);
    }
    const elementId = props.item.$containerId;
    if (!elementId) {
        throw new Error(`Cannot render tile container because layout definition does not have a container ID. Screen:${props.screenId}`);
    }
    const pageDefinition = getPageDefinitionFromState(props.screenId);
    const pageProperties = pageDefinition.metadata.uiComponentProperties[props.screenId];
    const fieldProperties = pageDefinition.metadata.uiComponentProperties[elementId];
    const accessRule = getElementAccessStatus({
        accessBindings: pageDefinition.accessBindings || {},
        bind: elementId,
        elementProperties: fieldProperties,
        contextNode: pageProperties?.node,
        nodeTypes: state.nodeTypes,
        dataTypes: state.dataTypes,
    });
    const accessBindings = pageDefinition.accessBindings || {};
    const manager = new TileGridManager(props.availableColumns);
    // Precalculates all columns of the responsive grid to be able to anticipate the structure, such as which fields are going to be first or last in a row.
    props.item.$layout.$items.forEach(i => {
        const fieldElementId = String(i.$bind || i.$containerId);
        const elementProperties = pageDefinition.metadata.uiComponentProperties[fieldElementId];
        const fieldAccessRule = getElementAccessStatus({
            accessBindings,
            bind: fieldElementId,
            elementProperties,
            contextNode: pageProperties?.node,
            nodeTypes: state.nodeTypes,
            dataTypes: state.dataTypes,
        });
        const componentType = elementProperties._controlObjectType;
        const columnSpan = calculateFieldWidth(state.browser.is, componentType, props.availableColumns, i.$isFullWidth, i.$columnWidth);
        const isColHidden = isHidden(i, state.browser) ||
            fieldAccessRule === 'unavailable' ||
            resolveByValue({
                propertyValue: elementProperties.isHidden,
                rowValue: null,
                screenId: props.screenId,
                fieldValue: null,
                skipHexFormat: true,
            });
        manager.appendItem({
            elementId: fieldElementId,
            columnSpan,
            isHidden: isColHidden,
        });
    });
    const columnInfo = manager.getState();
    return {
        ...props,
        columnInfo,
        accessRule,
        fieldProperties,
        screenType: pageDefinition.type,
        browser: state.browser,
    };
};
export const ConnectedTileComponent = connect(mapStateToProps)(TileComponent);
//# sourceMappingURL=tile-component.js.map