import { AccordionGroup } from 'carbon-react/esm/components/accordion';
import * as React from 'react';
import { connect } from 'react-redux';
import styled from 'styled-components';
import * as xtremRedux from '../../../redux';
import { RenderingRouter } from '../../../render/rendering-router';
import { localize } from '../../../service/i18n-service';
import { ContextType } from '../../../types';
import { getElementAccessStatus } from '../../../utils/access-utils';
import { getDataTestIdAttribute, isHidden } from '../../../utils/dom';
import { useParentElementSize } from '../../../utils/hooks/effects/use-parent-element-size';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { getGutterSize } from '../../../utils/responsive-utils';
import { getPageDefinitionFromState } from '../../../utils/state-utils';
import { getFieldTitle } from '../../field/carbon-helpers';
import { FieldKey } from '../../types';
import { GridRow } from '@sage/xtrem-ui-components';
import ButtonMinor from 'carbon-react/esm/components/button-minor';
const CustomStyledAccordionGroup = styled(AccordionGroup) `
    overflow: hidden;
`;
const hasBlocks = (props) => !!props.item.$layout.$items.find(x => x.$category === 'block');
const isSectionHeaderDisplayed = (props) => props.pageMode === 'tabs' ? props.fieldProperties.isTitleHidden === false : !props.fieldProperties.isTitleHidden;
const isFullScreenIconDisplayed = (props) => hasBlocks(props) && props.pageMode === 'default' && props.contextType === ContextType.page;
const isCollapseIconDisplayed = (props, isFullScreen) => hasBlocks(props) && props.pageMode === 'default' && !isFullScreen && props.contextType === ContextType.page;
const canRenderSectionBody = (props, isFullScreen) => props.fieldProperties.isOpen !== false || isFullScreen;
const onOpen = (props) => {
    props.setFieldProperties(props.item.$containerId, { ...props.fieldProperties, isOpen: true });
    props.updateUserCustomization([
        { componentId: props.item.$containerId, propertyName: 'isOpen', propertyValue: true },
    ]);
};
const onClose = (props) => {
    props.setFieldProperties(props.item.$containerId, { ...props.fieldProperties, isOpen: false });
    props.updateUserCustomization([
        { componentId: props.item.$containerId, propertyName: 'isOpen', propertyValue: false },
    ]);
};
const isSectionDisabled = (props) => props.accessRule === 'unauthorized' ||
    resolveByValue({
        screenId: props.screenId,
        fieldValue: null,
        rowValue: null,
        propertyValue: props.fieldProperties.isDisabled,
        skipHexFormat: true,
    });
const isSectionHidden = (props) => isHidden(props.item, props.browser) ||
    !!resolveByValue({
        propertyValue: props.fieldProperties.isHidden,
        skipHexFormat: true,
        rowValue: null,
        fieldValue: null,
        screenId: props.screenId,
    }) ||
    props.accessRule === 'unavailable';
const getFixedHeight = (props, height) => {
    // Fixed height is only applied to the section if it contains a single field (and not a block), such as a table or a code editor.
    // In this case fixedHeight is the parent element's height (whatever is available below the header)
    const shouldUseFixedHeight = props.contextType === ContextType.page &&
        props.item.$layout?.$items.length === 1 &&
        props.firstFieldType &&
        props.browser?.greaterThan.s &&
        Object.values(FieldKey).includes(props.firstFieldType);
    if (!shouldUseFixedHeight) {
        return undefined;
    }
    // Here we need to return something not falsy.
    // AG Grid has a bug and if the grid is initiated with `autoHeight` mode, it will always remain in auto height
    if (!height) {
        return 10;
    }
    if (props.hasFooter) {
        return height - 90;
    }
    return height - 20;
};
const getComponentClass = (props, isFullScreen) => {
    const classes = ['e-section'];
    if (isFullScreen) {
        classes.push('e-section-fullscreen');
    }
    if (props.pageMode === 'tabs') {
        classes.push('e-single-section');
    }
    if (isSectionHidden(props)) {
        classes.push('e-hidden');
    }
    if (props.contextType) {
        classes.push(`e-section-context-${props.contextType}`);
    }
    if (props.item.$layout?.$items.length === 1 && Object.values(FieldKey).includes(props.firstFieldType)) {
        classes.push('e-section-single-field-child');
    }
    return classes.join(' ');
};
export function SectionComponent(props) {
    const [isFullScreen, setFullScreen] = React.useState(false);
    const [containerRef, { height }] = useParentElementSize();
    const fixedHeight = getFixedHeight(props, height);
    const title = getFieldTitle(props.screenId, props.fieldProperties, null);
    const collapseSectionLabel = localize('@sage/xtrem-ui/collapse-section', 'Collapse section');
    const openSectionLabel = localize('@sage/xtrem-ui/open-section', 'Open section');
    const closeFullScreenLabel = localize('@sage/xtrem-ui/close-full-screen', 'Close fullscreen');
    const openFullScreenLabel = localize('@sage/xtrem-ui/open-full-screen', 'Open fullscreen');
    const onOpenFullscreen = () => setFullScreen(true);
    const onCloseFullscreen = () => setFullScreen(false);
    const renderNormalBody = () => {
        const gridGutter = getGutterSize(props.browser.is);
        return (React.createElement(GridRow, { className: "e-section-body", margin: 0, gutter: gridGutter, verticalMargin: 0, columns: props.availableColumns }, props.item.$layout.$items.map(item => (React.createElement(RenderingRouter, { screenId: props.screenId, key: item.$containerId, item: item, contextType: props.contextType, availableColumns: props.availableColumns, isParentDisabled: props.isParentDisabled || isSectionDisabled(props), isParentHidden: props.isParentHidden || isSectionHidden(props), fixedHeight: fixedHeight })))));
    };
    const renderAccordionBody = () => {
        return (React.createElement(CustomStyledAccordionGroup, null, props.item.$layout.$items.map(item => (React.createElement(RenderingRouter, { screenId: props.screenId, item: item, key: item.$containerId, contextType: ContextType.accordion, availableColumns: props.availableColumns, isParentDisabled: props.isParentDisabled || isSectionDisabled(props) })))));
    };
    return (React.createElement("section", { ref: containerRef, className: getComponentClass(props, isFullScreen), "data-testid": getDataTestIdAttribute('section', title, props.item.$containerId), id: props.item.$containerId },
        isSectionHeaderDisplayed(props) && (React.createElement("div", { className: "e-section-header" },
            !!title && React.createElement("h2", { className: "e-section-title" }, title),
            !title && React.createElement("span", { className: "e-section-title" }),
            isFullScreenIconDisplayed(props) &&
                (isFullScreen ? (React.createElement(ButtonMinor, { key: "close", "data-testid": "e-icon-fullscreen", onClick: onCloseFullscreen, iconType: "close", iconTooltipMessage: closeFullScreenLabel, "aria-label": closeFullScreenLabel, size: "small", buttonType: "tertiary" })) : (React.createElement(ButtonMinor, { key: "open", "data-testid": "e-icon-fullscreen", onClick: onOpenFullscreen, iconType: "link", iconTooltipMessage: openFullScreenLabel, "aria-label": openFullScreenLabel, size: "small", buttonType: "tertiary" }))),
            isCollapseIconDisplayed(props, isFullScreen) &&
                (props.fieldProperties.isOpen ? (React.createElement(ButtonMinor, { key: "toggle-close", "data-testid": "e-icon-close", onClick: () => onClose(props), iconType: "chevron_down", iconTooltipMessage: collapseSectionLabel, "aria-label": collapseSectionLabel, size: "small", buttonType: "tertiary" })) : (React.createElement(ButtonMinor, { key: "toggle-open", "data-testid": "e-icon-open", onClick: () => onOpen(props), iconType: "chevron_up", iconTooltipMessage: openSectionLabel, "aria-label": openSectionLabel, size: "small", buttonType: "tertiary" }))))),
        canRenderSectionBody(props, isFullScreen) &&
            (props.fieldProperties.mode === 'normal' || !props.fieldProperties.mode) &&
            renderNormalBody(),
        canRenderSectionBody(props, isFullScreen) &&
            props.fieldProperties.mode === 'accordion' &&
            renderAccordionBody()));
}
const mapStateToProps = (state, props) => {
    const elementId = props.item.$containerId;
    const pageDefinition = getPageDefinitionFromState(props.screenId);
    const pageProperties = pageDefinition.metadata.uiComponentProperties[props.screenId];
    const pageMode = pageProperties ? pageProperties.mode : 'default';
    const fieldProperties = pageDefinition.metadata.uiComponentProperties[elementId];
    const accessRule = getElementAccessStatus({
        accessBindings: pageDefinition.accessBindings || {},
        bind: elementId,
        elementProperties: fieldProperties,
        contextNode: pageProperties?.node,
        nodeTypes: state.nodeTypes,
        dataTypes: state.dataTypes,
    });
    // Checks whether the first element is a field and if so what type
    const firstFieldId = props.item.$layout?.$items[0]?.$bind;
    const firstFieldType = firstFieldId
        ? pageDefinition.metadata.uiComponentProperties[firstFieldId]._controlObjectType
        : undefined;
    return {
        ...props,
        fieldProperties,
        browser: state.browser,
        firstFieldType,
        pageMode,
        accessRule,
        setFieldProperties: xtremRedux.actions.actionStub,
        updateUserCustomization: xtremRedux.actions.actionStub,
    };
};
const mapDispatchToProps = (dispatch, props) => ({
    setFieldProperties: (bind, state) => {
        dispatch(xtremRedux.actions.setFieldProperties(props.screenId, bind, state));
    },
    updateUserCustomization: (userCustomizations) => {
        dispatch(xtremRedux.actions.updateUserCustomization(props.screenId, userCustomizations));
    },
});
export const ConnectedSectionComponent = connect(mapStateToProps, mapDispatchToProps)(SectionComponent);
//# sourceMappingURL=section-component.js.map