import Dialog from 'carbon-react/esm/components/dialog';
import DialogFullScreen from 'carbon-react/esm/components/dialog-full-screen';
import Heading from 'carbon-react/esm/components/heading';
import Sidebar from 'carbon-react/esm/components/sidebar';
import { noop } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import * as xtremRedux from '../../../redux';
import { openDirtyPageConfirmationDialog } from '../../../service/dirty-state-service';
import { focusTopPage } from '../../../utils/dom';
import { getPageBodyColumnCount } from '../../../utils/responsive-utils';
import { isScreenDefinitionDirty } from '../../../utils/state-utils';
import { TableControlObject } from '../../field/table/table-control-object';
import { DialogBodyContent } from './body/dialog-body-content';
import { collectSections } from './dialog-utils';
import { DialogHeader } from './dialog-header';
import { DialogTitle } from './dialog-title';
import { getVisibleSections } from '../../../utils/layout-utils';
/**
 * This selector defines what elements are discovered by Carbon's dialog focus trap.
 */
const FOCUS_TRAP_SELECTOR = [
    'button.e-ui-confirmation-dialog-button',
    'button.e-ui-select-lookup-button',
    'div.e-ui-select-lookup-button button',
    'button.e-xtrem-tab-item',
    '.e-dialog-title-back-button-wrapper button',
    'button[data-element="close"]',
    '.e-file-deposit-field-content-wrapper button',
    'button.e-page-crud-button',
    '.e-business-action button',
    '[href]',
    'input:not([type="hidden"]):not([disabled]):not(.e-hidden)',
    'select:not([disabled])',
    'textarea:not([disabled])',
    '[tabindex]:not(.DayPicker-wrapper):not(.DayPicker-Day)',
    '[data-component="breadcrumbs"] button',
];
export class DialogComponent extends React.Component {
    constructor(props) {
        super(props);
        this.normalDialogRef = React.createRef();
        this.closeDialog = async () => {
            if (this.props.dialog.screenId &&
                this.props.dialog.type === 'page' &&
                !this.props.dialog.options.skipDirtyCheck &&
                xtremRedux.getStore().getState().screenDefinitions[this.props.dialog.screenId] &&
                isScreenDefinitionDirty(xtremRedux.getStore().getState().screenDefinitions[this.props.dialog.screenId])) {
                try {
                    await openDirtyPageConfirmationDialog(this.props.dialog.screenId);
                }
                catch {
                    return;
                }
            }
            if (this.props.dialog.options.resolveOnCancel) {
                this.props.dialog.dialogControl.resolve({});
            }
            else {
                this.props.dialog.dialogControl.cancel();
            }
            await this.props.closeDialog(this.props.dialog.dialogId);
            focusTopPage();
        };
        this.getDialogContentGridSize = () => {
            if (this.props.browser.is.xs) {
                return 4;
            }
            if (this.props.dialog.options.fullScreen) {
                return getPageBodyColumnCount(this.props.browser?.is);
            }
            if (this.props.dialog.options.size === 'extra-large') {
                return 8;
            }
            return 4;
        };
        this.getActiveSection = () => {
            if (this.props.dialog.content && this.props.dialog.type === 'page' && this.props.dialog.screenId) {
                const pageDefinition = xtremRedux.getStore().getState().screenDefinitions[this.props.dialog.screenId];
                if (!pageDefinition) {
                    return null;
                }
                if (pageDefinition.activeSection) {
                    return pageDefinition.activeSection;
                }
                const visibleSections = getVisibleSections(pageDefinition, {});
                return visibleSections[0]?.$containerId || null;
            }
            return this.state.activeSection;
        };
        /**
         *
         * @description Carbon Dialog focustrap and AgGrid internal state focus handling are incompatible.
         * They compete and overlap each other for the same browser events and imperative effects (.focus()).
         * Therefore, we disable the focus trap when the page dialog includes a table within.
         * For the same reasons we disable the 'Esc' key.
         *
         * `disableFocusTrap: boolean` does not currently work... so we use the hidden `bespokeFocusTrap`
         * callback property with a noop function, which does the same. We should consider a
         * replacement in the future, when `disableFocusTrap` is back in the Carbon library.
         *
         * Giving its current state, disabling by default the FocusTrap in all/most of the dialogs might
         * not be completely discarded. Browsers defaults may be enough for most cases.
         *
         */
        this.hasTableField = () => {
            const screenId = this.props.dialog.screenId;
            const ctrlObjs = (this.props.dialog.type === 'page' &&
                screenId &&
                xtremRedux.getStore().getState().screenDefinitions[screenId]?.metadata?.controlObjects) ||
                {};
            return Boolean(Object.values(ctrlObjs).find(ctrlObj => ctrlObj instanceof TableControlObject && !ctrlObj.isHidden));
        };
        this.setSelectedSection = (selectedSection) => {
            if (this.props.dialog.content && this.props.dialog.type === 'custom') {
                this.setState({ activeSection: selectedSection });
            }
            else if (this.props.dialog.content && this.props.dialog.type === 'page' && this.props.dialog.screenId) {
                this.props.setActiveSection(this.props.dialog.screenId, selectedSection);
            }
        };
        this.onStepOneSection = (direction) => {
            if (this.props.dialog.content && this.props.dialog.type === 'page' && this.props.dialog.screenId) {
                this.props.stepOneSection(this.props.dialog.screenId, direction);
            }
        };
        this.renderSidebarDialog = () => {
            const forwardRef = React.createRef();
            return (React.createElement(Sidebar, { size: this.props.dialog.options.size || 'medium', key: this.props.dialog.dialogId, onCancel: this.closeDialog, open: true, focusableSelectors: FOCUS_TRAP_SELECTOR.join(', '), "aria-label": this.props.dialog.title, header: React.createElement("div", { className: "e-dialog-sidebar-heading" },
                    React.createElement(Heading, { divider: false, subheader: this.props.dialog.subtitle, mb: "12px !important", title: this.props.dialog.type === 'page' ? (React.createElement(DialogTitle, { dialog: this.props.dialog, baseTopMargin: "0px", onCloseDialog: this.closeDialog })) : (this.props.dialog.title) }),
                    React.createElement(DialogHeader, { dialog: this.props.dialog, setSelectedSection: this.setSelectedSection, selectedSection: this.getActiveSection() })) }, this.renderDialogContent(forwardRef, true)));
        };
        this.renderNormalDialog = () => {
            const forwardRef = React.createRef();
            const hasTableField = this.hasTableField();
            return (React.createElement(Dialog, { key: this.props.dialog.dialogId, open: true, showCloseIcon: true, size: this.props.dialog.options.size || 'medium', focusableSelectors: FOCUS_TRAP_SELECTOR.join(', '), "aria-label": this.props.dialog.title, title: React.createElement(React.Fragment, null,
                    React.createElement(Heading, { divider: false, subheader: this.props.dialog.subtitle, mb: "12px !important", title: this.props.dialog.type === 'page' ? (React.createElement(DialogTitle, { dialog: this.props.dialog, baseTopMargin: "0px", onCloseDialog: this.closeDialog })) : (this.props.dialog.title) }),
                    !this.props.dialog.options.isDuplicate && (React.createElement(DialogHeader, { dialog: this.props.dialog, setSelectedSection: this.setSelectedSection, selectedSection: this.getActiveSection() }))), onCancel: this.closeDialog, className: "e-dialog-normal", disableAutoFocus: true, disableEscKey: hasTableField, bespokeFocusTrap: hasTableField ? noop : undefined }, this.renderDialogContent(forwardRef, true)));
        };
        this.renderStickerDialog = () => (React.createElement(Dialog, { className: "e-dialog-sticker", key: this.props.dialog.dialogId, "aria-label": this.props.dialog.title, open: true, showCloseIcon: true, size: this.props.dialog.options.size || 'small', focusableSelectors: FOCUS_TRAP_SELECTOR.join(', '), title: React.createElement(React.Fragment, null,
                React.createElement(Heading, { divider: false, subheader: this.props.dialog.subtitle, mb: "12px !important", title: this.props.dialog.type === 'page' ? (React.createElement(DialogTitle, { dialog: this.props.dialog, onCloseDialog: this.closeDialog })) : (this.props.dialog.title) }),
                React.createElement(DialogHeader, { dialog: this.props.dialog, setSelectedSection: this.setSelectedSection, selectedSection: this.getActiveSection() })), enableBackgroundUI: true, onCancel: this.closeDialog, disableAutoFocus: true }, this.renderDialogContent(undefined, true)));
        this.renderFullScreenStickerDialog = () => (React.createElement(DialogFullScreen, { className: "e-dialog-sticker", disableAutoFocus: true, focusableSelectors: FOCUS_TRAP_SELECTOR.join(', '), headerChildren: React.createElement(DialogHeader, { dialog: this.props.dialog, setSelectedSection: this.setSelectedSection, selectedSection: this.getActiveSection() }), key: this.props.dialog.dialogId, onCancel: this.closeDialog, open: true, showCloseIcon: true, subtitle: this.props.dialog.subtitle, "aria-label": this.props.dialog.title, title: this.props.dialog.type === 'page' ? (React.createElement(DialogTitle, { dialog: this.props.dialog, onCloseDialog: this.closeDialog })) : (this.props.dialog.title) }, this.renderDialogContent(undefined, true)));
        this.renderFullScreenDialog = () => {
            const forwardRef = React.createRef();
            return (React.createElement(DialogFullScreen, { disableAutoFocus: true, focusableSelectors: FOCUS_TRAP_SELECTOR.join(', '), headerChildren: React.createElement(DialogHeader, { dialog: this.props.dialog, setSelectedSection: this.setSelectedSection, selectedSection: this.getActiveSection() }), key: this.props.dialog.dialogId, onCancel: this.closeDialog, open: true, showCloseIcon: true, subtitle: this.props.dialog.subtitle, "aria-label": this.props.dialog.title, title: this.props.dialog.type === 'page' ? (React.createElement(DialogTitle, { dialog: this.props.dialog, onCloseDialog: this.closeDialog, baseBottomMargin: "12px", baseTopMargin: "12px" })) : (this.props.dialog.title) }, this.renderDialogContent(forwardRef, true)));
        };
        this.state = {
            activeSection: this.props.dialog.content && this.props.dialog.type === 'custom'
                ? collectSections(props.dialog, props.screenDefinitions)[0]?.id || null
                : null,
        };
    }
    componentDidUpdate(prevProps) {
        if (this.normalDialogRef.current) {
            this.normalDialogRef.current?.centerDialog();
        }
        if (!prevProps.dialog.content && this.props.dialog.content && this.props.dialog.type === 'custom') {
            this.setState({
                activeSection: collectSections(this.props.dialog, xtremRedux.getStore().getState().screenDefinitions)[0]?.id ||
                    null,
            });
        }
    }
    renderDialogContent(defaultFocusRef, noHeader = false) {
        const activeSection = this.getActiveSection();
        const sections = collectSections(this.props.dialog, xtremRedux.getStore().getState().screenDefinitions);
        const currentSectionIndex = sections.findIndex(s => s?.id === activeSection);
        const gridColumnCount = this.getDialogContentGridSize();
        const dialogContentClasses = [
            'e-dialog-content',
            `e-dialog-type-${this.props.dialog.type || 'unknown'}`,
        ];
        const screenDefinition = this.props.dialog.screenId
            ? xtremRedux.getStore().getState().screenDefinitions[this.props.dialog.screenId]
            : undefined;
        const isFirstSection = !activeSection || currentSectionIndex === 0;
        const isLastSection = sections.length === currentSectionIndex + 1;
        return (React.createElement("div", { className: dialogContentClasses.join(' ') },
            React.createElement("div", { className: "e-dialog-body", "data-testid": "e-dialog-body" },
                React.createElement(DialogBodyContent, { dialog: this.props.dialog, screenDefinition: screenDefinition, availableColumns: gridColumnCount, defaultFocusRef: defaultFocusRef, selectedSection: activeSection, noHeader: noHeader, onStepOneSection: this.onStepOneSection, isFirstSection: isFirstSection, isLastSection: isLastSection }))));
    }
    render() {
        if (this.props.dialog.type !== 'page' &&
            (Array.isArray(this.props.dialog.content)
                ? this.props.dialog.content
                : [this.props.dialog.content])
                .filter(c => c !== undefined && c !== null)
                .filter(c => !c.isHidden).length === 0) {
            return null;
        }
        if (this.props.isFullScreen || this.props.dialog.options.fullScreen) {
            return this.props.dialog.isSticker ? this.renderFullScreenStickerDialog() : this.renderFullScreenDialog();
        }
        if (this.props.dialog.isSticker) {
            return this.renderStickerDialog();
        }
        if (this.props.dialog.options.rightAligned) {
            return this.renderSidebarDialog();
        }
        return this.renderNormalDialog();
    }
}
const mapStateToProps = (state, props) => {
    const dialog = state.activeDialogs[props.dialogKey];
    return {
        ...props,
        browser: state.browser,
        closeDialog: xtremRedux.actions.actionStub,
        setActiveSection: xtremRedux.actions.actionStub,
        stepOneSection: xtremRedux.actions.actionStub,
        dialog,
        screenDefinitions: state.screenDefinitions,
    };
};
const mapDispatchToProps = (dispatch) => ({
    closeDialog: (dialogId) => dispatch(xtremRedux.actions.closeDialog(dialogId)),
    setActiveSection: (screenId, selectedSection) => dispatch(xtremRedux.actions.setActiveSection(screenId, selectedSection)),
    stepOneSection: (screenId, direction) => dispatch(xtremRedux.actions.stepOneSection(screenId, direction)),
});
export const ConnectedDialogComponent = connect(mapStateToProps, mapDispatchToProps)(DialogComponent);
//# sourceMappingURL=dialog-component.js.map