import { without } from 'lodash';
import { addCustomColumnToNestedFields, componentsWithLookup } from '../service/customization-service';
import { convertDeepBindToPathNotNull } from '../utils/nested-field-utils';
import { arrayMoveMutate } from '../utils/transformers';
export const findColumnIndex = (columns, match) => {
    const insertBeforeParts = match.split('__');
    // First try to find with comparing the `bind` AND the `valueField` properties.
    let index = columns.findIndex(baseColumnDef => baseColumnDef.properties.bind === insertBeforeParts[0] &&
        baseColumnDef.properties.valueField === insertBeforeParts[1]);
    // If not found, we just use the `bind` property.
    if (index === -1) {
        index = columns.findIndex(baseColumnDef => convertDeepBindToPathNotNull(baseColumnDef.properties.bind) ===
            convertDeepBindToPathNotNull(insertBeforeParts[0]));
    }
    return index;
};
export const insertColumn = (columns, column, override = false, onlyInsertIfFound = false) => {
    if (column?.properties.insertBefore) {
        const insertBeforeIdx = findColumnIndex(columns, column.properties.insertBefore);
        if (insertBeforeIdx !== -1) {
            if (override) {
                // Move column
                const currentIdx = findColumnIndex(columns, convertDeepBindToPathNotNull(column.properties.bind));
                arrayMoveMutate(columns, currentIdx, insertBeforeIdx);
                return;
            }
            // Splice column
            columns.splice(insertBeforeIdx, 0, column);
            return;
        }
    }
    else if (column?.properties.insertAfter) {
        const insertAfterIdx = findColumnIndex(columns, column.properties.insertAfter);
        if (insertAfterIdx !== -1) {
            if (override) {
                // Move column
                const currentIdx = findColumnIndex(columns, convertDeepBindToPathNotNull(column.properties.bind));
                arrayMoveMutate(columns, currentIdx, insertAfterIdx);
                return;
            }
            // Splice column
            columns.splice(insertAfterIdx + 1, 0, column);
            return;
        }
    }
    // Append column: If none of the conditions were met above, we just push the column to the end.
    if (!override && !onlyInsertIfFound) {
        columns.push(column);
    }
};
/**
 * It merges the custom fields into the sidePanelLayout and returns a copy with the custom fields on it
 */
export const mergeIntoSidebarLayout = (sidePanelLayout, customFields) => {
    const cloneSidePanelLayout = { ...sidePanelLayout };
    customFields.forEach(customField => {
        const { insertBefore, insertAfter } = customField.properties;
        Object.keys(cloneSidePanelLayout).forEach(sectionKey => {
            const section = cloneSidePanelLayout[sectionKey];
            Object.keys(section.blocks).forEach(blockKey => {
                const block = section.blocks[blockKey];
                const blockIndex = block.fields.findIndex(field => (field?.constructor?.name === 'String' || field?.constructor?.name === 'Object') &&
                    (convertDeepBindToPathNotNull(field) === insertBefore ||
                        convertDeepBindToPathNotNull(field) === insertAfter));
                if (insertBefore && blockIndex !== -1) {
                    block.fields.splice(blockIndex, 0, convertDeepBindToPathNotNull(customField.properties.bind));
                }
                else if (insertAfter && blockIndex !== -1) {
                    block.fields.splice(blockIndex + 1, 0, convertDeepBindToPathNotNull(customField.properties.bind));
                }
            });
        });
    });
    return cloneSidePanelLayout;
};
/**
 * It merges the columns defined by the base page and the columns provided by the extension.
 *  */
export const mergeColumnPropertyDefinitions = (currentColumns = [], columnExtensions = []) => {
    const columns = [...currentColumns];
    columnExtensions.forEach(extensionColumnDef => {
        insertColumn(columns, extensionColumnDef, false);
    });
    return columns;
};
export const overrideColumnPropertyDefinitions = (currentColumns = [], columnOverrides = []) => {
    const columns = [...currentColumns];
    columnOverrides?.forEach(override => {
        const bind = override?.properties?.bind;
        const valueField = override?.properties?.valueField;
        const columnToOverride = bind &&
            (valueField
                ? columns.find(column => column?.properties?.bind === bind &&
                    column?.properties?.valueField === valueField)
                : columns.find(column => column?.properties?.bind === bind));
        if (columnToOverride) {
            columnToOverride.properties = { ...columnToOverride.properties, ...override.properties };
            if (override.properties.insertBefore || override.properties.insertAfter) {
                insertColumn(columns, columnToOverride, true);
            }
        }
    });
    return columns;
};
/** Merge the level definition of existing levels */
export const mergeLevelPropertyDefinition = (currentLevels = [], levelExtensions = []) => {
    const levels = [...currentLevels];
    levelExtensions.forEach((extensionLevelDefinition, i) => {
        // We merge the columns by respecting the `insertBefore` or `insertAfter` property, so reuse the logic developed for the table columns
        levels[i].columns = mergeColumnPropertyDefinitions(levels[i]?.columns || [], extensionLevelDefinition.columns);
        levels[i].columns = overrideColumnPropertyDefinitions(levels[i]?.columns || [], extensionLevelDefinition.columnOverrides);
        levels[i].dropdownActions = [
            ...(levels[i]?.dropdownActions || []),
            ...(extensionLevelDefinition.dropdownActions || []),
        ];
    });
    return levels;
};
export const overrideExtendedProperties = (nodeTypes, elementId, pageMetadata, properties, componentType) => {
    const decoratorProperties = { ...properties };
    pageMetadata.extensionOverrideThunks[elementId]?.forEach(overrideThunk => {
        const overrideThunkProps = overrideThunk();
        const orderedPropertyNames = [
            ...without(Object.keys(overrideThunkProps), 'columnOverrides'),
            'columnOverrides',
        ];
        orderedPropertyNames.forEach(propertyName => {
            const propValue = overrideThunkProps[propertyName];
            if (propertyName.indexOf('After') !== -1 && typeof propValue === 'function') {
                if (decoratorProperties[propertyName]) {
                    decoratorProperties[propertyName].push(propValue);
                }
                else {
                    decoratorProperties[propertyName] = [propValue];
                }
            }
            else if (propertyName === 'columns' || propertyName === 'fields') {
                // We handle columns in a different way, this property allows the addition of new columns, not replaces the existing columns.
                decoratorProperties[propertyName] = mergeColumnPropertyDefinitions(decoratorProperties[propertyName], propValue);
            }
            else if (propertyName === 'columnOverrides') {
                if (decoratorProperties?.columns && propValue) {
                    decoratorProperties.columns = overrideColumnPropertyDefinitions(decoratorProperties.columns, propValue);
                }
            }
            else if (propertyName === 'fieldOverrides') {
                if (decoratorProperties?.fields && propValue) {
                    decoratorProperties.fields = overrideColumnPropertyDefinitions(decoratorProperties.fields, propValue);
                }
            }
            else if (propertyName === 'levels') {
                // Nested Grid levels are also special, we need to loop through all levels and merge the columns.
                decoratorProperties[propertyName] = mergeLevelPropertyDefinition(decoratorProperties[propertyName], propValue);
            }
            else {
                decoratorProperties[propertyName] = propValue;
            }
        });
    });
    const decoratorPropertiesWithColumnsAndNode = decoratorProperties;
    if (decoratorPropertiesWithColumnsAndNode.node &&
        decoratorPropertiesWithColumnsAndNode.columns &&
        pageMetadata.customizations) {
        addCustomColumnToNestedFields(String(decoratorPropertiesWithColumnsAndNode.node), nodeTypes, decoratorPropertiesWithColumnsAndNode.columns, elementId, componentType, false, pageMetadata.customizations);
        // We check if the columns have nested columns, and if so, we add the custom fields to the nested columns as well.
        decoratorPropertiesWithColumnsAndNode.columns.forEach(column => {
            const columnProperties = column.properties;
            if (componentsWithLookup.includes(column.type) && columnProperties.columns) {
                addCustomColumnToNestedFields(String(columnProperties.node), nodeTypes, columnProperties.columns, columnProperties.bind, column.type, true, pageMetadata.customizations);
            }
        });
    }
    return decoratorProperties;
};
//# sourceMappingURL=abstract-decorator-utils.js.map