"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TreeInput = void 0;
const textbox_1 = __importDefault(require("carbon-react/esm/components/textbox"));
const tree_1 = require("../tree/tree");
const react_1 = __importDefault(require("react"));
const xtrem_shared_1 = require("@sage/xtrem-shared");
const popover_component_1 = __importDefault(require("carbon-react/esm/__internal__/popover/popover.component"));
const uid_1 = __importDefault(require("uid"));
const ROOT_TOKEN = '$root';
function TreeInput({ fetchItems, localize, node, onChange, isDisabled, canSelectObjects, value, 'data-testid': dataTestId = 'e-tree-input', }) {
    const [isOpened, setIsOpen] = react_1.default.useState(false);
    const [inputValue, setInputValue] = react_1.default.useState(value?.labelPath || '');
    const [selectedItems, setSelectedItems] = react_1.default.useState(value ? { [value.id]: value } : {});
    const elementRef = react_1.default.useRef(null);
    const listContainerRef = react_1.default.useRef(null);
    // Element id cannot start with a number
    const id = react_1.default.useRef(`x${(0, uid_1.default)()}`);
    const onClick = react_1.default.useCallback((e) => {
        let element = e.target;
        if (e.target === window.document.body) {
            return;
        }
        while (!element || element?.tagName !== 'HTML') {
            if (element === elementRef.current || element === listContainerRef.current) {
                return;
            }
            element = element?.parentElement || null;
        }
        setIsOpen(false);
        window.removeEventListener('click', onClick);
    }, [elementRef]);
    const closeDropDown = react_1.default.useCallback(() => {
        setIsOpen(false);
        window.removeEventListener('click', onClick);
    }, [onClick]);
    const openDropDown = react_1.default.useCallback(() => {
        if (!isDisabled) {
            setIsOpen(true);
            window.addEventListener('click', onClick);
        }
    }, [isDisabled, onClick]);
    const onFocus = react_1.default.useCallback(() => {
        const inputElement = elementRef.current?.querySelector('input');
        if (!isDisabled && inputElement && inputElement.value.length > 0) {
            inputElement.setSelectionRange(0, inputElement.value.length);
        }
    }, [isDisabled]);
    const onCheckedItemsUpdated = react_1.default.useCallback((e) => {
        const keys = Object.keys(e);
        if (keys.length > 0) {
            const key = keys[0];
            const item = e[key];
            setInputValue(item.labelPath);
            closeDropDown();
            onChange(item);
        }
        else {
            onChange(null);
        }
        setSelectedItems(e);
        elementRef.current?.querySelector('input')?.focus();
        setTimeout(() => {
            setIsOpen(false);
        }, 0);
    }, [closeDropDown, onChange, elementRef]);
    const onKeyDown = react_1.default.useCallback((ev) => {
        if (ev.key === 'ArrowDown') {
            openDropDown();
            setTimeout(() => {
                listContainerRef.current?.querySelector('button')?.focus();
            }, 0);
        }
        if (ev.key === 'Escape') {
            closeDropDown();
            ev.preventDefault();
            ev.stopPropagation();
        }
        if (ev.key === 'Backspace' || ev.key === 'Delete') {
            setSelectedItems({});
            setInputValue('');
        }
        if (ev.key === 'Tab') {
            setTimeout(closeDropDown, 10);
        }
        if (ev.key.startsWith('Key') || ev.key.startsWith('Digit') || ev.key.startsWith('Numpad')) {
            // Prevent the user from typing in the input field
            ev.preventDefault();
            ev.stopPropagation();
        }
    }, [closeDropDown, openDropDown]);
    const onTreeSelfClose = react_1.default.useCallback(() => {
        elementRef.current?.querySelector('input')?.focus();
        setTimeout(() => {
            setIsOpen(false);
        }, 0);
    }, [elementRef]);
    const internalFetchItems = react_1.default.useCallback(async (parent) => {
        const fetchedItems = await fetchItems(parent.data);
        return fetchedItems
            .filter(data => !xtrem_shared_1.textStreams.includes(data.type) &&
            data.kind !== 'LIST' &&
            data.type !== xtrem_shared_1.GraphQLTypes.InputStream)
            .map((data) => {
            return {
                data: { ...data, node: data.type },
                id: `${parent.id}.${data.name}`.replace(/^\$root\./, ''),
                key: `${parent.key}.${data.name}`.replace(/^\$root\./, ''),
                label: data.label || data.name,
                labelKey: data.label || data.name,
                labelPath: `${parent.labelPath} > ${data.label}`.replace(/^\s>\s/, ''),
                canBeExpanded: data.kind === 'OBJECT',
                canBeSelected: canSelectObjects
                    ? data.kind === 'OBJECT' || data.kind === 'SCALAR'
                    : data.kind === 'SCALAR',
            };
        });
    }, [canSelectObjects, fetchItems]);
    return (react_1.default.createElement("div", { className: "e-tree-input", ref: elementRef },
        react_1.default.createElement(textbox_1.default, { onFocus: onFocus, onKeyDown: onKeyDown, inputIcon: "caret_down", value: inputValue, disabled: isDisabled, size: "small", "data-testid": dataTestId, "aria-expanded": isOpened, "aria-controls": id.current, onClick: openDropDown }),
        react_1.default.createElement(popover_component_1.default, { placement: "bottom", disablePortal: false, reference: elementRef, isOpen: isOpened, disableBackgroundUI: true, animationFrame: true },
            react_1.default.createElement("div", { ref: listContainerRef, id: id.current, className: "e-tree-input-dropdown", "data-testid": "e-tree-input-dropdown", style: {
                    top: (elementRef.current?.offsetHeight || 0) + (elementRef.current?.offsetTop || 0),
                    left: elementRef.current?.offsetLeft,
                    width: elementRef.current?.offsetWidth,
                    zIndex: 2,
                } },
                react_1.default.createElement(tree_1.Tree, { checkedItems: selectedItems, fetchItems: internalFetchItems, isFolderIconHidden: true, level: 0, localize: localize, onCheckedItemsUpdated: onCheckedItemsUpdated, onClose: onTreeSelfClose, selectionMode: "button", element: {
                        id: ROOT_TOKEN,
                        key: ROOT_TOKEN,
                        label: 'Root',
                        labelKey: 'Root',
                        labelPath: '',
                        canBeExpanded: true,
                        canBeSelected: false,
                        data: { name: 'Root', node, canSort: false, canFilter: false },
                    } })))));
}
exports.TreeInput = TreeInput;
//# sourceMappingURL=tree-input.js.map