"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.safeJsonParse = exports.dotNotate = exports.format = exports.handleBars = void 0;
const handlebars = require("handlebars");
// const handlebarsIntl = require('handlebars-intl');
// handlebarsIntl.registerWith(handlebars);
function formatGraphQInputData(data, options) {
    if (!data || typeof data !== 'object')
        return JSON.stringify(data);
    if (Array.isArray(data))
        return `[${data.map(value => formatGraphQInputData(value))}]`;
    const inner = Object.entries(data)
        .map(([key, value]) => `${key}:${formatGraphQInputData(value)}`)
        .join();
    return options?.withoutBraces ? inner : `{${inner}}`;
}
/**
 * `inputParameters` handlebars helper: formats `parameters` as a GraphQl input object.
 */
handlebars.registerHelper('inputParameters', (data) => {
    return new handlebars.SafeString(formatGraphQInputData(data.data.root.properties));
});
/**
 * `inputData` handlebars helper: formats `parameters` as a list of GraphQl input parameters
 * Same as `inputParameters` but without braces around the result.
 */
handlebars.registerHelper('inputData', (data) => {
    return new handlebars.SafeString(formatGraphQInputData(data.data.root.properties, { withoutBraces: true }));
});
const handleBars = (template, data) => {
    return {
        result: handlebars.compile(template, { knownHelpers: { helperMissing: true } })(data),
        variables: getHandleBarsVariables(template),
    };
};
exports.handleBars = handleBars;
/**
 * Gets all the variables used in a Handlebars template
 *
 * @param {string} template Handlebars template
 * @returns {string[]} all variables needed by the given template
 */
const getHandleBarsVariables = (template) => {
    const ast = handlebars.parse(template);
    return Object.values(ast.body)
        .filter(curr => curr.type === 'MustacheStatement')
        .map(curr => curr.path.original);
};
const format = (literal, locale, data = {}, noEscape = false) => {
    // Resolve arguments if they are provided
    if (Object.keys(data).length > 0) {
        const delegate = handlebars.compile(literal, { noEscape });
        const runtimeOptions = { data: { locale } };
        return delegate(data, runtimeOptions);
    }
    return literal;
};
exports.format = format;
const dotNotate = (input, prefix = '', target = {}) => {
    Object.keys(input).forEach(key => {
        const val = input[key];
        if (val !== null && typeof val === 'object') {
            (0, exports.dotNotate)(input[key], `${prefix + key}.`, target);
        }
        target[prefix + key] = val;
        return val;
    });
    return target;
};
exports.dotNotate = dotNotate;
const safeJsonParse = (input) => {
    try {
        return JSON.parse(input);
    }
    catch (e) {
        return undefined;
    }
};
exports.safeJsonParse = safeJsonParse;
//# sourceMappingURL=format-string.js.map