/// <reference types="node" />
/// <reference types="node" />
/// <reference types="node" />
/** @module @sage/xtrem-postgres */
import { AnyValue, AsyncReader, AsyncResponse } from '@sage/xtrem-async-helper';
import { LoggerInterface } from '@sage/xtrem-shared';
import { EventEmitter } from 'node:events';
import * as postgres from 'pg';
import { SqlRecorder } from './sql-recorder';
import { Connection, PoolConfig, PoolLoggerInterface, PostgresPoolReaderOptions, PostgresSqlExecuteOptions } from './types';
import Pool = require('pg-pool');
import copyStreams = require('pg-copy-streams');
export declare const copyTo: typeof copyStreams.to;
export declare const copyFrom: typeof copyStreams.from;
export type CommonPool = Pick<Pool<postgres.Client>, 'connect' | 'on' | 'end'>;
export interface DatabaseVersion {
    text: string;
    major: number;
    minor: number;
}
export interface DatabaseSettings {
    max_connections: number;
    reserved_connections: number;
    superuser_reserved_connections: number;
    shared_buffers: string;
}
export interface DatabaseInfo {
    version: DatabaseVersion;
    settings: DatabaseSettings;
}
export interface ConnectionPoolSetupOptions {
    logger: LoggerInterface;
    metricsEmitter?: EventEmitter;
}
export type ConnectionPoolType = 'server' | 'sys' | 'user';
export declare class ConnectionPool {
    #private;
    readonly type: ConnectionPoolType;
    private readonly config;
    private _user;
    private _pool;
    sqlRecorder?: SqlRecorder;
    trace: (...args: any[]) => void;
    constructor(type: ConnectionPoolType, config: PoolConfig);
    get user(): string;
    static logger: PoolLoggerInterface;
    private static metricsEmitter;
    private get stats();
    static setup(options: ConnectionPoolSetupOptions): void;
    private incrementRetryCounter;
    private updateMetrics;
    /**
     * Return the max count of connection the pool can open concurrently
     */
    get poolSize(): number;
    /**
     * Returns the configuration used by the postgres pool
     */
    private get runtimeConfig();
    private queryDatabaseInfo;
    getUsablePoolSize(): Promise<number>;
    private get pool();
    /**
     * Function to execute passed callback function passing the current allocated connection to the callback and then releasing the connection
     * @param body
     */
    withConnection<T extends AnyValue | void>(body: (cnx: Connection) => AsyncResponse<T>, connection?: postgres.PoolClient): Promise<T>;
    /**
     * Returns a buffer reader for the supplied BLOB
     * @param value
     */
    static readBlob(val: any): Promise<Buffer | null>;
    /**
     *  Returns a buffer reader for the supplied CLOB
     * @param value
     */
    static readClob(val: any): Promise<string | null>;
    /**
     * Returns the relevant buffer reader for the supplied BLOB
     * @param val
     */
    static blobReader(val: any): AsyncReader<Buffer>;
    /**
     *  Returns the relevant string reader for the supplied CLOB
     * @param val
     */
    static clobReader(val: any): AsyncReader<string>;
    onNotice: (notice: any) => void;
    /**
     * Gets the current database connection, if one does not exists then a connection is created
     */
    private _allocConnection;
    /**
     * Allocates a postgres connection.
     *
     * The allocation is throttled, to ensure that:
     * - we don't exhaust the pool
     * - concurrent requests are treated fairly.
     *
     * @param originId the originId of the request
     * @returns a postgres connection promise
     */
    allocConnection(originId: string): Promise<Connection>;
    /**
     * Releases a postgres connection.
     *
     * This method MUST be used to release connections allocated by the allocConnection method.
     *
     * @param connection the connection
     * @returns void
     */
    releaseConnection(connection: Connection): void;
    private getProfiler;
    /**
     * Function that executes SQL command on the provided connection and with the provided arguments and options.
     * @param cnx Database connection
     * @param sql SQL command
     * @param args SQL arguments
     * @param opts SQL options
     */
    execute<T extends AnyValue | void>(cnx: Connection, sql: string, args?: any[], opts?: PostgresSqlExecuteOptions): Promise<T>;
    private static convertArgs;
    /**
     * Creates a reader for the provided connection and SQL statement
     * @param cnx
     * @param sql
     * @param args
     * @param opts
     */
    createReader<T extends AnyValue>(cnx: Connection, sql: string, args: any[], opts?: PostgresPoolReaderOptions): AsyncReader<T>;
    /**
     * Release all database connections
     */
    release(): Promise<void>;
}
//# sourceMappingURL=pool.d.ts.map