import { WorkOrder } from '@sage/x3-manufacturing-api';
import { StandardOperations, WorkOrderProductLine } from '@sage/x3-manufacturing-api-partial';
import { WorkCenter } from '@sage/x3-manufacturing-data-api';
import { decimal, extractEdges } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { GetUOMNumberOfDecimalPlaces } from '../client-functions/get-unit-decimal-places';
import { validateWithDetails } from '../client-functions/validation';
import { OperationLine, TimeTrackingLine, TimeTrackingStorageObject } from './utils/types';

@ui.decorators.page<MobileTimeTrackingDetails>({
    title: 'Time tracking details',
    isTransient: false,
    isTitleHidden: true,
    module: 'x3-manufacturing',
    mode: 'default',
    headerCard() {
        return {
            title: this.workOrderNumber,
            line2: this.routingCode,
            titleRight: this.operationNumber,
            line2Right: this.operationDescription,
        };
    },
    businessActions() {
        return [this.addQuantity, this.nextPage];
    },
    async onLoad() {
        await this.initPage();
    },
})
export class MobileTimeTrackingDetails extends ui.Page {
    @ui.decorators.section<MobileTimeTrackingDetails>({
        isTitleHidden: true,
    })
    section: ui.containers.Section;

    @ui.decorators.block<MobileTimeTrackingDetails>({
        parent() {
            return this.section;
        },
        isTitleHidden: true,
    })
    block: ui.containers.Block;

    @ui.decorators.referenceField<MobileTimeTrackingDetails, WorkCenter>({
        parent() {
            return this.block;
        },
        title: 'Actual work center',
        valueField: 'code',
        node: '@sage/x3-manufacturing-data/WorkCenter',
        isTransientInput: false,
        isTransient: false,
        placeholder: 'Scan or select...',
        isAutoSelectEnabled: true,
        isMandatory: true,
        filter() {
            return {
                manufacturingSite: { code: this.getStockSite() },
                type: { _ne: 3 },
            };
        },
        onChange() {
            this.remainingQuantity.focus();
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'description',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkCenterGroup',
                bind: 'workCenterGroup',
                valueField: 'code',
            }),
            ui.nestedFields.text({
                bind: 'type',
                isReadOnly: true,
            }),
        ],
    })
    expectedWorkCenter: ui.fields.Reference;

    @ui.decorators.separatorField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
    })
    blank: ui.fields.Separator;

    @ui.decorators.textField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Time unit',
        isDisabled: true,
    })
    timeUnit: ui.fields.Text;

    @ui.decorators.textField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Unit',
        isDisabled: true,
    })
    stockUnit: ui.fields.Text;

    @ui.decorators.numericField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Quantity',
        isMandatory: true,
        validation: /^\d*[0-9](|.\d*[0-9]|,\d*[0-9])?$/,
        onChange() {
            if (this.remainingQuantity.value === undefined || this.remainingQuantity.value === null) {
                this.remainingQuantity.value = 0;
                validateWithDetails(this);
            }

            const { operationMethod }: TimeTrackingStorageObject = this.getStorageObject();
            if (operationMethod === 0) {
                this.setCloseField();
                if (this.remainingQuantity.value > this.getQtyAvailableToTrack()) {
                    this.$.showToast(
                        ui.localize(
                            '@sage/x3-manufacturing/pages__mobile_time_tracking_details___quantity__larger__expected__warning',
                            'The quantity is larger than expected.',
                        ),
                        { type: 'warning', timeout: 5000 },
                    );
                }
            }
        },
    })
    remainingQuantity: ui.fields.Numeric;

    @ui.decorators.numericField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Rejected qty.',
        validation: /^\d*[0-9](|.\d*[0-9]|,\d*[0-9])?$/,
        onChange() {
            // if user blanks out value set to zero and re-run screen validation
            if (this.rejectedQuantity.value === undefined || this.rejectedQuantity.value === null) {
                this.rejectedQuantity.value = 0;
                validateWithDetails(this);
            }
            if (this.remainingQuantity.value === undefined || this.remainingQuantity.value === null) {
                this.remainingQuantity.value = 0;
            }
            if (this.rejectedQuantity.value > this.remainingQuantity.value) {
                this.$.showToast(
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_time_tracking_details___rejected_qty_error',
                        'Rejected quantity greater than the quantity value.',
                    ),
                    { type: 'error', timeout: 5000 },
                );
            }
        },
    })
    rejectedQuantity: ui.fields.Numeric;

    @ui.decorators.numericField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Actual setup time',
        isHidden: true,
        isDisabled: true,
        scale: 4,
    })
    setupTime: ui.fields.Numeric;

    @ui.decorators.numericField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Actual run time',
        isHidden: true,
        isDisabled: true,
        scale: 4,
    })
    runTime: ui.fields.Numeric;

    @ui.decorators.numericField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Setup time hours',
        scale: 4,
        validation: /^\d*[0-9](|.\d*[0-9]|,\d*[0-9])?$/,
        onChange() {
            // if user blanks out value set to zero and re-run screen validation
            if (this.enteredSetupTimeHours.value === undefined || this.enteredSetupTimeHours.value === null) {
                this.enteredSetupTimeHours.value = 0;
                validateWithDetails(this);
            }
        },
    })
    enteredSetupTimeHours: ui.fields.Numeric;

    @ui.decorators.numericField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Setup time minutes',
        scale: 0,
        validation: /^\d*[0-9](|.\d*[0-9]|,\d*[0-9])?$/,
        onChange() {
            if (this.enteredSetupTimeMinutes.value === undefined || this.enteredSetupTimeMinutes.value === null) {
                this.enteredSetupTimeMinutes.value = 0;
                validateWithDetails(this);
            }

            if (this.enteredSetupTimeMinutes.value >= 60) {
                this.$.showToast(
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_time_tracking_details___minutes__entered__warning',
                        'Minutes are equal to or greater than 60.',
                    ),
                    { type: 'warning', timeout: 5000 },
                );
            }
        },
    })
    enteredSetupTimeMinutes: ui.fields.Numeric;

    @ui.decorators.numericField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Run time hours',
        scale: 4,
        validation: /^\d*[0-9](|.\d*[0-9]|,\d*[0-9])?$/,
        onChange() {
            if (this.enteredRunTimeHours.value === undefined || this.enteredRunTimeHours.value === null) {
                this.enteredRunTimeHours.value = 0;
                validateWithDetails(this);
            }
        },
    })
    enteredRunTimeHours: ui.fields.Numeric;

    @ui.decorators.numericField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Run time minutes',
        isMandatory: false,
        validation: /^\d*[0-9](|.\d*[0-9]|,\d*[0-9])?$/,
        scale: 0,
        onChange() {
            if (this.enteredRunTimeMinutes.value === undefined || this.enteredRunTimeMinutes.value === null) {
                this.enteredRunTimeMinutes.value = 0;
                validateWithDetails(this);
            }

            if (this.enteredRunTimeMinutes.value >= 60) {
                this.$.showToast(
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_time_tracking_details___minutes__entered__warning',
                        'Minutes are equal to or greater than 60.',
                    ),
                    { type: 'warning', timeout: 5000 },
                );
            }
        },
    })
    enteredRunTimeMinutes: ui.fields.Numeric;

    @ui.decorators.checkboxField<MobileTimeTrackingDetails>({
        parent() {
            return this.block;
        },
        title: 'Close',
        onChange() {
            if (this.close.value === null || this.close.value === undefined) {
                this.setCloseField();
            }
        },
    })
    close: ui.fields.Checkbox;

    @ui.decorators.pageAction<MobileTimeTrackingDetails>({
        title: 'Add quantity',
        buttonType: 'secondary',
        shortcut: ['f2'],
        async onClick() {
            if (!(await validateWithDetails(this))) return;
            await this.setActualRunTime();
            await this.setActualSetupTime();
            if ((await this.isTimeAndQuantityValid()) === false) {
                return;
            } else {
                await this.buttonsLogic(true);
                const { operationMethod }: TimeTrackingStorageObject = this.getStorageObject();
                //const { operationMethod, operationLine }: TimeTrackingStorageObject = this.getStorageObject();
                if (operationMethod === 0) {
                    await this.setRemainingQuantity();
                }
                await this.SetAllTimeFieldsToZero();
                await this.setRejectedQuantityToZero();
                this.expectedWorkCenter.focus();
            }
        },
    })
    addQuantity: ui.PageAction;

    @ui.decorators.pageAction<MobileTimeTrackingDetails>({
        title: 'Next',
        buttonType: 'primary',
        shortcut: ['f3'],
        async onClick() {
            if (!(await validateWithDetails(this))) return;
            await this.setActualRunTime();
            await this.setActualSetupTime();
            if ((await this.isTimeAndQuantityValid()) === false) {
                return;
            } else {
                await this.buttonsLogic(false);
                this.$.router.goTo('@sage/x3-manufacturing/MobileTimeTracking', {
                    gotLines: 'true',
                });
            }
        },
    })
    nextPage: ui.PageAction;

    /**  Header Fields */
    @ui.decorators.textField<MobileTimeTrackingDetails>({
        isDisabled: true,
        isTransient: true,
    })
    workOrderNumber: ui.fields.Text;

    @ui.decorators.textField<MobileTimeTrackingDetails>({
        isDisabled: true,
        isHidden: true,
    })
    releasedRouting: ui.fields.Text;

    @ui.decorators.textField<MobileTimeTrackingDetails>({
        isDisabled: true,
    })
    routingCode: ui.fields.Text;

    @ui.decorators.numericField<MobileTimeTrackingDetails>({
        isDisabled: true,
    })
    operationNumber: ui.fields.Numeric;

    @ui.decorators.textField<MobileTimeTrackingDetails>({
        isDisabled: true,
        isTransient: true,
    })
    operationDescription: ui.fields.Text;

    @ui.decorators.textField<MobileTimeTrackingDetails>({
        isDisabled: true,
        isHidden: true,
    })
    bom: ui.fields.Text;

    @ui.decorators.textField<MobileTimeTrackingDetails>({
        isDisabled: true,
    })
    bomCode: ui.fields.Text;
    // - For Information only the SelectedOperationMethod are
    //   RouterOperation is 0
    //   StandardOperationWithData is 1
    //   StandardOperationNoData is 2
    //   UserDefinedOperation is 3

    async initPage() {
        //const stockSite: string = this.getStockSite();
        const {
            userDefinedOperationNumber,
            userDefinedOperationDescription,
            operationMethod,
            standardOperations,
            operationLine,
            workOrderNumber,
            releaseTimeUnit,
            releaseUOM,
        }: TimeTrackingStorageObject = this.getStorageObject();
        this.setHeaderFields(
            userDefinedOperationNumber,
            operationLine,
            workOrderNumber,
            operationMethod,
            userDefinedOperationDescription,
        );
        this.setDefaultValues(operationLine, standardOperations, operationMethod, releaseUOM, releaseTimeUnit);
        this.workOrderNumber.value = workOrderNumber.number;
        this.releasedRouting.value = workOrderNumber.releasedRouting.routing.code;
    }

    // Set header field values
    setHeaderFields(
        userDefinedOperationNumber: number | null,
        operationLine: Partial<OperationLine> | null,
        workOrder: WorkOrder,
        operationMethod: number,
        userDefinedOperationDescription: string | null | undefined,
    ) {
        this.workOrderNumber.value = workOrder.number;
        this.routingCode.value = workOrder.routingCode.code.toString();

        if (operationMethod !== 0) {
            this.operationNumber.value = userDefinedOperationNumber;
            this.operationDescription.value = userDefinedOperationDescription;
        }

        if (operationMethod === 0) {
            this.operationNumber.value = operationLine?.operationNumber;
            this.operationDescription.value = operationLine?.operationDescription;
        }
    }

    /** function Set initial page field values */
    async setDefaultValues(
        operationLine: Partial<OperationLine>,
        standardOperations: Partial<StandardOperations>,
        operationMethod: number,
        releaseUOM: string,
        releaseTimeUnit: string,
    ) {
        // item independant of operation method - generic
        this.rejectedQuantity.value = 0;
        this.enteredSetupTimeMinutes.value = 0;
        this.enteredSetupTimeHours.value = 0;
        this.enteredRunTimeMinutes.value = 0;
        this.enteredRunTimeHours.value = 0;

        // 0 = Routing Operation Method
        if (operationMethod !== 0) {
            // 1 = Standard Operation with data Method
            if (operationMethod === 1) {
                try {
                    this.stockUnit.value = standardOperations.operationUom.code;
                } catch (error) {
                    this.stockUnit.value = releaseUOM;
                }
                await this.setWorkCenter(standardOperations.mainWorkCenter.code);
            }

            if (operationMethod === 2) {
                this.stockUnit.value = releaseUOM;
                this.timeUnit.value = releaseTimeUnit;
                this.remainingQuantity.value = 1;
                this.setScaleForNumericFields(this.stockUnit.value);
            }

            this.close.value = true;
            this.close.isDisabled = true;
            this.setupTime.value = Number(standardOperations.setupTime);
            this.runTime.value = Number(standardOperations.runTime);
            this.remainingQuantity.value = 1;
            this.timeUnit.value = releaseTimeUnit;
        }

        // 0 = Routing Operation Method
        if (operationMethod === 0) {
            const bomValues = await this.getBomCode();
            this.bomCode.value = bomValues.bomCode.code.toString();
            this.bom.value = bomValues.product.code;
            this.stockUnit.value = operationLine.operationUnit.code;
            this.runTime.value = operationLine.runTime;
            this.setupTime.value = operationLine.setupTime;
            this.timeUnit.value = operationLine.timeUnit;
            await this.setRemainingQuantity();
            await this.setWorkCenter(operationLine.expectedWorkCenter.code);
            this.setCloseField();
        }

        this.setScaleForNumericFields(this.stockUnit.value);
    }

    /** Set the scale for numeric field based on their unit of measure */
    async setScaleForNumericFields(stockUnit: string) {
        const numberOfDecimalPlaces = await GetUOMNumberOfDecimalPlaces(this, stockUnit);
        this.remainingQuantity.scale = numberOfDecimalPlaces;
        this.rejectedQuantity.scale = numberOfDecimalPlaces;
    }

    /** set the remaining quantity value. Expected qty minus the completed qty
     *  if completed exceeds expected then set qty to zero */
    async setRemainingQuantity() {
        const lines: TimeTrackingLine[] = this.getTrackingLines();
        const { operationLine }: TimeTrackingStorageObject = this.getStorageObject();
        const qtyTrackedOnDeviceOnly = this.calculateLineQuantity(lines, operationLine);
        const remainingQtyInX3Database =
            (operationLine?.expectedQuantity ?? 0) - (operationLine?.completedQuantity ?? 0);
        this.remainingQuantity.value = remainingQtyInX3Database - qtyTrackedOnDeviceOnly;

        if (this.remainingQuantity.value < 0) {
            this.remainingQuantity.value = 0;
        }
    }

    getStockSite() {
        try {
            return this.$.storage.get('mobile-selected-stock-site') as string;
        } catch (err) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_completed_quantity___stock_site_error',
                    'The stock site is not defined',
                ),
                { type: 'error', timeout: 5000 },
            );
            return '';
        }
    }

    getStorageObject(): TimeTrackingStorageObject | undefined {
        try {
            const storageObject = this.$.storage.get('time_tracking') as any;
            return storageObject;
        } catch (err) {
            this.showStorageError();
            return undefined;
        }
    }

    getTrackingLines(): TimeTrackingLine[] {
        try {
            const { timeTracking } = this.getStorageObject() || {};
            return timeTracking.workOrderOperationTrackingLines;
        } catch (err) {
            this.showStorageError();
            return;
        }
    }

    showStorageError(): void {
        this.$.showToast(
            ui.localize(
                '@sage/x3-manufacturing/pages__mobile_completed_quantity_details___empty_storage_object_error',
                'Storage object is empty',
            ),
            { type: 'error', timeout: 5000 },
        );
        this.$.router.goTo('@sage/x3-manufacturing/MobileCompletedQuantity');
    }

    initTrackingLines(storageObject: any): TimeTrackingLine[] {
        const { employeeId, transaction, trackingDate, timeTracking, standardOperations } = storageObject;
        const stockSite = this.getStockSite();

        const trackingLine = timeTracking.workOrderOperationTrackingLines;
        let employee: number = 0;
        let lineCounter: number = 0;
        let standardOperation: number = 0;
        let close: number = 1;
        if (this.close.value === true) {
            close = 2;
        }

        const YYYYMMDDdate = (date: string): string => {
            const year = date.slice(0, 4);
            const month = date.slice(5, 7);
            const day = date.slice(8, 10);
            return `${year}-${month}-${day}`;
        };

        trackingLine.length !== 0 && (lineCounter = trackingLine.length);

        if (employeeId !== null) {
            employee = employeeId.teamId;
        }

        if (standardOperations !== null) {
            standardOperation = standardOperations.code;
        }

        trackingLine[lineCounter] = {
            ...trackingLine[lineCounter],
            productionSite: stockSite,
            workOrderNumber: this.workOrderNumber.value,
            routing: this.releasedRouting.value,
            routingCode: this.routingCode.value,
            operationNumber: this.operationNumber.value,
            description: this.operationDescription.value,
            postingDate: YYYYMMDDdate(trackingDate),
            transaction: transaction,
            employeeId: employee,
            operationUnit: this.stockUnit.value,
            operationConversion: 1,
            actualWorkCenter: this.expectedWorkCenter.value.code,
            totalCompletedQuantity: this.remainingQuantity.value,
            actualRejectedQuantity: this.rejectedQuantity.value,
            actualSetupTime: this.setupTime.value,
            actualRunTime: this.runTime.value,
            balance: close,
            rejections: [],
            bom: this.bom.value,
            bomCode: this.bomCode.value,
            operationSplit: 0,
            alternativeIndex: 0,
            project: '',
            standardOperation: standardOperation,
            timeUnit: this.timeUnit.value,
        } as TimeTrackingLine[];

        return trackingLine;
    }

    saveTimeTracking(storageObject: TimeTrackingStorageObject) {
        const { timeTracking } = storageObject;
        timeTracking.workOrderOperationTrackingLines = this.initTrackingLines(storageObject);
        this.$.storage.set('time_tracking', storageObject as any);
    }

    async buttonsLogic(displayMsg: boolean) {
        const storageObject = this.getStorageObject();
        this.saveTimeTracking(storageObject);
        if (displayMsg === true) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_completed_quantity_details___quantity_added',
                    'Quantity added.',
                ),
                { type: 'success', timeout: 3000 },
            );
        }
        this.$.setPageClean();
    }

    /** Function coverts the setup time entered by the user to operation time unit
     *  updates runtime page field with new value return: true | false was calculation successfully
     */
    async setActualRunTime(): Promise<boolean> {
        if (this.enteredRunTimeHours.value + this.enteredRunTimeMinutes.value === 0) {
            this.runTime.value = 0;
            return true;
        }

        try {
            if (this.timeUnit.value === 'hours') {
                const time = Math.floor(this.enteredRunTimeHours.value * 60);
                this.runTime.value = (this.enteredRunTimeMinutes.value + time) / 60;
            }
            if (this.timeUnit.value === 'minutes') {
                const time = Math.floor(this.enteredRunTimeHours.value * 60);
                this.runTime.value = this.enteredRunTimeMinutes.value + time;
            }
            return true;
        } catch (error) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking_details___update__runtime__error',
                    'The actual run time calculation failed.',
                ),
                { type: 'error', timeout: 5000 },
            );
            return false;
        }
    }

    /** Function coverts the setup time entered by the user to operation time unit
     *  updates setuptime page field with new value return: true | fales was calculation succefully
     */
    async setActualSetupTime(): Promise<boolean> {
        if (this.enteredSetupTimeHours.value + this.enteredSetupTimeMinutes.value === 0) {
            this.setupTime.value = 0;
            return true;
        }
        try {
            if (this.timeUnit.value === 'hours') {
                const time = Math.floor(this.enteredSetupTimeHours.value * 60);
                this.setupTime.value = (this.enteredSetupTimeMinutes.value + time) / 60;
            }
            if (this.timeUnit.value === 'minutes') {
                const time = Math.floor(this.enteredSetupTimeHours.value * 60);
                this.setupTime.value = this.enteredSetupTimeMinutes.value + time;
            }
            return true;
        } catch (error) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking_details___update__setup__time__error',
                    'The actual setup time calculation failed.',
                ),
                { type: 'error', timeout: 5000 },
            );
            return false;
        }
    }

    async setWorkCenter(workCenter: string) {
        try {
            this.expectedWorkCenter.value = extractEdges<WorkCenter>(
                await this.$.graph
                    .node('@sage/x3-manufacturing-data/WorkCenter')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                code: true,
                            },
                            {
                                filter: {
                                    code: workCenter,
                                },
                            },
                        ),
                    )
                    .execute(),
            );
            if (this.expectedWorkCenter.value) {
                this.expectedWorkCenter.value.code = workCenter;
            } else {
                this.expectedWorkCenter.value.code = '';
            }
        } catch (error) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_completed_quantity_details___loading_work_centre_error',
                    'Error loading work center.',
                ),
                String(error),
            );
        }
    }

    /** function verify if the time and quantity values enter are permitted - True | False */
    async isTimeAndQuantityValid(): Promise<boolean> {
        let result: boolean = true;
        const totaltime = this.runTime.value + this.setupTime.value;
        const totalQuantity = this.remainingQuantity.value - this.rejectedQuantity.value;

        if (totalQuantity < 0) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking_details___rejected_qty_error',
                    'Rejected quantity greater than the quantity value.',
                ),
                { type: 'error', timeout: 5000 },
            );
            result = false;
        }

        if (this.remainingQuantity.value + totaltime === 0) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking_details___invalid__quantity__time___values__error',
                    'Check your entry for quantity and time.',
                ),
                { type: 'error', timeout: 5000 },
            );
            result = false;
        }

        if (totaltime === 0 && totalQuantity > 0) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking_details___no__time__entered__warning',
                    'No time entered.',
                ),
                { type: 'warning', timeout: 5000 },
            );
            result = true;
        }

        return result;
    }

    /** function to set the Close field - yes/no based on the quantity being tracked */
    async setCloseField() {
        const lines: TimeTrackingLine[] = this.getTrackingLines();
        const { operationLine }: TimeTrackingStorageObject = this.getStorageObject();
        let automaticClosingPercentage: decimal = 100;
        let completedQuantityPercentage: decimal = 0;

        // Total quantity = quantity of tracked lines + quantity to track + completed quantity
        const totalQuantity =
            this.calculateLineQuantity(lines, operationLine) +
            this.remainingQuantity.value +
            operationLine.completedQuantity;

        if (totalQuantity > 0 && operationLine.expectedQuantity > 0) {
            completedQuantityPercentage = (totalQuantity / operationLine.expectedQuantity) * 100;
        }
        automaticClosingPercentage = await this.getAutomaticClosingPercent(this.expectedWorkCenter.value.code);

        if (automaticClosingPercentage != 0) {
            if (completedQuantityPercentage >= automaticClosingPercentage) {
                this.close.value = true;
            } else {
                this.close.value = false;
            }
        } else {
            this.close.value = false;
        }
    }

    /** for the current work order and operation set the total tracked quantity
     *  return = qty already tracked in X3 + qty tracked on device */
    getQtyAvailableToTrack(): number {
        const lines: TimeTrackingLine[] = this.getTrackingLines();
        const { operationLine }: TimeTrackingStorageObject = this.getStorageObject();
        const qtyTrackedOnDeviceOnly = this.calculateLineQuantity(lines, operationLine);
        const remainingQtyInX3Database = operationLine.expectedQuantity - operationLine.completedQuantity;
        const remainingQty: number = remainingQtyInX3Database - qtyTrackedOnDeviceOnly;
        if (remainingQty) {
            return remainingQty;
        }
        return 0;
    }

    /**
     * Calculate the total quantity - lines tracked for a given work order and operation
     * @param lines
     * @param operationLine
     * @returns
     */
    calculateLineQuantity(lines: TimeTrackingLine[], operationLine: Partial<OperationLine>): number {
        const { workOrder, operationNumber }: Partial<OperationLine> = operationLine;
        const currentOperation: TimeTrackingLine[] = lines.filter(
            (line: TimeTrackingLine) =>
                line.workOrderNumber === workOrder?.number && line.operationNumber === operationNumber,
        );

        const totalLineQuantity: number = currentOperation.reduce((sum: number, operation: any) => {
            return sum + operation.totalCompletedQuantity;
        }, 0);
        return totalLineQuantity;
    }

    /**
     * Automatic closing percentage for expected work center & site
     * @param workCenter
     * @returns
     */
    async getAutomaticClosingPercent(workCenter: string): Promise<decimal> {
        try {
            const result = extractEdges<WorkCenter>(
                await this.$.graph
                    .node('@sage/x3-manufacturing-data/WorkCenter')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                automaticClosingPercentage: true,
                            },
                            {
                                filter: {
                                    code: workCenter,
                                    manufacturingSite: { code: this.getStockSite() },
                                },
                            },
                        ),
                    )
                    .execute(),
            );
            return parseFloat(result[0].automaticClosingPercentage);
        } catch (err) {
            return 0;
        }
    }

    /**
     * get BOM code required for operation tracking/creation of lines
     * @returns
     */
    async getBomCode(): Promise<any> {
        try {
            const result = extractEdges<WorkOrderProductLine>(
                await this.$.graph
                    .node('@sage/x3-manufacturing/WorkOrderProductLine')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                bomCode: {
                                    code: true,
                                },
                                product: {
                                    code: true,
                                },
                            },
                            {
                                filter: {
                                    workOrder: this.workOrderNumber.value,
                                },
                            },
                        ),
                    )

                    .execute(),
            );
            return result[0];
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_completed_time_tracking___loading_BOM_code_error',
                    'Error loading BOM code',
                ),
                String(err),
            );
            return;
        }
    }

    /**
     * Set the rejected qnatity field to zero
     */
    async setRejectedQuantityToZero() {
        this.rejectedQuantity.value = 0;
    }

    /**
     * Set all runtime and setup time fields to zero
     */
    async SetAllTimeFieldsToZero() {
        this.setupTime.value = 0;
        this.runTime.value = 0;
        this.enteredRunTimeHours.value = 0;
        this.enteredRunTimeMinutes.value = 0;
        this.enteredSetupTimeHours.value = 0;
        this.enteredSetupTimeMinutes.value = 0;
        const { operationMethod }: TimeTrackingStorageObject = this.getStorageObject();
        if (operationMethod !== 0) {
            this.remainingQuantity.value = 1;
        }
    }
}
