import { TransactionProductionReporting, WorkOrder, WorkOrderProductLine } from '@sage/x3-manufacturing-api';
import { Filter, extractEdges } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
import { formatImportMessagefile } from '../client-functions/format-import-message-file';
import { validateWithDetails } from '../client-functions/validation';
import { manufacturing } from '../menu-items/manufacturing';
import { StorageObject, TrackingLine, TrackingLineTable, X3Response } from './utils/types';

@ui.decorators.page<MobileProductionReporting>({
    title: 'Production reporting',
    module: 'x3-manufacturing',
    mode: 'default',
    isTitleHidden: true,
    menuItem: manufacturing,
    authorizationCode: 'CWSAPT',
    access: { node: '@sage/x3-manufacturing/WorkOrder' },
    priority: 100,
    businessActions() {
        return [this.createButton];
    },
    async onLoad() {
        await this.initPage();
    },
})
export class MobileProductionReporting extends ui.Page {
    @ui.decorators.section<MobileProductionReporting>({
        isTitleHidden: true,
    })
    section: ui.containers.Section;

    @ui.decorators.block<MobileProductionReporting>({
        parent() {
            return this.section;
        },
        isTitleHidden: true,
    })
    block: ui.containers.Block;

    @ui.decorators.dateField<MobileProductionReporting>({
        parent() {
            return this.block;
        },
        title: 'Tracking date',
        isMandatory: true,
        maxDate: DateValue.today().toString(),
        onChange() {
            this.transaction.focus();
        },
    })
    trackingDate: ui.fields.Date;

    @ui.decorators.labelField<MobileProductionReporting>({
        parent() {
            return this.block;
        },
        title: 'Site',
        isHidden: true,
    })
    stockSite: ui.fields.Label;

    @ui.decorators.selectField<MobileProductionReporting>({
        parent() {
            return this.block;
        },
        title: 'Transaction',
        isTransient: true,
        isMandatory: true,
        async onChange() {
            !this.transaction.value ? (this.workOrder.isDisabled = true) : (this.workOrder.isDisabled = false);
        },
    })
    transaction: ui.fields.Select;

    @ui.decorators.referenceField<MobileProductionReporting, WorkOrder>({
        parent() {
            return this.block;
        },
        title: 'Work order',
        valueField: 'number',
        node: '@sage/x3-manufacturing/WorkOrder',
        placeholder: 'Scan or select...',
        isAutoSelectEnabled: true,
        isFullWidth: true,
        isDisabled: true,
        filter() {
            const filter: Filter<WorkOrder> = {
                productionSite: { code: this.stockSite.value },
                orderStatus: { _in: ['firm'] },
                modeType: 'complete',
                workOrderSuspendedFlag: false,
            };
            return filter;
        },
        onChange() {
            this.toggleProductLineField();
            if (this.productLine.isDisabled === false) {
                this.getSetWOProduct();
            }
        },
        columns: [
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrder',
                bind: 'productionSite',
                valueField: 'code',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'number',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrder',
                bind: 'releasedRouting',
                valueField: { routing: { code: true } },
            }),
            ui.nestedFields.text({
                bind: 'startDate',
                title: 'Start',
                isReadOnly: true,
                prefix: 'Start:',
            }),
        ],
    })
    workOrder: ui.fields.Reference;

    @ui.decorators.referenceField<MobileProductionReporting, WorkOrderProductLine>({
        parent() {
            return this.block;
        },
        title: 'Product',
        valueField: { product: { code: true } },
        node: '@sage/x3-manufacturing/WorkOrderProductLine',
        placeholder: 'Scan or select...',
        isFullWidth: true,
        isDisabled: true,
        isAutoSelectEnabled: true,
        async onChange() {
            this.validateGoToDetailsPage();
        },
        filter() {
            const filter: Filter<WorkOrderProductLine> = {
                workOrder: { number: this.workOrder.value?.number },
            };
            return filter;
        },
        columns: [
            ui.nestedFields.reference({
                bind: 'product',
                valueField: 'code',
                node: '@sage/x3-manufacturing/WorkOrderProductLine',
            }),
            ui.nestedFields.text({
                bind: 'releaseQuantity',
            }),
            ui.nestedFields.reference({
                bind: 'product',
                valueField: 'localizedDescription1',
                node: '@sage/x3-manufacturing/WorkOrderProductLine',
            }),
            ui.nestedFields.reference({
                bind: 'stockUnit',
                valueField: 'code',
                node: '@sage/x3-manufacturing/WorkOrderProductLine',
            }),
            ui.nestedFields.text({
                bind: 'remainingQuantity',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                bind: 'productCategory',
                valueField: 'code',
                node: '@sage/x3-manufacturing/WorkOrderProductLine',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                bind: 'product',
                valueField: 'lotManagementMode',
                node: '@sage/x3-manufacturing/WorkOrderProductLine',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                bind: 'product',
                valueField: 'lotSequenceNumber',
                node: '@sage/x3-manufacturing/WorkOrderProductLine',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                bind: 'product',
                valueField: 'serialNumberManagementMode',
                node: '@sage/x3-manufacturing/WorkOrderProductLine',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                bind: 'product',
                valueField: 'serialSequenceNumber',
                node: '@sage/x3-manufacturing/WorkOrderProductLine',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'lot',
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'lineNumber',
            }),
        ],
    })
    productLine: ui.fields.Reference;

    @ui.decorators.block<MobileProductionReporting>({
        parent() {
            return this.section;
        },
        isHidden: true,
    })
    trackingLinesBlock: ui.containers.Block;

    @ui.decorators.tableField<MobileProductionReporting>({
        parent() {
            return this.trackingLinesBlock;
        },
        title: 'Products',
        isTransient: true,
        canSelect: false,
        canFilter: false,
        columns: [
            ui.nestedFields.text({
                bind: 'product',
            }),
            ui.nestedFields.text({
                bind: 'quantityAndStockUnit',
            }),
            ui.nestedFields.text({
                bind: 'description',
            }),
        ],
        dropdownActions: [
            {
                icon: 'delete',
                title: 'Delete',
                onClick(rowId: any) {
                    let trackingLines = this.getLines();
                    trackingLines?.splice(rowId, 1);

                    if (!trackingLines?.length) {
                        this.workOrder.isDirty = !this.workOrder.isDirty;
                        this.$.router.goTo('@sage/x3-manufacturing/MobileProductionReporting');
                        this.toggleCreateButton(trackingLines.length);
                    } else {
                        this.setTableTitle(trackingLines, true);
                        this.setTableLines(trackingLines);
                        this.setSessionStorage();
                    }
                },
            },
        ],
    })
    trackingLinesTable: ui.fields.Table<TrackingLineTable>;

    @ui.decorators.pageAction<MobileProductionReporting>({
        title: 'Create',
        buttonType: 'primary',
        shortcut: ['f2'],
        isDisabled: true,
        async onClick() {
            this.$.loader.isHidden = false;
            this.prepareCreation();
            const result = await this.callCreationAPI();
            this.$.loader.isHidden = true;

            if (this.getCreationStatus(result)) {
                this.workOrder.isDirty = !this.workOrder.isDirty;
                this.successScreen(result);
            } else {
                this.errorScreen(result);
            }
        },
    })
    createButton: ui.PageAction;

    /** Create Button Methods */

    /**
     * Checks the response from X3 for errors. Controls the display success error screen.
     * First it handles a special case when in X3response:
     * severities 3 exists (hard fail),
     * workOrderProductionReporting.create.trackingNumber = null,
     * It Looks for severity 3 and 4, when none found Counts as success.
     * The rest is just an error.
     * @param {*} result The response from X3.
     * @return {*}  {boolean} True for win. False for fail.
     */

    getCreationStatus(result: X3Response): boolean {
        if (!!this.getTrackingNumber(result)) {
            return true;
        }
        if (this.checkForErrorSeverities(result)) {
            return true;
        }
        return false;
    }

    checkForErrorSeverities(result: X3Response): boolean {
        if (
            result.diagnoses?.filter((diagnoses: { severity: number; message: string }) => {
                diagnoses.severity > 2 && diagnoses.message;
            }).length !== 0 ||
            result.message
        ) {
            return false;
        } else {
            return true;
        }
    }

    async errorScreen(result: X3Response): Promise<void> {
        await this.$.dialog
            .confirmation(
                'error',
                'Error',
                `${ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting__creation_error',
                    'An error occurred in {{documentId}}.',
                    {
                        documentId: await formatImportMessagefile(result),
                    },
                )}`,
                this.errorDialogOption(),
            )
            .catch(() => {
                this.resetSessionStorage();
            });
    }

    async successScreen(result: X3Response): Promise<void> {
        await this.$.dialog
            .message(
                'info',
                'Success',
                `${ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting__creation_success',
                    'Work order tracking {{documentId}} created. {{errorMessages}}',
                    {
                        documentId: this.getTrackingNumber(result),
                        errorMessages: await formatImportMessagefile(result),
                    },
                )}`,
                this.successDialogOption(),
            )
            .finally(() => this.resetSessionStorage());
    }

    getTrackingNumber(result: X3Response) {
        const trackingNumber = result.x3Manufacturing?.workOrderProductionReporting.create?.trackingNumber;
        return trackingNumber;
    }

    /** The result diagnoses messages are formatted in such a way that even displaying them is a subject to debate
     * getResultMessage(result: X3Response): string {
     *   return result.diagnoses ? result.diagnoses[0]?.message : result.message;
     * }
     */

    /* Setting gotLines to false will cause initPageStorageObject method to reset the Page and Session Storage */
    resetSessionStorage(): void {
        this.$.router.goTo('@sage/x3-manufacturing/MobileProductionReporting', {
            gotLines: 'false',
        });
    }

    successDialogOption(): ui.dialogs.DialogOptions {
        const options = {
            acceptButton: {
                text: 'OK',
            },
        };
        return options;
    }

    errorDialogOption(): ui.dialogs.DialogOptions {
        const options = {
            acceptButton: {
                text: 'Back',
            },
            cancelButton: {
                text: 'Reset',
            },
        };
        return options;
    }

    toggleCreateButton(linesLength: number): boolean {
        if (linesLength > 0) {
            return (this.createButton.isDisabled = false);
        } else {
            return (this.createButton.isDisabled = true);
        }
    }

    prepareCreation(): void {
        const mutationObject = this.storageObject.quantityTracking.workOrderProductionReportingLines;
        mutationObject?.forEach(line => delete line.description);
        mutationObject?.forEach(line => delete line.quantityInStockUnit);
    }

    async callCreationAPI(): Promise<X3Response> {
        let result: X3Response;
        const inputData = { productionSite: this.stockSite.value, ...this.storageObject.quantityTracking };
        const payload = `
        mutation {
            x3Manufacturing {
                workOrderProductionReporting {
                    create(data: ${ui.queryUtils.serializeToGraphQL(inputData)}){
                        trackingNumber
                    }
                }
            }
        }
    `;

        try {
            result = await this.$.graph.raw(payload, true);
            if (!result) {
                throw Error(
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_production_reporting___no_results',
                        'No results received for the creation',
                    ),
                );
            }
        } catch (error) {
            return error;
        }
        return result;
    }

    async initPage() {
        let transactions = await this.getEntryTransaction();
        await this.setMandatoryFields(transactions);
        await this.setStorageObject();
    }

    /** Storage Methods */

    /**
     * Page Storage refers to the storageObject on page one.
     * Session Storage refers to the browsers Session Storage.
     */

    async setStorageObject() {
        this.initPageStorageObject();
        this.getSessionStorage();

        let linesLength = this.getLinesLength();
        let lines = this.getLines();

        if (linesLength) {
            this.setCurrentTrackingDate();
            this.disableEntryTransaction();
            this.setCurrentWorkOrder();
            this.toggleProductLineField();
            this.toggleTable(lines);
            this.toggleCreateButton(linesLength);
        }
    }

    private storageObject: StorageObject;

    initPageStorageObject() {
        this.storageObject = {
            transaction: '',
            trackingDate: '',
            workOrderNumber: '',
            productLine: {},
            quantityTracking: {
                workOrderProductionReportingLines: new Array<TrackingLine>(),
            },
        } as StorageObject;
        /**
         * Overwrite the Session Storage with empty StorageObject
         * if coming from place other than Details page. Resets the memory.
         */
        let gotLines = this.$.queryParameters['gotLines'];
        if (gotLines != 'true') this.setSessionStorage(this.storageObject);
    }

    /** Updates the Page Storage with Session Storage */
    getSessionStorage(): StorageObject {
        return (this.storageObject = this.$.storage?.get('production_reporting') as unknown as StorageObject);
    }

    /** Updates the Session Storage with the Page Storage */
    setSessionStorage(data = this.storageObject) {
        this.$.storage.set('production_reporting', { ...data } as any);
    }

    /** Executed on Product selection */
    updatePageStorage(): void {
        this.storageObject.productLine = this.productLine.value;
        this.storageObject.transaction = this.transaction.value;
        this.storageObject.trackingDate = this.trackingDate.value;
        this.storageObject.workOrderNumber = this.workOrder.value;
    }

    /** Fields Management */

    /** Stock site value comes from the sticker */
    async getStockSite(): Promise<string> {
        try {
            const site = this.$.storage.get('mobile-selected-stock-site') as string;
            return site;
        } catch (error) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting___stock_site_error',
                    'The stock site is not defined',
                ),
                { type: 'error', timeout: 5000 },
            );
            return;
        }
    }

    async getDate(): Promise<string> {
        return DateValue.today().toString();
    }

    async setMandatoryFields(transactions: string[]): Promise<boolean> {
        let transactionIsDisabled;

        this.stockSite.value = await this.getStockSite();
        if (this.stockSite.value) {
            this.trackingDate.value = await this.getDate();

            if (!transactions || transactions.length === 0) {
                this.block.isDisabled = true;
                this.$.showToast(
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_production_reporting___no_transaction_error',
                        'No transactions were found',
                    ),
                    { type: 'error', timeout: 5000 },
                );
                return (this.block.isDisabled = true);
            }
            transactionIsDisabled = transactions.length < 1;
            if (transactionIsDisabled) {
                this.setEntryTransaction(transactions, transactionIsDisabled, true);
            } else {
                this.setEntryTransaction(transactions, transactionIsDisabled, false);
            }

            if (this.stockSite.value && this.trackingDate.value && this.transaction.options) {
                return (this.workOrder.isDisabled = false);
            } else {
                return (this.block.isDisabled = true);
            }
        } else {
            this.block.isDisabled = true;
            this.$.dialog.message(
                'error',
                'Error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__default_stock_site_error',
                    'Define a default stock site on the user function profile.',
                ),
            );
            return (this.block.isDisabled = true);
        }
    }

    setEntryTransaction(transaction: string[], transactionIsDisabled: boolean, workOrderIsDisabled: boolean) {
        this.transaction.options = transaction;
        this.transaction.value = transaction[0];
        this.transaction.isDisabled = transactionIsDisabled;
        this.workOrder.isDisabled = workOrderIsDisabled;
    }

    toggleProductLineField(): void {
        if (this.workOrder.value) {
            this.productLine.isDisabled = false;
            this.productLine.focus();
        } else {
            this.productLine.isDisabled = true;
            this.productLine.value = null;
        }
    }

    /** Methods active when at least one production tracking line have been created */
    getLines(): TrackingLine[] | null | undefined {
        let trackingLines = this.storageObject?.quantityTracking.workOrderProductionReportingLines;
        if (trackingLines?.length > 0) {
            return trackingLines;
        }
    }

    getLinesLength(): number {
        let trackingLines = this.getLines();
        if (trackingLines) {
            return trackingLines.length;
        }
        return 0;
    }

    disableEntryTransaction(): void {
        this.transaction.isDisabled = true;
        this.transaction.value = this.storageObject.transaction;
    }

    setCurrentTrackingDate(): void {
        this.trackingDate.value = this.storageObject.trackingDate;
    }

    setCurrentWorkOrder(): WorkOrder {
        this.workOrder.isDirty = !this.workOrder.isDirty;
        return (this.workOrder.value = this.storageObject.productLine.workOrder);
    }

    /**
     * If production line is present in the storage show the table,
     * execute the setTableTitle, setTableLines and Order 66.
     * @param {TrackingLine[]} lines workOrderProductionReportingLines from the Page Storage.
     * @return {*}  {boolean} toggle the Table.
     */

    toggleTable(lines: TrackingLine[]): boolean {
        if (lines) {
            this.setTableTitle(lines);
            this.setTableLines(lines);
            return (this.trackingLinesBlock.isHidden = false);
        }
        return (this.trackingLinesBlock.isHidden = true);
    }

    /** The product counter number becomes a part of table title string. Needs a special handling when deleting lines */
    setTableTitle(lines: TrackingLine[], tableLineDelete?: boolean): string {
        let linesCounter = lines?.length;
        if (tableLineDelete) {
            this.trackingLinesTable.title = this.trackingLinesTable.title.split(':')[0];
        }
        return (this.trackingLinesTable.title = `${this.trackingLinesTable.title}: ${linesCounter}`);
    }

    /** Get the lines from Page Storage and map them to the Table lines */
    setTableLines(lines: TrackingLine[]): Partial<TrackingLineTable>[] {
        return (this.trackingLinesTable.value = this.mapTableLines(lines));
    }

    mapTableLines(lines: TrackingLine[]) {
        return lines.map((line: TrackingLine, index) => {
            let { product, description, actualQuantity, releaseUnit }: TrackingLine = line;
            return {
                _id: `${index}`,
                product: product,
                description: description,
                quantityAndStockUnit: `${actualQuantity} ${releaseUnit}`,
            };
        });
    }

    /** GraphQL Getters */

    async getEntryTransaction(): Promise<string[]> {
        let results: string[] = [];
        try {
            const transactions = extractEdges<TransactionProductionReporting>(
                await this.$.graph
                    .node('@sage/x3-manufacturing/TransactionProductionReporting')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                transaction: true,
                            },
                            {
                                filter: {
                                    type: 62,
                                    isActive: true,
                                    isPotency: false,
                                    isLotCustomText1: false,
                                    isLotCustomText2: false,
                                    isLotCustomNumber: false,
                                    isLotCustomDate: false,
                                },
                            },
                        ),
                    )
                    .execute(),
            ) as TransactionProductionReporting[];

            transactions.length === 1 ? (this.transaction.isHidden = true) : (this.transaction.isHidden = false);

            results = transactions.map(trs => trs.transaction);

            return results;
        } catch (err) {
            return;
        }
    }

    /** Get the product line details and set the field value when work order has only one released product */
    async getSetWOProduct() {
        try {
            const productLine = extractEdges<WorkOrderProductLine>(
                await this.$.graph
                    .node('@sage/x3-manufacturing/WorkOrderProductLine')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                product: {
                                    code: true,
                                    lotSequenceNumber: true,
                                    serialNumberManagementMode: true,
                                    lotManagementMode: true,
                                    localizedDescription1: true,
                                },
                                releaseQuantity: true,
                                stockUnit: { code: true },
                                remainingQuantity: true,
                                productCategory: { code: true },
                                lot: true,
                                workOrder: {
                                    number: true,
                                },
                                lineNumber: true,
                            },
                            {
                                filter: {
                                    workOrder: this.workOrder.value.number,
                                },
                            },
                        ),
                    )
                    .execute(),
            );
            if (productLine.length === 1) {
                this.productLine.value = productLine[0];
                this.validateGoToDetailsPage();
            }
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_completed_time_tracking___loading_product_error',
                    'Error while loading product',
                ),
                String(err),
            );
        }
    }

    /** Validate the page, set storage and session, go to the details page */
    async validateGoToDetailsPage() {
        if (!(await validateWithDetails(this))) return;

        this.updatePageStorage();
        this.setSessionStorage();
        this.$.setPageClean();
        this.$.router.goTo('@sage/x3-manufacturing/MobileProductionReportingDetails');
    }
}
