import { Container, ProductSite, UnitOfMeasure } from '@sage/x3-master-data-api';
import {
    LicensePlateNumber,
    Location,
    Lot,
    SerialNumber,
    StockManagementRules,
    StockStatus,
} from '@sage/x3-stock-data-api';
import { Filter, extractEdges } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
import { getUOM } from '../client-functions/get-unit-of-measure';
import { findDefaultLocation, findStockManagementRules } from '../client-functions/stock-management-rules';
import { validateWithDetails } from '../client-functions/validation';
import { PackingUnit, Product, ProductLine, StorageObject, TrackingLine } from './utils/types';

let lotManagement: string = '';
let stockManagementRule: StockManagementRules;

type StockLocation = {
    code: string;
    type: string;
};
type StockLocationEdge = {
    node: {
        location: StockLocation;
    };
};
type StockLocationResponse = {
    edges: StockLocationEdge[];
};

@ui.decorators.page<MobileProductionReportingDetails>({
    title: 'Production reporting details',
    isTransient: false,
    isTitleHidden: true,
    module: 'x3-manufacturing',
    mode: 'default',
    headerCard() {
        return {
            title: this.product,
            line2: this.localizedDescription,
            titleRight: this.workOrderNumber,
            line2Right: this.site,
            image: this.image,
        };
    },
    businessActions() {
        return [this.addQuantity, this.nextPage];
    },
    async onLoad() {
        await this.initPage();
        this.remainingQuantity.focus();
    },
})
export class MobileProductionReportingDetails extends ui.Page {
    private productSite: ProductSite;
    private packingUnits: PackingUnit[];
    private productUnitOfMeasure: Partial<UnitOfMeasure>;

    @ui.decorators.textField<MobileProductionReportingDetails>({
        isDisabled: true,
    })
    product: ui.fields.Text;

    @ui.decorators.textField<MobileProductionReportingDetails>({
        isDisabled: true,
        isTransient: true,
    })
    localizedDescription: ui.fields.Text;

    @ui.decorators.textField<MobileProductionReportingDetails>({
        isDisabled: true,
        isTransient: true,
        isHidden: true,
        prefix: 'Site:',
    })
    site: ui.fields.Text;

    @ui.decorators.textField<MobileProductionReportingDetails>({
        isDisabled: true,
        isTransient: true,
    })
    workOrderNumber: ui.fields.Text;

    @ui.decorators.section<MobileProductionReportingDetails>({
        isTitleHidden: true,
    })
    section: ui.containers.Section;

    @ui.decorators.block<MobileProductionReportingDetails>({
        parent() {
            return this.section;
        },
        isTitleHidden: true,
    })
    block: ui.containers.Block;

    @ui.decorators.imageField<MobileProductionReportingDetails>({
        isTransient: true,
        isReadOnly: true,
    })
    image: ui.fields.Image;

    @ui.decorators.dropdownListField<MobileProductionReportingDetails>({
        parent() {
            return this.block;
        },
        title: 'Unit',
        width: 'small',
        options: ['UN'],
        placeholder: 'Scan or select...',
        isMandatory: true,
        async onChange() {
            if (!this.stockUnit.value) return;
            const selectedValue = this.stockUnit.value;
            const packingUnitIndex = this.packingUnits
                .map(packingUnit => packingUnit.node.packingUnit.code)
                .indexOf(selectedValue);
            if (packingUnitIndex !== -1) {
                const selectedUnit = this.packingUnits[packingUnitIndex].node;
                this.conversionFactor.value = Number(selectedUnit.packingUnitToStockUnitConversionFactor);
                this.conversionFactor.isDisabled = !selectedUnit.isPackingFactorEntryAllowed;
                this.remainingQuantity.value = 0;
                this.remainingQuantity.scale = selectedUnit.packingUnit.numberOfDecimals;
            } else {
                this.conversionFactor.value = 1;
                this.conversionFactor.isDisabled = true;
                if (this.stockUnit.value) {
                    this.productUnitOfMeasure = await getUOM(this, this.stockUnit.value ?? '');
                    this.remainingQuantity.value = 0;
                    this.remainingQuantity.scale = this.productUnitOfMeasure.numberOfDecimals;
                }
            }
        },
    })
    stockUnit: ui.fields.DropdownList<string, MobileProductionReportingDetails>;

    @ui.decorators.numericField<MobileProductionReportingDetails>({
        parent() {
            return this.block;
        },
        title: 'Quantity',
        isNotZero: true,
        isMandatory: true,
        validation: /^([1-9][0-9]*(\.[0-9]+)?|[0]+\.[0-9]*[1-9][0-9]*)$/,
        onChange() {
            let lines: TrackingLine[] = this.getTrackingLines();
            let productLine: Partial<ProductLine> = this.getProductLine();
            let remainingQuantity = this.calculateQuantity(lines, productLine);
            let actualQuantityStockUnit: number = 0;
            let stockSite = this.getStockSite();

            actualQuantityStockUnit = this.convertUnitToStockUnit(
                this,
                productLine.product?.code ?? '',
                stockSite,
                Number(this.remainingQuantity.value),
                this.stockUnit.value ?? '',
                productLine.stockUnit?.code ?? '',
                this.conversionFactor.value ?? 1,
            );

            if (actualQuantityStockUnit) {
                if (actualQuantityStockUnit > remainingQuantity) {
                    this.$.showToast(
                        ui.localize(
                            '@sage/x3-manufacturing/pages__mobile_time_tracking_details___quantity__larger__expected__warning',
                            'The quantity is larger than expected.',
                        ),
                        { type: 'warning', timeout: 5000 },
                    );
                }
                if (actualQuantityStockUnit >= remainingQuantity) {
                    this.close.value = true;
                } else {
                    this.close.value = false;
                }
            }
        },
    })
    remainingQuantity: ui.fields.Numeric;

    @ui.decorators.numericField<MobileProductionReportingDetails>({
        parent() {
            return this.block;
        },
        title: 'Conversion factor',
        isDisabled: true,
        isTransient: true,
        isMandatory: true,
        isTransientInput: true,
        placeholder: 'Enter...',
        isNotZero: true,
        scale: 6,
    })
    conversionFactor: ui.fields.Numeric<MobileProductionReportingDetails>;

    @ui.decorators.selectField<MobileProductionReportingDetails>({
        parent() {
            return this.block;
        },
        title: 'Status',
        isMandatory: true,
        isTransient: false,
    })
    stockStatus: ui.fields.Select;

    @ui.decorators.referenceField<MobileProductionReportingDetails, Location>({
        parent() {
            return this.block;
        },
        title: 'Suggested location',
        node: '@sage/x3-stock-data/Location',
        valueField: 'code',
        isReadOnly: true,
        width: 'large',
        isTransient: true,
        canFilter: false,
    })
    suggestedLocation: ui.fields.Reference;

    @ui.decorators.referenceField<MobileProductionReportingDetails, Location>({
        parent() {
            return this.block;
        },
        node: '@sage/x3-stock-data/Location',
        title: 'Location',
        valueField: 'code',
        placeholder: 'Scan or select...',
        isMandatory: true,
        isAutoSelectEnabled: true,
        minLookupCharacters: 1,
        isFullWidth: true,
        isHidden: true,
        filter() {
            let stockSite = this.getStockSite();
            return {
                stockSite: { code: stockSite },
            };
        },
        onChange() {
            this.containerFieldLogic(this.container.value, this.location.value);
            this.location.getNextField(true)?.focus();
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Code',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'type',
                title: 'Type',
                isReadOnly: true,
            }),
        ],
    })
    location: ui.fields.Reference;

    @ui.decorators.referenceField<MobileProductionReportingDetails, Lot>({
        parent() {
            return this.block;
        },
        node: '@sage/x3-stock-data/Lot',
        title: 'Lot',
        valueField: 'code',
        isMandatory: false,
        placeholder: 'Scan or select...',
        isAutoSelectEnabled: true,
        filter() {
            let { product }: Partial<ProductLine> = this.getProductLine();

            let filter: Filter<Lot> = {
                product: { code: product?.code },
                code: { _ne: '' },
            };
            return filter;
        },
        async onChange() {
            if (this.lot.value) {
                this.lot.value.code = this.lot.value.code.toUpperCase();
                if (!this.sublot.isHidden) {
                    this.sublot.value = this.lot.value.sublot;
                    this.sublot.isDisabled = true;
                    await this.$.commitValueAndPropertyChanges();
                }
            } else {
                if (!this.sublot.isHidden) {
                    this.sublot.value = '';
                    await this.$.commitValueAndPropertyChanges();
                }
            }
            if (this.sublot.isHidden || this.sublot.isDisabled) {
                this.container.focus();
            }

            this.lotText.value = '';
            this.sublot.isReadOnly = false;
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Lot',
            }),
            ui.nestedFields.text({
                bind: 'sublot',
                title: 'Sublot',
            }),
        ],
    })
    lot: ui.fields.Reference;

    @ui.decorators.textField<MobileProductionReportingDetails>({
        parent() {
            return this.block;
        },
        title: 'Lot',
        maxLength: 15,
        validation: /^$|^[^|]+$/,
        async onChange() {
            if (this.lotText.value !== null) {
                this.lotText.value = this.lotText.value.trim().toUpperCase();
                const stocklot = await this.lotExists(this.lotText.value, this.product.value);
                if (stocklot > 0) {
                    await this.$.dialog.message(
                        'error',
                        ui.localize(
                            '@sage/x3-manufacturing/pages__mobile_production_reporting_details_quantity_notification__Lot_error',
                            'Error with the lot number',
                        ),
                        ui.localize(
                            '@sage/x3-manufacturing/pages__mobile_production_reporting_details_quantity_notification__Lot_exists_error',
                            'The lot number already exists',
                        ),
                    );
                    this.lotText.value = '';
                    this.lotText.focus();
                }
            }

            this.lot.value = null;
            this.sublot.isReadOnly = false;
            this.sublot.isDisabled = false;
        },
    })
    lotText: ui.fields.Text;

    @ui.decorators.textField<MobileProductionReportingDetails>({
        parent() {
            return this.block;
        },
        title: 'Sub-lot',
        isMandatory: true,
        maxLength: 5,
        validation: /^$|^[^|]+$/,
    })
    sublot: ui.fields.Text;

    @ui.decorators.dateField<MobileProductionReportingDetails>({
        parent() {
            return this.block;
        },
        title: 'Expiration date',
        isMandatory: true,
    })
    expirationDate: ui.fields.Date;

    @ui.decorators.textField<MobileProductionReportingDetails>({
        parent() {
            return this.block;
        },
        title: 'Serial number',
        validation: /^$|^[^|]+$/,
        isMandatory: true,
        isHidden: true,
        async onChange() {
            let { product } = this.getProductLine();
            let serialNumberCheck = await this.findSerialNumber(product, this.serialNumber.value);

            if (!serialNumberCheck) {
                await this.$.dialog.message(
                    'error',
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_completed_quantity-details-quantity_notification__serial_number_error',
                        'Error with the serial number',
                    ),
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_completed_quantity-details-quantity_notification__serial_number_exists_error',
                        'The serial number already exists',
                    ),
                );
                this.serialNumber.value = '';
                this.serialNumber.focus();
            }
        },
    })
    serialNumber: ui.fields.Text;

    @ui.decorators.referenceField<MobileProductionReportingDetails, Container>({
        parent() {
            return this.block;
        },
        node: '@sage/x3-master-data/Container',
        title: 'Container',
        valueField: 'code',
        placeholder: 'Scan or select...',
        isTransient: true,
        isFullWidth: true,
        isMandatory: true,
        isAutoSelectEnabled: true,
        shouldSuggestionsIncludeColumns: true,
        columns: [
            ui.nestedFields.text({
                bind: 'code',
            }),
        ],
        onChange() {
            this.containerFieldLogic(this.container.value, this.location.value);
            if (!this.licensePlateNumber.isDisabled) {
                this.licensePlateNumber.focus();
            }
        },
        onError(error: any, sourceScreenId: string, sourceElementId: string) {
            return 'An error!';
        },
        isHidden: true,
    })
    container: ui.fields.Reference;

    @ui.decorators.referenceField<MobileProductionReportingDetails, LicensePlateNumber>({
        parent() {
            return this.block;
        },
        node: '@sage/x3-stock-data/LicensePlateNumber',
        title: 'License plate number',
        valueField: 'code',
        placeholder: 'Scan or select...',
        isTransient: true,
        isFullWidth: true,
        isAutoSelectEnabled: true,
        isMandatory: true,
        columns: [
            ui.nestedFields.text({
                bind: 'code',
            }),
            ui.nestedFields.reference({
                bind: 'container',
                valueField: 'code',
                node: '@sage/x3-stock-data/LicensePlateNumber',
            }),
            ui.nestedFields.reference({
                bind: 'location',
                valueField: 'code',
                node: '@sage/x3-stock-data/LicensePlateNumber',
            }),
        ],
        onChange() {
            this.close.focus();
        },
        filter() {
            try {
                return {
                    container: this.container.value,
                    isActive: { _eq: true },
                    stockSite: { code: this.site.value },
                    location: {
                        code: {
                            _in: [this.location.value.code, undefined],
                        },
                    },
                };
            } catch (err) {
                return;
            }
        },
        isHidden: true,
    })
    licensePlateNumber: ui.fields.Reference;

    @ui.decorators.checkboxField<MobileProductionReportingDetails>({
        parent() {
            return this.block;
        },
        title: 'Close',
    })
    close: ui.fields.Checkbox;

    @ui.decorators.numericField<MobileProductionReportingDetails>({
        isTransient: true,
        isReadOnly: true,
        isHidden: true,
    })
    lotManaged: ui.fields.Numeric;

    @ui.decorators.numericField<MobileProductionReportingDetails>({
        isTransient: true,
        isReadOnly: true,
        isHidden: true,
    })
    sublotManaged: ui.fields.Numeric;

    @ui.decorators.pageAction<MobileProductionReportingDetails>({
        title: 'Add quantity',
        buttonType: 'secondary',
        shortcut: ['f2'],
        async onClick() {
            if (!(await this.lotValuesAreValid(lotManagement))) return;
            if (!(await validateWithDetails(this))) return;
            await this.buttonsLogic(true);
            this.$.router.goTo('@sage/x3-manufacturing/MobileProductionReportingDetails');
        },
    })
    addQuantity: ui.PageAction;

    @ui.decorators.pageAction<MobileProductionReportingDetails>({
        title: 'Next',
        buttonType: 'primary',
        shortcut: ['f3'],
        async onClick() {
            if (!(await this.lotValuesAreValid(lotManagement))) return;
            if (!(await validateWithDetails(this))) return;
            await this.buttonsLogic(false);
            this.$.router.goTo('@sage/x3-manufacturing/MobileProductionReporting', {
                gotLines: 'true',
            });
        },
    })
    nextPage: ui.PageAction;

    async initPage() {
        const stockSite: string = this.getStockSite();
        const { productLine, workOrderNumber }: StorageObject = this.getStorageObject();
        const { product, stockUnit, productCategory, lot }: Partial<ProductLine> = productLine;
        const unitOfMeasure: string | undefined = stockUnit?.code;
        const { serialNumberManagementMode, code }: Product = product;
        const lines: TrackingLine[] = this.getTrackingLines();
        const locationManagementMode = await this.getLocationManagementRules(code, stockSite);
        const { defaultInternalContainer } = locationManagementMode;

        lotManagement = await this.lotManagement(lot, productCategory);
        stockManagementRule = await findStockManagementRules(
            this.getStockSite(),
            productCategory || '',
            '5', // Work order receipt
            null,
            this,
        );
        this.setHeaderFields(product, stockSite);
        this.workOrderNumber.value = workOrderNumber.number;
        this.stockStatus.options = await this.getStockStatuses();
        this.stockStatus.value = stockManagementRule.defaultStatus;
        await this.expirationManagement(lotManagement, productLine.product.code);
        this.container.value = defaultInternalContainer;
        this.containerFieldLogic(this.container.value, this.location.value);
        this.close.value = true;
        this.toggleManageByLocationFields(locationManagementMode);
        this.toggleManageBySerialNumberField(serialNumberManagementMode);
        await this.setLotFieldBehaviour(lotManagement);
        this.productSite = await this.getProductSite();
        try {
            if (!this.suggestedLocation.isHidden) {
                this.suggestedLocation.value = await this.getLocations(stockSite);
                this.suggestedLocation.value.code = await findDefaultLocation(
                    this.productSite,
                    stockManagementRule,
                    this,
                );
                if (this.suggestedLocation.value.code && !this.location.isHidden) {
                    this.location.value = this.suggestedLocation.value;
                    this.location.value.code = this.suggestedLocation.value.code;
                }
                if (!this.suggestedLocation.value.code) this.suggestedLocation.isHidden = true;
            }
        } catch (error) {
            this.suggestedLocation.isHidden = true;
        }
        this.initPackingUnitFields();
        this.stockUnit.value = unitOfMeasure ?? 'UN';
        this.initPackingUnitToStockUnitConversionFactor();
        this.remainingQuantity.value = this.calculateQuantity(lines, productLine);
    }

    private async getProductSite() {
        try {
            const productSiteToReceive = await this.$.graph
                .node('@sage/x3-master-data/ProductSite')
                .read(
                    {
                        stockSite: { code: true },
                        isLocationManaged: true,
                        isLicensePlateNumberManaged: true,
                        defaultInternalContainer: {
                            code: true,
                        },
                        defaultLocations: {
                            query: {
                                edges: {
                                    node: {
                                        defaultLocation: true,
                                        locationNumber: true,
                                        defaultLocationType: true,
                                    },
                                },
                            },
                        },
                        product: {
                            code: true,
                            localizedDescription1: true,
                            productCategory: {
                                code: true,
                            },
                            lotManagementMode: true,
                            serialNumberManagementMode: true,
                            stockVersionMode: true,
                            expirationManagementMode: true,
                            serialSequenceNumber: true,
                            lotSequenceNumber: true,
                            stockManagementMode: true,
                            defaultPotencyInPercentage: true,
                            expirationLeadTime: true,
                            expirationTimeUnit: true,
                            useByDateCoefficient: true,
                            stockUnit: {
                                code: true,
                                numberOfDecimals: true,
                                _id: true,
                            },
                            packingUnits: {
                                query: {
                                    edges: {
                                        node: {
                                            packingUnit: {
                                                code: true,
                                                numberOfDecimals: true,
                                                _id: true,
                                            },
                                            packingUnitToStockUnitConversionFactor: true,
                                            isPackingFactorEntryAllowed: true,
                                            _id: true,
                                        },
                                    },
                                },
                            },
                        },
                    },

                    `${this.product.value}|${this.site.value}`,
                )
                .execute();
            return productSiteToReceive as any;
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details___product_site_error',
                    'Error while reading product site {{product_site}} for the product {{product}}.',
                    {
                        product_site: this.site.value,
                        product: this.product.value,
                    },
                ),
                String(err),
            );
            return {} as any;
        }
    }

    /**
     * Get stock site from this.$.storage
     * @returns string -  site code
     */
    getStockSite(): string {
        try {
            const stockSite = this.$.storage.get('mobile-selected-stock-site') as string;
            return stockSite;
        } catch (err) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_completed_quantity___stock_site_error',
                    'The stock site is not defined',
                ),
                { type: 'error', timeout: 5000 },
            );
            return '';
        }
    }

    getStorageObject(): StorageObject | undefined {
        try {
            let storageObject = this.$.storage.get('production_reporting') as any;
            return storageObject;
        } catch (err) {
            this.showStorageError();
            return undefined;
        }
    }

    getProductLine(): Partial<ProductLine> {
        try {
            let { productLine } = this.getStorageObject() || {};
            return productLine;
        } catch (err) {
            this.showStorageError();
            return {};
        }
    }

    getTrackingLines(): TrackingLine[] {
        try {
            let { quantityTracking } = this.getStorageObject() || {};
            return quantityTracking.workOrderProductionReportingLines;
        } catch (err) {
            this.showStorageError();
            return [];
        }
    }

    showStorageError(): void {
        this.$.showToast(
            ui.localize(
                '@sage/x3-manufacturing/pages__mobile_production_reporting_details___empty_storage_object_error',
                'Storage object is empty',
            ),
            { type: 'error', timeout: 5000 },
        );
        this.$.router.goTo('@sage/x3-manufacturing/MobileProductionReporting');
    }

    initTrackingLines(storageObject: any): TrackingLine[] {
        let { transaction, trackingDate, productLine, quantityTracking } = storageObject;
        let { product, stockUnit, workOrder } = productLine;
        let stockSite = this.getStockSite();

        let trackingLine = quantityTracking.workOrderProductionReportingLines;
        let lineCounter = 0;
        let actualQuantityStockUnit: number = 0;

        const YYYYMMDDdate = (date: string): string => {
            const year = date.slice(0, 4);
            const month = date.slice(5, 7);
            const day = date.slice(8, 10);
            return `${year}-${month}-${day}`;
        };

        let close: number = 1;
        if (this.close.value === true) {
            close = 2;
        }

        let lotValue = this.lotText.value === '' ? this.lot.value?.code : this.lotText.value;

        actualQuantityStockUnit = this.convertUnitToStockUnit(
            this,
            product.code,
            stockSite,
            Number(this.remainingQuantity.value),
            this.stockUnit.value ?? '',
            stockUnit.code,
            this.conversionFactor.value ?? 1,
        );

        trackingLine.length !== 0 && (lineCounter = trackingLine.length);

        trackingLine[lineCounter] = {
            ...trackingLine[lineCounter],
            productionSite: stockSite,
            orderNo: workOrder.number,
            product: product.code,
            orderLineNumber: productLine.lineNumber,
            actualQuantity: this.remainingQuantity.value,
            releaseUnit: this.stockUnit.value,
            stkConversion: this.conversionFactor.value ?? 1,
            postingDate: trackingDate,
            transactionCode: transaction,
            balance: close,
            rejectedQty: '',
            project: '',
            licensePlateNumber: this.licensePlateNumber.value?.code ?? '',
            serialNumber: this.serialNumber.value ?? '',
            destinationCode: this.$.storage.get('mobile-label-destination') as string,
            /** description removed before mutation */
            description: product.localizedDescription1,
            quantityInStockUnit: actualQuantityStockUnit,
            stockJournal: [
                {
                    expirationDate: YYYYMMDDdate(this.expirationDate.value ?? '2099-12-31'),
                    packingUnit: this.stockUnit.value,
                    quantityInPackingUnit: this.remainingQuantity.value,
                    packingUnitToStockUnitConversionFactor: this.conversionFactor.value ?? 1,
                    location: this.location.value?.code ?? '',
                    status: this.stockStatus.value,
                    lot: lotValue ?? '',
                    sublot: this.sublot.value ?? '',
                    identifier1: '',
                    identifier2: '',
                },
            ],
        } as TrackingLine[];
        return trackingLine;
    }

    saveQuantityTracking(storageObject: StorageObject) {
        let { quantityTracking } = storageObject;
        quantityTracking.workOrderProductionReportingLines = this.initTrackingLines(storageObject);
        this.$.storage.set('production_reporting', storageObject as any);
    }

    async buttonsLogic(displayMsg: boolean) {
        let storageObject = this.getStorageObject();
        this.saveQuantityTracking(storageObject);
        if (displayMsg === true) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details___quantity_added',
                    'Quantity added.',
                ),
                { type: 'success', timeout: 3000 },
            );
        }
        this.$.setPageClean();
    }

    /** Fields Methods */

    calculateQuantity(lines: TrackingLine[], productLine: Partial<ProductLine>): number {
        let { product, remainingQuantity, workOrder }: Partial<ProductLine> = productLine;
        let currentProduct: TrackingLine[] = lines.filter(
            (line: TrackingLine) => line.product === product?.code && line.orderNo === workOrder?.number,
        );
        let blockedQuantity: number = currentProduct.reduce((sum: number, item: any) => {
            return (sum = sum + item.quantityInStockUnit);
        }, 0);
        const workOrderRemainingQuantity = Number(remainingQuantity) || 0;
        let availableQuantity = Math.max(0, workOrderRemainingQuantity - blockedQuantity);
        return availableQuantity;
    }

    toggleManageBySerialNumberField(serialNumberManagementMode: string): boolean {
        if (serialNumberManagementMode === 'notManaged') {
            return (this.serialNumber.isHidden = true);
        } else {
            return (this.serialNumber.isHidden = false);
        }
    }

    toggleManageByLocationFields(locationManagementMode: any): void {
        let { isLocationManaged, isLicensePlateNumberManaged } = locationManagementMode;

        this.location.isHidden = !isLocationManaged;
        this.container.isHidden = !isLicensePlateNumberManaged;
        this.licensePlateNumber.isHidden = !isLicensePlateNumberManaged;
    }

    private async lotManagement(lot: string, productCategory: string) {
        const lotManagementMode = await this.getLotManagementMode();
        let lotManaged = 0;
        lotManaged = lotManagementMode === 'notManaged' ? 0 : 1;
        if (lotManaged !== 0) {
            if (lot !== '') {
                this.lotText.value = lot;
                this.lotText.isReadOnly = true;
                this.lot.isReadOnly = true;
                this.sublot.value = '00001';
                this.sublot.isReadOnly = true;
            }
        }
        return lotManagementMode;
    }

    async expirationManagement(lotManagement: string, product: string) {
        if (!lotManagement) {
            this.expirationDate.isHidden = true;
        } else {
            let expirationManagement = await this.getExpirationManagementMode(product);
            let expirationUOM = 0;
            expirationUOM = expirationManagement[0].expirationTimeUnit === 'month' ? 0 : 1;

            let calculatedDate = DateValue.today();

            switch (expirationManagement[0].expirationManagementMode) {
                case 'notManaged': {
                    this.expirationDate.isDisabled = true;
                    return;
                }
                case 'withoutRounding':
                case 'mandatoryEntry': {
                    if (expirationUOM) {
                        calculatedDate = calculatedDate.addDays(expirationManagement[0].expirationLeadTime);
                    } else {
                        calculatedDate = calculatedDate.addMonths(expirationManagement[0].expirationLeadTime);
                    }
                    break;
                }
                case 'roundingMonthEnd': {
                    if (expirationUOM) {
                        calculatedDate = calculatedDate.endOfMonth();
                    } else {
                        calculatedDate = calculatedDate
                            .addMonths(expirationManagement[0].expirationLeadTime)
                            .endOfMonth();
                    }
                    break;
                }
                case 'roundingBeginningMonth1': {
                    if (expirationUOM) {
                        calculatedDate = calculatedDate.addMonths(1).begOfMonth();
                    } else {
                        calculatedDate = calculatedDate
                            .addMonths(expirationManagement[0].expirationLeadTime + 1)
                            .begOfMonth();
                    }
                    break;
                }
                case 'manualEntry': {
                    return;
                }
            }
            this.expirationDate.value = calculatedDate.toString();
        }
    }

    containerFieldLogic(containerValue: Partial<Container>, locationValue: Partial<Location>) {
        if (containerValue && locationValue) {
            this.licensePlateNumber.isDisabled = false;
        } else {
            this.licensePlateNumber.isDisabled = true;
            this.licensePlateNumber.value = null;
        }
    }

    /**
     * Populate header card
     * @param {Partial<Product>} { code, localizedDescription1 }
     * @param {string} site
     * @memberof MobileProductionReportingDetails
     */
    async setHeaderFields({ code, localizedDescription1 }: Partial<Product>, site: string) {
        this.product.value = code;
        this.localizedDescription.value = localizedDescription1;
        this.site.value = site;

        // Display product picture
        const productPicture = await this.$.graph
            .node('@sage/x3-master-data/Product')
            .read({ picture: { value: true } }, `${this.product.value}| `)
            .execute();
        this.image.value = productPicture?.picture ?? undefined;
        this.$.setPageClean();
    }

    async getStockStatuses(): Promise<string[]> {
        try {
            return extractEdges<StockStatus>(
                await this.$.graph
                    .node('@sage/x3-stock-data/StockStatus')
                    .query(
                        ui.queryUtils.edgesSelector({
                            code: true,
                        }),
                    )
                    .execute(),
            ).map((status: StockStatus) => status.code);
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details___loading_stock_status_error',
                    'Error loading stock statuses',
                ),
                String(err),
            );
            return [''];
        }
    }

    /**
     * Get the locations for the stock site
     * @param site
     * @returns StockLocation[]
     */
    async getLocations(site: string): Promise<StockLocation[]> {
        const response: StockLocationResponse = await this.$.graph
            .node('@sage/x3-stock-data/Stock')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        location: { code: true, type: true },
                    },
                    {
                        filter: {
                            stockSite: { code: site },
                        },
                    },
                ),
            )
            .execute();

        if (!response.edges || response.edges.length === 0) {
            throw new Error(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details__loading_location_error',
                    'No location found for the stock site',
                ),
            );
        }
        return response.edges.map(({ node: { location } }) => location);
    }

    async getLocationManagementRules(product: string, stockSite: string) {
        try {
            const result = extractEdges<ProductSite>(
                await this.$.graph
                    .node('@sage/x3-master-data/ProductSite')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                isLocationManaged: true,
                                isLicensePlateNumberManaged: true,
                                defaultInternalContainer: { code: true },
                            },
                            {
                                filter: {
                                    product: { code: product },
                                    stockSite: {
                                        code: stockSite,
                                    },
                                },
                            },
                        ),
                    )
                    .execute(),
            );
            return result[0];
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details___loading_location_management_error',
                    'Error loading location management status',
                ),
                String(err),
            );
            return;
        }
    }

    async getLotManagementMode(): Promise<string> {
        let { product }: Partial<ProductLine> = this.getProductLine();
        try {
            const result = extractEdges<Product>(
                await this.$.graph
                    .node('@sage/x3-master-data/Product')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                lotManagementMode: true,
                            },
                            {
                                filter: {
                                    code: product?.code,
                                },
                            },
                        ),
                    )
                    .execute(),
            );
            return result[0].lotManagementMode as string;
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details___getting_lot_management_product_error',
                    'Error getting lot management for product',
                ),
                String(err),
            );
            return '';
        }
    }

    async getCategoryLotManagementMode(productCategory: string): Promise<string> {
        try {
            const result = extractEdges<Product>(
                await this.$.graph
                    .node('@sage/x3-master-data/ProductCategory')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                lotManagementMode: true,
                            },
                            {
                                filter: {
                                    code: productCategory,
                                },
                            },
                        ),
                    )
                    .execute(),
            );
            return result[0].lotManagementMode as string;
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details___getting_lot_management_product_category_error',
                    'Error getting lot management for product category',
                ),
                String(err),
            );
            return '';
        }
    }

    private async lotExists(lotCode: string, product: string): Promise<number> {
        try {
            const response = await this.$.graph
                .node('@sage/x3-stock-data/Lot')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            code: true,
                            sublot: true,
                        },
                        {
                            filter: {
                                code: lotCode,
                                product: product,
                            },
                        },
                    ),
                )
                .execute();
            return response.edges.length;
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details___loading_lot_error',
                    'Error loading lot',
                ),
                String(err),
            );
            return 0;
        }
    }

    /**
     * findSerialNumber - search table STOSER for a selected product serial number
     * @param product
     * @param serialNumber
     * @returns boolean
     */
    async findSerialNumber(product: any, serialNumber: string): Promise<boolean> {
        try {
            const serialNoResults = extractEdges<SerialNumber>(
                await this.$.graph
                    .node('@sage/x3-stock-data/SerialNumber')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                code: true,
                            },
                            {
                                filter: {
                                    product: product.code,
                                    code: serialNumber,
                                },
                            },
                        ),
                    )
                    .execute(),
            );

            if (serialNoResults.length > 0) {
                return false;
            } else {
                return true;
            }
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details___checking_serial_number_error',
                    'Error checking existing serial numbers',
                ),
                String(err),
            );
            return false;
        }
    }

    async getExpirationManagementMode(product: string): Promise<any> {
        try {
            const result = extractEdges<Product>(
                await this.$.graph
                    .node('@sage/x3-master-data/Product')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                expirationManagementMode: true,
                                expirationStockStatus: true,
                                expirationLeadTime: true,
                                expirationTimeUnit: true,
                            },
                            {
                                filter: {
                                    code: product,
                                },
                            },
                        ),
                    )
                    .execute(),
            );
            return result;
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details___expiration_management_error',
                    'Error getting expiration management for product',
                ),
                String(err),
            );
            return;
        }
    }

    /** Get the LOT sequence for a product */
    async getLotSequence(): Promise<string> {
        let { product }: Partial<ProductLine> = this.getProductLine();
        try {
            const result = extractEdges<Product>(
                await this.$.graph
                    .node('@sage/x3-master-data/Product')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                lotSequenceNumber: true,
                            },
                            {
                                filter: {
                                    code: product?.code,
                                },
                            },
                        ),
                    )
                    .execute(),
            );
            return result[0].lotSequenceNumber as string;
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_production_reporting_details___getting_lot_sequence_product_error',
                    'Error getting lot sequence for product.',
                ),
                String(err),
            );
            return '';
        }
    }

    /**
     *  Set the isHidden and isMandatory values for UI
     *  properties affected by a products lot management mode
     *
     * @private
     * @param {string} mode
     * @memberof MobileProductionReportingDetails
     */
    private async setLotFieldBehaviour(mode: string) {
        if (mode === 'notManaged') {
            this.lot.isHidden = true;
            this.lotText.isHidden = true;
            this.sublot.isHidden = true;
            this.expirationDate.isHidden = true;
        }
        if (mode === 'optionalLot' || lotManagement === 'mandatoryLot') {
            this.lot.isHidden = false;
            this.lotText.isHidden = false;
            this.lot.isMandatory = false;
            this.lotText.isMandatory = false;
            this.sublot.isHidden = true;
        }

        if (mode === 'lotAndSublot') {
            this.lot.isHidden = false;
            this.lotText.isHidden = false;
            this.lot.isMandatory = false;
            this.lotText.isMandatory = false;
            this.sublot.isHidden = false;
        }
    }
    /**
     * Test that the required lot values have been entered as
     * required by the products lot management rules
     * @param {*} mode
     * @return {*}  {Promise<boolean>}
     * @memberof MobileProductionReportingDetails
     */
    async lotValuesAreValid(mode: string): Promise<boolean> {
        if (mode === 'mandatoryLot' || mode === 'lotAndSublot') {
            if (stockManagementRule.lotEntry !== 'free') {
                if (!this.lot.value && !this.lotText.value) {
                    this.$.showToast(
                        ui.localize(
                            '@sage/x3-manufacturing/pages__mobile_production_reporting_details___lot_not_set_error',
                            'A lot has not been entered',
                        ),
                        { type: 'error', timeout: 5000 },
                    );
                    return false;
                }
            }
        }
        return true;
    }

    // Hotfix functions for unit of measure - packing units
    /**
     * Determines all the packing units of the product site and sets them to the packingUnits array.
     * Fills the stock unit options with the stock unit and packing units.
     */

    initPackingUnitFields() {
        // Gets all the packing units from the product site.
        let productPackingList = extractEdges(this.productSite.product.packingUnits.query).filter(productPacking => {
            return !!productPacking.packingUnit?.code;
        });
        // Gets the node for each packing unit.
        this.packingUnits = productPackingList.map(productPacking => {
            return { node: productPacking };
        });
        // Adds the stock unit as a packing unit.
        this.packingUnits.push({
            node: {
                packingUnit: {
                    code: this.productSite.product.stockUnit.code,
                    numberOfDecimals: this.productSite.product.stockUnit.numberOfDecimals,
                    _id: this.productSite.product.stockUnit._id,
                },
                packingUnitToStockUnitConversionFactor: '1',
                isPackingFactorEntryAllowed: false,
                _id: this.productSite.product.stockUnit._id,
            },
        });
        // Gets only the packing unit codes from the product packing list.
        let productPackingUnitSelectValues = productPackingList.map(productPacking => {
            return productPacking.packingUnit.code;
        });

        this.conversionFactor.isHidden = false;
        this.stockUnit.isHidden = false;
        // Fills the stock unit options with the stock unit and packing units.
        this.stockUnit.options = [this.productSite.product.stockUnit.code, ...productPackingUnitSelectValues];
    }

    /**
     * Determines for the stock unit the conversion factor and enables or disables the conversion factor due to the packing unit array.
     * Sets the number of decimals (scale) for the remaining quantity field.
     */
    async initPackingUnitToStockUnitConversionFactor() {
        const selectedValue = this.stockUnit.value;
        const packingUnitIndex = selectedValue
            ? this.packingUnits.map(packingUnit => packingUnit.node.packingUnit.code).indexOf(selectedValue)
            : -1;

        if (packingUnitIndex !== -1) {
            const selectedUnit = this.packingUnits[packingUnitIndex].node;
            this.conversionFactor.value = Number(selectedUnit.packingUnitToStockUnitConversionFactor);
            this.conversionFactor.isDisabled = !selectedUnit.isPackingFactorEntryAllowed;
            this.remainingQuantity.scale = selectedUnit.packingUnit.numberOfDecimals;
        } else {
            this.conversionFactor.value = 1;
            this.conversionFactor.isDisabled = true;
            if (this.stockUnit.value) {
                this.productUnitOfMeasure = await getUOM(this, this.stockUnit.value ?? '');
                this.remainingQuantity.scale = this.productUnitOfMeasure.numberOfDecimals;
            }
        }
    }

    /**
     * Converts a quantity from a given unit of measure (UOM) to the stock unit.
     * @param page The current page instance.
     * @param productCode The product code for which the conversion is done.
     * @param stockSite The stock site code where the product is located.
     * @param quantity The quantity to convert.
     * @param uom The unit of measure code of the quantity.
     * @param stockUnitUom The stock unit of measure code.
     * @param conversionFactor The conversion factor to apply if different to the definition entered.
     * @returns The converted quantity in stock unit.
     */
    convertUnitToStockUnit(
        page: MobileProductionReportingDetails,
        productCode: string,
        stockSite: string,
        quantity: number,
        uom: string,
        stockUnitUom: string,
        conversionFactor: number,
    ): number {
        if (!productCode || !stockSite || !quantity || !uom || !stockUnitUom || !conversionFactor) return 0;

        let packingUnits: PackingUnit[];
        let convertedQuantity: number = 0;

        packingUnits = page.productSite.product.packingUnits?.query.edges.map(packingUnit => {
            return packingUnit;
        });
        packingUnits.push({
            node: {
                packingUnit: {
                    code: page.productSite.product.stockUnit.code,
                    numberOfDecimals: page.productSite.product.stockUnit.numberOfDecimals,
                    _id: page.productSite.product.stockUnit._id,
                },
                packingUnitToStockUnitConversionFactor: '1',
                isPackingFactorEntryAllowed: false,
                _id: page.productSite.product.stockUnit._id,
            },
        });

        const packingUnit = packingUnits.find(pu => pu.node.packingUnit.code === uom);
        let factor = packingUnit ? Number(packingUnit.node.packingUnitToStockUnitConversionFactor) || 1 : 1;

        if (conversionFactor !== factor) {
            factor = conversionFactor;
        }

        let decimals = 0;
        const stockUnit = packingUnits.find(pu => pu.node.packingUnit.code === stockUnitUom);
        if (packingUnit && stockUnit?.node.packingUnit.numberOfDecimals !== undefined) {
            decimals = stockUnit.node.packingUnit.numberOfDecimals;
        }

        convertedQuantity = Number(quantity * factor);
        convertedQuantity = Number(convertedQuantity.toFixed(decimals));

        return convertedQuantity;
    }
}
