import { GraphApi, WorkOrder, WorkOrderMaterialLine, WorkOrderProductLine } from '@sage/x3-manufacturing-api';
import { ProductSite } from '@sage/x3-master-data-api';
import { Filter, integer } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
import { formatImportMessagefile } from '../client-functions/format-import-message-file';
import { getSelectedStockSite } from '../client-functions/get-selected-stock-site';
import {
    MaterialsConsumptionDetailsRecordManager,
    MaterialsConsumptionSession,
    materialsConsumptionDetail,
    validate,
} from '../client-functions/materials-consumption-details-control';
import { manufacturing } from '../menu-items/manufacturing';
import { MaterialTrackingLine, StorageObjectMaterial, X3Response, keyMaterialTrackingLine } from './utils/types';

let WorkUnplannedMaterials: boolean[];
let WorkNumberOfProductsReleased: integer;

interface StockWithdrawalMode {
    entryTransactionMode: string | undefined;
    permittedProductModes: string | undefined | null;
    permittedComponents: string[] | undefined;
}

// Object to store modes and components used to filter components
const stockWithdrawalMode: StockWithdrawalMode = {
    entryTransactionMode: '',
    permittedProductModes: '',
    permittedComponents: [],
};

@ui.decorators.page<MobileMaterialsConsumption>({
    title: 'Material consumption',
    module: 'x3-manufacturing',
    mode: 'default',
    menuItem: manufacturing,
    priority: 200,
    isTransient: false,
    isTitleHidden: true,
    authorizationCode: 'CWSAMT',
    access: { node: '@sage/x3-manufacturing/TransactionProductionReporting' },
    async onLoad() {
        stockWithdrawalMode.permittedProductModes = await this.getStockWithdrawalMode(61, this.transaction.value || '');
        await this.initMaterialsConsumptionPage();
    },
    businessActions() {
        return [this.createButton];
    },
})
export class MobileMaterialsConsumption extends ui.Page<GraphApi> {
    private _materialsConsumptionRecordManager: MaterialsConsumptionDetailsRecordManager;
    private _materialsConsumptionSession: MaterialsConsumptionSession;
    private _gDosFab: string;
    private _gMtkAllMgt: string;
    private _productIsNotReleasedIfShortage: string;

    @ui.decorators.pageAction<MobileMaterialsConsumption>({
        title: 'Create',
        buttonType: 'primary',
        shortcut: ['f2'],
        isDisabled: true,
        async onClick() {
            if (!this._materialsConsumptionSession) {
                this.$.showToast(
                    ui.localize(
                        '@sage/x3-manufacturing/notification-error-materials-consumption-no-products',
                        `Please input at least one product`,
                    ),
                    { type: 'error', timeout: 5000 },
                );
                return;
            }

            let validateError;
            if ((validateError = await this.trackingDate.validate())) {
                this.$.removeToasts();
                this.$.showToast(
                    `${ui.localize(
                        '@sage/x3-manufacturing/notification-error-materials-consumption-error',
                        'Error',
                    )}: ${validateError}`,
                    { type: 'error', timeout: 30000 },
                );
                return;
            }

            this.createButton.isDisabled = true;
            this.$.loader.isHidden = false;
            this.prepareCreation();
            this.sortCreation();
            this.calculateQuantityCreation();
            const result = await this.callCreationAPI();
            ui.console.log('CreationAPI', result);
            this.$.loader.isHidden = true;

            if (this.getCreationStatus(result)) {
                this.workOrder.isDirty = !this.workOrder.isDirty;
                this.successScreen(result);
            } else {
                this.errorScreen(result);
                this.createButton.isDisabled = false;
            }
        },
    })
    createButton: ui.PageAction;

    /** Create Button Methods */

    /**
     * Checks the response from X3 for errors. Controls the display success error screen.
     * First it handles a special case when in X3response:
     * severities 3 exists (hard fail),
     * workOrderProductionReporting.create.trackingNumber = null,
     * but the MFGTRACKNUM is being returned in diagnoses.message. Counts as success.
     * Than it Looks for severity 3 and 4, when none found Counts as success.
     * The rest is just an error.
     * @param {*} result The response from X3.
     * @return {*}  {boolean} True for win. False for fail.
     */

    getCreationStatus(result: X3Response): boolean {
        if (!!this.getTrackingNumber(result)) {
            return true;
        }
        if (this.checkForErrorSeverities(result)) {
            return true;
        }
        return false;
    }

    getTrackingNumber(result: X3Response): string | undefined {
        let trackingNumber: string = '';
        if (
            result.x3Manufacturing?.workOrderMaterialTracking.create?.trackingNumber !== null &&
            result.x3Manufacturing?.workOrderMaterialTracking.create?.trackingNumber !== undefined
        ) {
            trackingNumber = result.x3Manufacturing?.workOrderMaterialTracking.create?.trackingNumber;
        }
        return trackingNumber;
    }

    checkForErrorSeverities(result: X3Response): boolean {
        if (
            result.diagnoses.filter((diagnoses: { severity: number; message: string }) => {
                diagnoses.severity > 2 && diagnoses.message;
            }).length !== 0 ||
            result.message
        ) {
            return false;
        } else {
            return true;
        }
    }

    async errorScreen(result: X3Response): Promise<void> {
        await this.$.dialog.confirmation(
            'error',
            'Error',
            `${ui.localize(
                '@sage/x3-manufacturing/pages__mobile_materials_consumption__creation_error',
                'An error occurred in {{documentId}}.',
                {
                    documentId: await formatImportMessagefile(result, 2),
                },
            )}`,
            this.errorDialogOption(),
        );
    }

    async successScreen(result: X3Response): Promise<void> {
        await this.$.dialog
            .message(
                'info',
                'Success',
                `${ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_materials_consumption__creation_success',
                    'Work order material consumption {{documentId}} created. {{errorMessages}}',
                    {
                        documentId: this.getTrackingNumber(result),
                        errorMessages: await formatImportMessagefile(result, 2),
                    },
                )}`,
                this.successDialogOption(),
            )
            .finally(() => this.resetSessionStorage());
    }

    errorDialogOption(): ui.dialogs.DialogOptions {
        const options = {
            acceptButton: {
                text: 'Back',
            },
            cancelButton: {
                text: 'Reset',
            },
        };
        return options;
    }

    successDialogOption(): ui.dialogs.DialogOptions {
        const options = {
            acceptButton: {
                text: 'OK',
            },
        };
        return options;
    }

    resetSessionStorage(): void {
        this.$.router.goTo('@sage/x3-manufacturing/MobileMaterialsConsumption', {
            gotLines: 'false',
        });
    }

    @ui.decorators.section<MobileMaterialsConsumption>({
        isTitleHidden: true,
    })
    mainSection: ui.containers.Section;

    @ui.decorators.block<MobileMaterialsConsumption>({
        parent() {
            return this.mainSection;
        },
        isTitleHidden: true,
    })
    productBlock: ui.containers.Block;

    @ui.decorators.block<MobileMaterialsConsumption>({
        parent() {
            return this.mainSection;
        },
        isTitleHidden: true,
        isHidden: true,
    })
    workOrderBlock: ui.containers.Block;

    @ui.decorators.dateField<MobileMaterialsConsumption>({
        parent() {
            return this.productBlock;
        },
        title: 'Tracking date',
        isTransient: true,
        isMandatory: true,
        maxDate: DateValue.today().toString(), // restricts to current date or earlier
        onChange() {
            if (this.trackingDate.value) {
                this.transaction.focus();
            }
        },
    })
    trackingDate: ui.fields.Date;

    @ui.decorators.labelField<MobileMaterialsConsumption>({
        parent() {
            return this.productBlock;
        },
        title: 'Site',
        isTransient: true,
        isHidden: true,
    })
    materialsConsumptionSite: ui.fields.Label;

    @ui.decorators.selectField<MobileMaterialsConsumption>({
        parent() {
            return this.productBlock;
        },
        title: 'Transaction',
        options: [],
        isMandatory: true,
        async onChange() {
            if (this.transaction.value) {
                if (WorkUnplannedMaterials[this.transaction.options.indexOf(this.transaction.value)] === true) {
                    // Set it to Yes & visible
                    this.unplannedMaterials.value = false;
                    this.unplannedMaterials.isHidden = false;
                } else {
                    // Set it to No & not visible
                    this.unplannedMaterials.value = false;
                    this.unplannedMaterials.isHidden = true;
                }
                this.workOrder.value = null;
                stockWithdrawalMode.permittedProductModes = await this.getStockWithdrawalMode(
                    61,
                    this.transaction.value,
                );
                this.workOrder.focus();
            }
        },
    })
    transaction: ui.fields.Select;

    @ui.decorators.labelField<MobileMaterialsConsumption>({
        isTransient: true,
        isHidden: true,
    })
    legalCompany: ui.fields.Label;

    @ui.decorators.referenceField<MobileMaterialsConsumption, WorkOrder>({
        parent() {
            return this.productBlock;
        },
        title: 'Work order',
        placeholder: 'Scan or select...',
        node: '@sage/x3-manufacturing/WorkOrder',
        valueField: 'number',
        filter() {
            const filter: Filter<WorkOrder> = {
                _and: [
                    { productionSite: { code: this.materialsConsumptionSite.value } },
                    { orderStatus: 'firm' },
                    { modeType: 'complete' },
                    { workOrderSuspendedFlag: false },
                ],
            };
            if (this._gDosFab === '2') {
                filter._and.push({
                    trackingFlag: { _nin: ['pending', 'beingOptimized'] },
                });
            }
            if (this._gMtkAllMgt === '2') {
                filter._and.push({
                    _or: [{ allocationStatus: { _in: ['complete', 'completeShortage'] } }, { numberOfMaterials: '0' }],
                });
            }
            return filter;
        },
        async onChange() {
            if (this.workOrder.value) {
                const isReleasedIfShortage = await this._isReleasedIfShortage(this.workOrder.value.number);
                if (isReleasedIfShortage === false) {
                    await this.$.dialog
                        .message(
                            'error' as ui.dialogs.DialogLevel,
                            'Error',
                            ui.localize(
                                '@sage/x3-manufacturing/dialog-error-material-is-short-and-is-prohibited-to-track-this-WO',
                                'Material is short and it is prohibited to track this WO. ({{productId}})',
                                {
                                    productId: this._productIsNotReleasedIfShortage,
                                },
                            ),
                            { fullScreen: true, acceptButton: { text: 'OK' } },
                        )
                        .catch(() => {
                            return;
                        });
                    this.workOrder.value = null;
                }
            }
            if (this.workOrder.value) {
                await this.calcProductsReleased();
                stockWithdrawalMode.permittedComponents = await this.getComponents(
                    this.workOrder.value.number,
                    this.materialsConsumptionSite.value,
                    stockWithdrawalMode.permittedProductModes,
                );
            }

            if (this.workOrder.value) {
                this.unplannedMaterials.value = false;
                if (WorkNumberOfProductsReleased === 1) {
                    this.product.isHidden = false;
                    this.unplannedProduct.isHidden = true;
                    this.product.isMandatory = true;
                    this.unplannedProduct.isMandatory = false;
                    this.product.value = {};
                    this.unplannedProduct.value = {};
                    this.product.focus();
                } else {
                    this.product.isHidden = true;
                    this.unplannedProduct.isHidden = true;
                    this.product.isMandatory = false;
                    this.unplannedProduct.isMandatory = false;
                    this.product.value = {};
                    this.unplannedProduct.value = {};
                    this.productsReleased.focus();
                }
            }
        },
        isTransient: true,
        isFullWidth: true,
        isMandatory: true,
        isAutoSelectEnabled: true,
        columns: [
            ui.nestedFields.text({
                bind: 'number',
                title: 'Order no.',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrder',
                bind: 'releasedRouting',
                valueField: { routing: { code: true } },
            }),

            ui.nestedFields.text({
                bind: 'startDate',
                title: 'Start date',
                isReadOnly: true,
                prefix: 'Start date:',
            }),
        ],
    })
    workOrder: ui.fields.Reference;

    @ui.decorators.checkboxField<MobileMaterialsConsumption>({
        parent() {
            return this.productBlock;
        },
        title: 'Unplanned materials',
        //optionType: '@sage/xtrem-x3-master-data/NoYes',
        onChange() {
            if (this.unplannedMaterials.value == false) {
                this.product.isHidden = false;
                this.unplannedProduct.isHidden = true;
                this.product.isMandatory = true;
                this.unplannedProduct.isMandatory = false;
                this.product.value = {};
                this.unplannedProduct.value = {};
            } else if (this.unplannedMaterials.value == true) {
                this.product.isHidden = true;
                this.unplannedProduct.isHidden = false;
                this.product.isMandatory = false;
                this.unplannedProduct.isMandatory = true;
                this.product.value = {};
                this.unplannedProduct.value = {};
            }
        },
    })
    unplannedMaterials: ui.fields.Checkbox;

    @ui.decorators.checkboxField<MobileMaterialsConsumption>({
        parent() {
            return this.productBlock;
        },
        isHidden: true,
    })
    workCurrentUnplannedMaterials: ui.fields.Checkbox;

    @ui.decorators.referenceField<MobileMaterialsConsumption, WorkOrderProductLine>({
        parent() {
            return this.productBlock;
        },
        title: 'Product',
        placeholder: 'Scan or select...',
        isTransientInput: false,
        isTransient: false,
        node: '@sage/x3-manufacturing/WorkOrderProductLine',
        valueField: { product: { code: true } },
        filter() {
            return { workOrder: { number: this.workOrder.value?.number }, productType: { _eq: 'product' } };
        },
        columns: [
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/Product',
                bind: 'product',
                valueField: '_id',
                title: 'Product',
                isReadOnly: true,
                isTitleHidden: false,
            }),
            ui.nestedFields.text({
                bind: 'lineNumber',
                title: 'Line',
                isReadOnly: true,
                isTitleHidden: false,
            }),

            ui.nestedFields.reference({
                node: '@sage/x3-master-data/Product',
                bind: 'product',
                valueField: 'description1',
                title: 'Description',
                isReadOnly: true,
                isTitleHidden: false,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrderProductLine',
                bind: 'bomCode',
                valueField: { code: true },
                isReadOnly: true,
                isHidden: true,
            }),
        ],
        orderBy: { product: { code: 1 } },

        isFullWidth: true,
        isAutoSelectEnabled: true,
        isHidden: true,
        async onChange() {
            if (this.productsReleased.id != '') {
                if (this.unplannedMaterials.value == false) {
                    this.product.isHidden = false;
                    this.unplannedProduct.isHidden = true;
                    this.product.isMandatory = true;
                    this.unplannedProduct.isMandatory = false;
                    this.product.focus();
                } else {
                    this.product.isHidden = true;
                    this.unplannedProduct.isHidden = false;
                    this.product.isMandatory = false;
                    this.unplannedProduct.isMandatory = true;
                    this.unplannedProduct.focus();
                }
                this.product.value = {};
                this.unplannedProduct.value = {};
            }
        },
    })
    productsReleased: ui.fields.Reference;

    private async _isReleasedIfShortage(workOrderNumber: string): Promise<boolean> {
        const payload = `
        query {
            x3Manufacturing {
                workOrderMaterialLine {
                    query (first: 100, filter: "{ number: '${workOrderNumber}', shortageQuantity: { _ne: 0 } }") {
                        edges {
                            node {
                                number {
                                    number
                                }
                                product {
                                    code
                                    productSites {
                                        query (filter: "{ stockSite: { code: '${this.materialsConsumptionSite.value}' } , isReleasedIfShortage: false }") {
                                            edges {
                                                node {
                                                    stockSite {
                                                        code
                                                    }
                                                    isReleasedIfShortage
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }`;

        const response = await this.$.graph.raw(payload, true);
        let isReleasedIfShortage: boolean = true;
        this._productIsNotReleasedIfShortage = '';

        if (response.x3Manufacturing.workOrderMaterialLine.query.edges.length !== 0) {
            response.x3Manufacturing.workOrderMaterialLine.query.edges.some(
                (workOrderMaterialLine: {
                    node: {
                        number: {
                            number: string;
                        };
                        product: {
                            code: string;
                            productSites: {
                                query: {
                                    edges: [
                                        node: {
                                            stockSite: {
                                                code: string;
                                            };
                                            isReleasedIfShortage: boolean;
                                        },
                                    ];
                                };
                            };
                        };
                    };
                }) => {
                    workOrderMaterialLine.node.product.productSites.query.edges.some(
                        (node: {
                            stockSite: {
                                code: string;
                            };
                            isReleasedIfShortage: boolean;
                        }) => {
                            isReleasedIfShortage = false;
                            this._productIsNotReleasedIfShortage = workOrderMaterialLine.node.product.code;
                        },
                    );
                },
            );
        }
        return isReleasedIfShortage;
    }

    @ui.decorators.referenceField<MobileMaterialsConsumption, WorkOrderMaterialLine>({
        parent() {
            return this.productBlock;
        },
        title: 'Component',
        placeholder: 'Scan or select...',
        node: '@sage/x3-manufacturing/WorkOrderMaterialLine',
        valueField: { product: { code: true } },
        helperTextField: { product: { description1: true } },

        filter() {
            return {
                number: { number: this.workOrder.value?.number },
                product: { code: { _in: stockWithdrawalMode.permittedComponents } },
                lineNumber: { lineNumber: { _in: [this.productsReleased.value.lineNumber] } },
                materialStatus: { _ne: 'cancelled' },
            } as any;
        },
        onError(error: any, originScreenId: string, originElementId: string) {
            ui.console.warn(`Error on ${originScreenId} ${originElementId}: ${error.message || error}`);
        },
        async onChange() {
            if (!this.product.value) return;
            if (!(await validate(this))) {
                this.product.value = null;
                return;
            }

            const response = await this.$.graph
                .node('@sage/x3-master-data/ProductSite')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            isBeingCounted: true,
                            product: {
                                code: true,
                                description1: true,
                            },
                        },
                        {
                            filter: {
                                product: {
                                    code: this.product.value.code,
                                },
                                stockSite: {
                                    code: this.$.storage.get('mobile-selected-stock-site').toString(),
                                    //code: this.materialsConsumptionSite.value,
                                },
                            },
                        },
                    ),
                )
                .execute();

            if (response.edges[0].node.isBeingCounted === true) {
                this.$.dialog
                    .confirmation('warn', 'Warning', 'Product blocked by count. Do you want to continue?', {
                        acceptButton: {
                            text: 'Yes',
                        },
                        cancelButton: {
                            text: 'No',
                        },
                    })
                    .then(response => {
                        this.goToDetailsPage();
                    })
                    .catch(error => {
                        this.product.value = null;
                        this.product.focus();
                    });
            } else {
                this.goToDetailsPage();
            }
        },
        isTransient: true,
        isMandatory: false,
        isFullWidth: true,
        isAutoSelectEnabled: true,
        isHidden: true,
        columns: [
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrderMaterialLine',
                bind: 'product',
                valueField: 'code',
                title: 'Product',
                isReadOnly: true,
                isTitleHidden: false,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrderMaterialLine',
                bind: 'product',
                valueField: 'description1',
                title: 'Description',
                isReadOnly: true,
                isTitleHidden: false,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrderMaterialLine',
                bind: 'lineNumber',
                valueField: 'lineNumber',
                title: 'Line',
                isReadOnly: true,
                isTitleHidden: false,
            }),
            ui.nestedFields.text({
                bind: 'bomOperationNumber',
                title: 'Operation number',
                isReadOnly: true,
                isTitleHidden: false,
                prefix: 'Operation:',
            }),
            ui.nestedFields.text({
                bind: 'requirementDate',
                title: 'Requirement date',
                isReadOnly: true,
                isTitleHidden: false,
                prefix: 'Requirement:',
            }),
            ui.nestedFields.text({
                bind: 'consumedQuantity',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'requiredQuantity',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'bomSequence',
                isHidden: true,
            }),
        ],
    })
    product: ui.fields.Reference;

    @ui.decorators.referenceField<MobileMaterialsConsumption, ProductSite>({
        parent() {
            return this.productBlock;
        },
        title: 'Component',
        placeholder: 'Scan or select...',
        node: '@sage/x3-master-data/ProductSite',
        valueField: { product: { code: true } },
        helperTextField: { product: { description1: true } },
        filter() {
            return {
                stockSite: { code: this.materialsConsumptionSite.value },
            };
        },
        onError(error: any, originScreenId: string, originElementId: string) {
            ui.console.warn(`Error on ${originScreenId} ${originElementId}: ${error.message || error}`);
        },
        async onChange() {
            if (!this.unplannedProduct.value) return; // if user closes the lookup instead of selecting a record
            if (!(await validate(this))) {
                this.unplannedProduct.value = null;
                return;
            }
            const response = await this.$.graph
                .node('@sage/x3-master-data/ProductSite')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            isBeingCounted: true,
                            product: {
                                code: true,
                                description1: true,
                            },
                        },
                        {
                            filter: {
                                product: {
                                    code: this.unplannedProduct.value.code,
                                },
                                stockSite: {
                                    code: this.materialsConsumptionSite.value,
                                },
                            },
                        },
                    ),
                )
                .execute();

            if (response.edges[0].node.isBeingCounted === true) {
                this.$.dialog
                    .confirmation('warn', 'Warning', 'Product blocked by count. Do you want to continue?', {
                        acceptButton: {
                            text: 'Yes',
                        },
                        cancelButton: {
                            text: 'No',
                        },
                    })
                    .then(response => {
                        this.goToDetailsPage();
                    })
                    .catch(error => {
                        this.unplannedProduct.value = null;
                        this.unplannedProduct.focus();
                        return;
                    });
            } else {
                this.goToDetailsPage();
            }
        },
        isTransient: true,
        isMandatory: false,
        isFullWidth: true,
        isAutoSelectEnabled: true,
        isHidden: true,
        columns: [
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/Product',
                bind: 'product',
                valueField: 'code',
                title: 'Product',
                isReadOnly: true,
                isTitleHidden: false,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/Product',
                bind: 'product',
                valueField: 'description1',
                title: 'Description',
                isReadOnly: true,
                isTitleHidden: false,
            }),
        ],
    })
    unplannedProduct: ui.fields.Reference;

    @ui.decorators.tableField<MobileMaterialsConsumption>({
        parent() {
            return this.workOrderBlock;
        },
        title: 'Products',
        isTitleHidden: false,
        isTransient: true,
        canSelect: false,
        canFilter: false,
        columns: [
            ui.nestedFields.text({
                bind: 'workOrder',
                title: 'Work order',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'product',
                title: 'Product',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'description1',
                title: 'Description',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'unit',
                title: 'unit',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'quantity',
                title: 'quantity',
                isReadOnly: true,
            }),
        ],
        dropdownActions: [
            {
                icon: 'delete',
                title: 'Delete',
                onClick(rowId: any, data: any) {
                    this._materialsConsumptionSession.detail.splice(rowId, 1);
                    if (this._materialsConsumptionSession.detail.length === 0) {
                        this._materialsConsumptionRecordManager.clearSession();
                        this.$.router.goTo(`@sage/x3-manufacturing/${this.$.page.id}`, {
                            returnFromDetailsPage: 'false',
                        });
                        this.createButton.isDisabled = true;
                    } else {
                        this.materialsConsumptionToCreate.value = this._mapMaterialsConsumptionDetails(
                            this._materialsConsumptionSession.detail,
                        );
                        this.materialsConsumptionToCreate.title = `Products ${this._materialsConsumptionSession.detail.length.toString()}`;

                        this._materialsConsumptionRecordManager.materialsConsumptionSession = {
                            username: this.$.userCode,
                            workOrder: this._materialsConsumptionSession.workOrder,
                            product: this._materialsConsumptionSession.product,
                            transaction: this._materialsConsumptionSession.transaction,
                            unplannedMaterials: this._materialsConsumptionSession.unplannedMaterials,
                            productsReleased: this._materialsConsumptionSession.productsReleased,
                            detail: this._materialsConsumptionSession.detail,
                            productCode: this._materialsConsumptionSession.productCode,
                            productDescription: this._materialsConsumptionSession.productDescription,
                            trackingDate: this._materialsConsumptionSession.trackingDate,
                            materialsConsumptionSite: this.materialsConsumptionSite.value,
                            numberProductsReleased: WorkNumberOfProductsReleased,
                        };
                    }
                },
            },
        ],
        mobileCard: {
            title: ui.nestedFields.text({
                bind: 'product',
            }),
            line2: ui.nestedFields.text({
                bind: 'description1',
            }),
            titleRight: ui.nestedFields.text({
                bind: 'quantityAndUnit',
            }),
            line2Right: ui.nestedFields.text({
                bind: 'workOrder',
            }),
        },
    })
    materialsConsumptionToCreate: ui.fields.Table<any>;

    async initMaterialsConsumptionPage() {
        await this._initSite();
        await this.getListTransaction();
        await this.getMaterialsConsumptionSession();
        await this.getGeneralParameters();
        if (this.materialsConsumptionSite.value) {
            this.legalCompany.value = await this._getLegalCompany(this.materialsConsumptionSite.value);
        }
    }

    private async _initSite(): Promise<void> {
        this.materialsConsumptionSite.value = await getSelectedStockSite(this);
    }

    private async getListTransaction() {
        const response = await this.$.graph
            .node('@sage/x3-manufacturing/TransactionProductionReporting')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        transaction: true,
                        unplannedMaterials: true,
                    },
                    {
                        filter: {
                            type: 61,
                            isAutomaticIssue: false,
                            isActive: true,
                        },
                    },
                ),
            )
            .execute();

        if (response.edges.length !== 0) {
            let transactions: string[] = [];
            let unplannedMaterialsFlags: boolean[] = [];
            response.edges.some(edge => {
                transactions.push(edge.node.transaction);
            });
            response.edges.some(edge => {
                unplannedMaterialsFlags.push(edge.node.unplannedMaterials);
            });
            this.transaction.options = transactions;
            WorkUnplannedMaterials = unplannedMaterialsFlags;

            if (WorkUnplannedMaterials[this.transaction.options.indexOf(transactions[0])] === true) {
                this.unplannedMaterials.isHidden = false;
            } else {
                this.unplannedMaterials.isHidden = true;
            }

            transactions.length == 1 ? (this.transaction.isHidden = true) : (this.transaction.isHidden = false);
        }
    }

    private async getGeneralParameters() {
        this._gMtkAllMgt = await this.getParamValue(this.materialsConsumptionSite.value, 'MTKALLMGT');
        this._gDosFab = await this.getParamValue(this.materialsConsumptionSite.value, 'DOSFAB');
    }

    private async getMaterialsConsumptionSession() {
        this._materialsConsumptionRecordManager = new MaterialsConsumptionDetailsRecordManager(this, true);
        this._materialsConsumptionSession = this._materialsConsumptionRecordManager.materialsConsumptionSession;

        if (!this._materialsConsumptionSession) {
            this.trackingDate.value = DateValue.today().toString();
            if (this.transaction.options.length !== 0) {
                this.transaction.value = this.transaction.options[0];
            }

            try {
            } catch (error) {
                this.$.showToast(error.message, { type: 'error' });
                return;
            }
        } else {
            this.materialsConsumptionSite.value = this.$.storage.get('mobile-selected-stock-site').toString();
            this.transaction.value = this._materialsConsumptionSession.transaction;
            this.unplannedMaterials.value = this._materialsConsumptionSession.unplannedMaterials;
            if (this.transaction.value) {
                if (WorkUnplannedMaterials[this.transaction.options.indexOf(this.transaction.value)] === true) {
                    this.unplannedMaterials.isHidden = false;
                } else {
                    this.unplannedMaterials.isHidden = true;
                }
            }
            if (this.unplannedMaterials.value == false) {
                this.product.value = null;
            } else {
                this.unplannedProduct.value = null;
            }

            this.productsReleased.value = this._materialsConsumptionSession.productsReleased;

            this.workOrder.value = this._materialsConsumptionSession.workOrder;

            stockWithdrawalMode.permittedComponents = await this.getComponents(
                this._materialsConsumptionSession.workOrder.number,
                this.materialsConsumptionSite.value,
                stockWithdrawalMode.permittedProductModes,
            );

            WorkNumberOfProductsReleased = this._materialsConsumptionSession.numberProductsReleased;

            this.workOrder.isDirty = !this.workOrder.isDirty;
            this.trackingDate.value = this._materialsConsumptionSession.trackingDate;

            this.materialsConsumptionToCreate.value = this._mapMaterialsConsumptionDetails(
                this._materialsConsumptionSession.detail,
            );
            this.materialsConsumptionToCreate.title = `Products ${this._materialsConsumptionSession.detail.length.toString()}`;
            this.workOrderBlock.isHidden = false;
            this.workOrder.isDisabled = false;
            this.transaction.isDisabled = false;
            this.createButton.isDisabled = false;

            if (WorkNumberOfProductsReleased === 1) {
                this.productsReleased.isHidden = true;
                this.productsReleased.isMandatory = false;
            } else {
                this.productsReleased.isHidden = false;
                this.productsReleased.isMandatory = true;
            }

            if (this.unplannedMaterials.value == false) {
                this.product.isHidden = false;
                this.unplannedProduct.isHidden = true;
                this.product.isMandatory = true;
                this.unplannedProduct.isMandatory = false;
                this.product.focus();
            } else {
                this.product.isHidden = true;
                this.unplannedProduct.isHidden = false;
                this.product.isMandatory = false;
                this.unplannedProduct.isMandatory = true;
                this.unplannedProduct.focus();
            }
            this.transaction.isDisabled = true;
        }
    }

    private goToDetailsPage() {
        if (!this._materialsConsumptionSession) {
            if (this.unplannedMaterials.value == false) {
                this._materialsConsumptionRecordManager.materialsConsumptionSession = {
                    username: this.$.userCode,
                    workOrder: this.workOrder.value,
                    product: this.product.value,
                    transaction: this.transaction.value,
                    unplannedMaterials: this.unplannedMaterials.value,
                    productsReleased: this.productsReleased.value,
                    detail: [],
                    productDescription: this.product.value.product.description1,
                    productCode: this.product.value.product.code,
                    trackingDate: this.trackingDate.value,
                    materialsConsumptionSite: this.materialsConsumptionSite.value,
                    numberProductsReleased: WorkNumberOfProductsReleased,
                };
            } else {
                this._materialsConsumptionRecordManager.materialsConsumptionSession = {
                    username: this.$.userCode,
                    workOrder: this.workOrder.value,
                    product: this.unplannedProduct.value,
                    transaction: this.transaction.value,
                    unplannedMaterials: this.unplannedMaterials.value,
                    productsReleased: this.productsReleased.value,
                    detail: [],
                    productDescription: this.unplannedProduct.value.product.description1,
                    productCode: this.unplannedProduct.value.product.code,
                    trackingDate: this.trackingDate.value,
                    materialsConsumptionSite: this.materialsConsumptionSite.value,
                    numberProductsReleased: WorkNumberOfProductsReleased,
                };
            }
        } else {
            if (this.unplannedMaterials.value == false) {
                this._materialsConsumptionRecordManager.materialsConsumptionSession = {
                    username: this.$.userCode,
                    workOrder: this.workOrder.value,
                    product: this.product.value,
                    transaction: this.transaction.value,
                    unplannedMaterials: this.unplannedMaterials.value,
                    productsReleased: this.productsReleased.value,
                    detail: this._materialsConsumptionSession.detail,
                    productDescription: this.product.value.product.description1,
                    productCode: this.product.value.product.code,
                    trackingDate: this.trackingDate.value,
                    materialsConsumptionSite: this.materialsConsumptionSite.value,
                    numberProductsReleased: WorkNumberOfProductsReleased,
                };
            } else {
                this._materialsConsumptionRecordManager.materialsConsumptionSession = {
                    username: this.$.userCode,
                    workOrder: this.workOrder.value,
                    product: this.unplannedProduct.value,
                    transaction: this.transaction.value,
                    unplannedMaterials: this.unplannedMaterials.value,
                    productsReleased: this.productsReleased.value,
                    detail: this._materialsConsumptionSession.detail,
                    productDescription: this.unplannedProduct.value.product.description1,
                    productCode: this.unplannedProduct.value.product.code,
                    trackingDate: this.trackingDate.value,
                    materialsConsumptionSite: this.materialsConsumptionSite.value,
                    numberProductsReleased: WorkNumberOfProductsReleased,
                };
            }
        }

        this.workCurrentUnplannedMaterials.value = this.unplannedMaterials.value;

        this.$.setPageClean();
        this.$.router.goTo('@sage/x3-manufacturing/MobileMaterialsConsumptionDetails');
    }

    private async _getLegalCompany(defaultSite: string): Promise<string> {
        const response = await this.$.graph
            .node('@sage/x3-system/Site')
            .read(
                {
                    _id: true,
                    legalCompany: { code: true },
                },
                defaultSite,
            )
            .execute();

        if (!response) {
            throw new Error(
                ui.localize(
                    '@sage/x3-manufacturing/error-materials-consumption-no-legal-company',
                    'The invalid site {{site}} has no legal company.',
                    {
                        site: defaultSite,
                    },
                ),
            );
        }

        return response.legalCompany.code;
    }
    async callCreationAPI(): Promise<X3Response> {
        let result: X3Response;
        const payload = `
        mutation {
            x3Manufacturing {
                workOrderMaterialTracking {
                    create(data: ${ui.queryUtils.serializeToGraphQL(this.storageObject.materialTracking)}){
                        trackingNumber
                    }
                }
            }
        }
    `;
        try {
            result = await this.$.graph.raw(payload, true);
            if (!result) {
                throw Error(
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_materials_consumption___no_results',
                        'No results received for the creation',
                    ),
                );
            }
        } catch (error) {
            return error;
        }
        return result;
    }

    private _mapMaterialsConsumptionDetails(materialsConsumptionDetails: materialsConsumptionDetail[]) {
        let rowCount = 0;
        return materialsConsumptionDetails.map((line: materialsConsumptionDetail) => {
            return {
                _id: String(rowCount++),
                workOrder: line.workOrder,
                unit: line.unit,
                quantity: line.quantity,
                product: line.product,
                description1: line.description,
                quantityAndUnit: `${line.quantity} ${line.unit}`,
            };
        });
    }

    async calcProductsReleased() {
        const response = await this.$.graph
            .node('@sage/x3-manufacturing/WorkOrderProductLine')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        _id: true,
                        workOrder: { number: true },
                        product: { code: true },
                        lineNumber: true,
                        productType: true,
                        bomCode: { code: true },
                    },
                    {
                        filter: {
                            workOrder: { number: this.workOrder.value.number },
                            productType: { _eq: 'product' },
                        },
                    },
                ),
            )
            .execute();
        WorkNumberOfProductsReleased = response.edges.length;
        if (WorkNumberOfProductsReleased === 1) {
            // initialize object if filled
            if (this.productsReleased.value) {
                this.productsReleased.value = {};
            }
            // assign new product
            this.productsReleased.value = response.edges[0].node;
            // field properties
            this.productsReleased.isHidden = true;
            this.productsReleased.isMandatory = false;
        } else {
            // field initialize
            this.productsReleased.value = {};
            // field properties
            this.productsReleased.isHidden = false;
            this.productsReleased.isMandatory = true;
        }
    }

    private storageObject: StorageObjectMaterial;

    prepareCreation(): void {
        this.initStorageObjectMaterial();
        if (this.materialsConsumptionSite.value !== null) {
            this.storageObject.materialTracking.productionSite = this.materialsConsumptionSite.value;
        }

        this._materialsConsumptionSession.detail.forEach((line, index) => {
            const bomAlternativeA: any = { ba: line.bomAlternative };
            const bomAlternativeB: number = bomAlternativeA.ba.code;
            // adjust boolean value from checkbox to yes/no values needed for import script
            let closeWoLineA: number = 1; // No
            if (line.closeWoLine !== null) {
                if (line.closeWoLine === true) {
                    closeWoLineA = 2; // Yes
                }
            }
            let materialLine: any = {
                workOrder: line.workOrder,
                productionSite: this.materialsConsumptionSite.value,
                lineNumber: line.lineNumber,
                bomSequence: line.bomSequence,
                product: line.product,
                bomNumber: line.bomNumber.code,
                bomAlternative: bomAlternativeB,
                consumedQuantity: line.quantity,
                postingDate: line.trackingDate,
                transaction: this.transaction.value,
                balance: closeWoLineA,
                mfgTrackingNumber: '',
                preferredLot: '',
                project: '',
                stockJournal: [
                    {
                        lot: line.lot.code,
                        sublot: line.sublot,
                        location: line.location.code,
                        stockUnit: line.unit,
                        quantityInStockUnit: line.quantity,
                        status: line.status,
                        serialNumber: line.serialNumber.code,
                        licensePlateNumber: line.licensePlateNumber.code,
                        identifier1: '',
                        identifier2: '',
                    },
                ],
            };
            this.insertMaterialLine(materialLine);
        });
    }

    insertMaterialLine(materialLine: MaterialTrackingLine) {
        this.storageObject.materialTracking.workOrderMaterialTrackingLines.push(materialLine);
    }

    initStorageObjectMaterial() {
        this.storageObject = {
            materialTracking: {
                productionSite: new String(),
                workOrderMaterialTrackingLines: new Array<MaterialTrackingLine>(),
            },
        } as StorageObjectMaterial;
    }

    sortCreation(): void {
        ui.console.log('Current before sorting', this.storageObject.materialTracking.workOrderMaterialTrackingLines);
        this.storageObject.materialTracking.workOrderMaterialTrackingLines.sort(function (a, b) {
            let result: number = 0;
            if (result === 0) {
                result =
                    a.workOrder.toString() < b.workOrder.toString()
                        ? -1
                        : a.workOrder.toString() > b.workOrder.toString()
                          ? 1
                          : 0;
            }
            if (result === 0) {
                result =
                    a.productionSite.toString() < b.productionSite.toString()
                        ? -1
                        : a.productionSite.toString() > b.productionSite.toString()
                          ? 1
                          : 0;
            }
            if (result === 0) {
                result =
                    a.lineNumber.toString() < b.lineNumber.toString()
                        ? -1
                        : a.lineNumber.toString() > b.lineNumber.toString()
                          ? 1
                          : 0;
            }
            if (result === 0) {
                result =
                    a.bomSequence.toString() < b.bomSequence.toString()
                        ? -1
                        : a.bomSequence.toString() > b.bomSequence.toString()
                          ? 1
                          : 0;
            }
            if (result === 0) {
                result =
                    a.product.toString() < b.product.toString()
                        ? -1
                        : a.product.toString() > b.product.toString()
                          ? 1
                          : 0;
            }
            if (result === 0) {
                result =
                    a.bomNumber.toString() < b.bomNumber.toString()
                        ? -1
                        : a.bomNumber.toString() > b.bomNumber.toString()
                          ? 1
                          : 0;
            }
            if (result === 0) {
                result =
                    a.bomAlternative.toString() < b.bomAlternative.toString()
                        ? -1
                        : a.bomAlternative.toString() > b.bomAlternative.toString()
                          ? 1
                          : 0;
            }
            if (result === 0) {
                result =
                    a.mfgTrackingNumber.toString() < b.mfgTrackingNumber.toString()
                        ? -1
                        : a.mfgTrackingNumber.toString() > b.mfgTrackingNumber.toString()
                          ? 1
                          : 0;
            }
            if (result === 0) {
                result =
                    a.preferredLot.toString() < b.preferredLot.toString()
                        ? -1
                        : a.preferredLot.toString() > b.preferredLot.toString()
                          ? 1
                          : 0;
            }
            if (result === 0) {
                result =
                    a.project.toString() < b.project.toString()
                        ? -1
                        : a.project.toString() > b.project.toString()
                          ? 1
                          : 0;
            }

            return result;
        });
        ui.console.log('Current after sorting', this.storageObject.materialTracking.workOrderMaterialTrackingLines);
    }

    calculateQuantityCreation(): void {
        let lines: number = 0;
        let keyCumulatedConsumedQuantity: number = 0;
        let processedLineArray = new Array(0).fill(' ');
        let workCounter: number = 0;
        for (lines = 0; lines < this.storageObject.materialTracking.workOrderMaterialTrackingLines.length; lines++) {
            // process record only if not already processed
            if (!processedLineArray.includes(lines)) {
                // save key criteria
                let keyObject: keyMaterialTrackingLine = {
                    workOrder: this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].workOrder,
                    productionSite:
                        this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].productionSite,
                    lineNumber: this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].lineNumber,
                    bomSequence: this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].bomSequence,
                    product: this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].product,
                    bomNumber: this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].bomNumber,
                    bomAlternative:
                        this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].bomAlternative,
                    mfgTrackingNumber:
                        this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].mfgTrackingNumber,
                    preferredLot:
                        this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].stockJournal[0].lot,
                    project: this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].project,
                };

                // go through the array, see if records are equal and cumulate quantity
                // if there is an equal record then save its line number in the array
                // count the equal records in order to know for the next step if a quantity needs to be set for all
                // the equal records of this key
                keyCumulatedConsumedQuantity = 0;
                workCounter = 0;
                for (
                    let noLines: number = lines + 1;
                    noLines < this.storageObject.materialTracking.workOrderMaterialTrackingLines.length;
                    noLines++
                ) {
                    if (
                        keyObject.workOrder ==
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines].workOrder &&
                        keyObject.productionSite ==
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines]
                                .productionSite &&
                        keyObject.lineNumber ==
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines].lineNumber &&
                        keyObject.bomSequence ==
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines].bomSequence &&
                        keyObject.product ==
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines].product &&
                        keyObject.bomNumber ==
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines].bomNumber &&
                        keyObject.bomAlternative ==
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines]
                                .bomAlternative &&
                        keyObject.mfgTrackingNumber ==
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines]
                                .mfgTrackingNumber &&
                        keyObject.preferredLot ==
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines].preferredLot &&
                        keyObject.preferredLot ==
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines].project
                    ) {
                        keyCumulatedConsumedQuantity +=
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[noLines]
                                .consumedQuantity;
                        processedLineArray[processedLineArray.length] = noLines;
                        workCounter += 1;
                    } else {
                        break;
                    }
                }

                // at least one equal record found. Add base record to the array of already processed lines
                // add quantity to total quantity for this key
                if (workCounter > 0) {
                    processedLineArray[processedLineArray.length] = lines;
                    keyCumulatedConsumedQuantity +=
                        this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].consumedQuantity;

                    // assign total quantity to each of the equal records
                    // firstly to the base record
                    this.storageObject.materialTracking.workOrderMaterialTrackingLines[lines].consumedQuantity =
                        keyCumulatedConsumedQuantity;
                    for (
                        let qtyLines = lines + 1;
                        qtyLines < this.storageObject.materialTracking.workOrderMaterialTrackingLines.length;
                        qtyLines++
                    ) {
                        if (
                            keyObject.workOrder ==
                                this.storageObject.materialTracking.workOrderMaterialTrackingLines[qtyLines]
                                    .workOrder &&
                            keyObject.productionSite ==
                                this.storageObject.materialTracking.workOrderMaterialTrackingLines[qtyLines]
                                    .productionSite &&
                            keyObject.lineNumber ==
                                this.storageObject.materialTracking.workOrderMaterialTrackingLines[qtyLines]
                                    .lineNumber &&
                            keyObject.bomSequence ==
                                this.storageObject.materialTracking.workOrderMaterialTrackingLines[qtyLines]
                                    .bomSequence &&
                            keyObject.product ==
                                this.storageObject.materialTracking.workOrderMaterialTrackingLines[qtyLines].product &&
                            keyObject.bomNumber ==
                                this.storageObject.materialTracking.workOrderMaterialTrackingLines[qtyLines]
                                    .bomNumber &&
                            keyObject.bomAlternative ==
                                this.storageObject.materialTracking.workOrderMaterialTrackingLines[qtyLines]
                                    .bomAlternative &&
                            keyObject.mfgTrackingNumber ==
                                this.storageObject.materialTracking.workOrderMaterialTrackingLines[qtyLines]
                                    .mfgTrackingNumber &&
                            keyObject.preferredLot ==
                                this.storageObject.materialTracking.workOrderMaterialTrackingLines[qtyLines]
                                    .preferredLot &&
                            keyObject.preferredLot ==
                                this.storageObject.materialTracking.workOrderMaterialTrackingLines[qtyLines].project
                        ) {
                            this.storageObject.materialTracking.workOrderMaterialTrackingLines[
                                qtyLines
                            ].consumedQuantity = keyCumulatedConsumedQuantity;
                        } else {
                            break;
                        }
                    }
                }
            }
        }
        ui.console.log('Current after Cumulation', this.storageObject.materialTracking.workOrderMaterialTrackingLines);
    }

    /**
     * Get the stock withdrawal mode to be used when selecting componenets
     * @param type Entry Tranaction type
     * @param code Entry Tranaction Code
     * @returns string mode to be used
     */
    async getStockWithdrawalMode(type: number, code: string): Promise<string> {
        try {
            const data = await this.$.graph
                .node('@sage/x3-manufacturing/TransactionProductionReporting')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            stockWithdrawalMode: true,
                        },
                        {
                            filter: {
                                transaction: code,
                                type: type,
                            },
                        },
                    ),
                )
                .execute();

            stockWithdrawalMode.entryTransactionMode = data.edges[0].node.stockWithdrawalMode;
        } catch (error) {
            stockWithdrawalMode.entryTransactionMode = undefined;
        }

        if (
            stockWithdrawalMode.entryTransactionMode === 'all' ||
            stockWithdrawalMode.entryTransactionMode === undefined
        ) {
            return 'notManaged';
        } else {
            return stockWithdrawalMode.entryTransactionMode;
        }
    }

    /**
     * Get all the product codes for a given work order / product site
     *  with a selected stock withdrawal mode
     * @param workOrder
     * @param Site
     * @param stockWithdrawalMode
     * @returns componentList an aArray of product codes
     */
    async getComponents(
        workOrder: string | null,
        site: string | null,
        withdrawalMode: string | null,
    ): Promise<string[]> {
        const componentList: string[] = [];
        let payload = '';

        if (withdrawalMode !== 'notManaged') {
            payload = `
             query {
                x3Manufacturing {
                    workOrderMaterialLine {
                        query (first: 100, filter: "{ number: '${workOrder}', componentType: 'normal'}") {
                            edges {
                                node {
                                    product {
                                        productSites {
                                          query (filter: "{stockSite: { code: '${site}'},  stockWithdrawalMode: { _in: ['${withdrawalMode}']} }")
                                              {
                                                edges {
                                                    node {
                                                    stockWithdrawalMode
                                                     product {
                                                        code

                                                      }
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
           `;
        } else {
            payload = `
             query {
                x3Manufacturing {
                    workOrderMaterialLine {
                        query (first: 100, filter: "{ number: '${workOrder}', componentType: 'normal'}") {
                            edges {
                                node {
                                    product {
                                        productSites {
                                          query (filter: "{stockSite: { code: '${site}'} }")
                                              {
                                                edges {
                                                    node {
                                                    stockWithdrawalMode
                                                     product {
                                                        code

                                                      }
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
           `;
        }

        const response = await this.$.graph.raw(payload, true);
        const jsonObject: any[] = JSON.parse(
            JSON.stringify(response.x3Manufacturing.workOrderMaterialLine.query.edges),
        );
        jsonObject.forEach((item: any) => {
            const code = item.node.product.productSites.query.edges[0]?.node.product.code;
            if (code) {
                componentList.push(code);
            }
        });
        return componentList;
    }

    // TODO NEED REFACTOR AND MOVING TO CLIENT FUNCTION - PARAMETER READ LOGIN NEED IMPROVING
    // TODO ADD ERROR MESSAGE WAITING UNTIL TRANSLATION UTIL HAS BEEN RUN
    /**
     * Read Parameter Value
     *
     * @param parameterName
     * @param site
     * @returns
     */
    async getParamValue(site: string | null, parameterName: string): Promise<string> {
        let payload = '';
        payload = `
          mutation {
              x3System {
               generalParametersData {
                readParameterValueString(
                parameters: {company: "", site: "${site}", code: "${parameterName}"})
                {
                value
                }
                }}}
           `;
        try {
            const response = await this.$.graph.raw(payload, true);
            return response.x3System.generalParametersData.readParameterValueString.value;
        } catch (error) {
            return '';
        }
    }
}
