import {
    GraphApi,
    WorkOrderInput,
    WorkOrderMaterialLineInput,
    WorkOrderProductLineInput,
} from '@sage/x3-manufacturing-api';
import { integer } from '@sage/xtrem-client';
import { Page, localize } from '@sage/xtrem-ui';
export interface materialsConsumptionDetail {
    workOrder: string;
    product: string;
    description: string;
    unit: string;
    quantity: number;
    status: string;
    location: { code: string };
    lot: { code: string };
    sublot: string;
    serialNumber: { code: string };
    licensePlateNumber: { code: string; container: { code: string } };
    closeWoLine: boolean;
    lineNumber: integer;
    bomSequence: integer;
    bomNumber: { code: string; description1: string };
    bomAlternative: integer;
    trackingDate: string;
}

export async function validate(page: Page): Promise<boolean> {
    const errors = await page.$.page.validate();
    if (errors.length === 0) {
        return true;
    }

    page.$.showToast(
        `${localize('@sage/x3-manufacturing/notification-error-materials-consumption-error', 'Error')}: ${errors[0]}`,
        { type: 'error', timeout: 30000 },
    );

    return false;
}

export type MaterialsConsumptionSession = {
    username: string;
    workOrder: WorkOrderInput;
    product: WorkOrderMaterialLineInput;
    transaction: string;
    unplannedMaterials: boolean;
    productsReleased: WorkOrderProductLineInput;
    detail: materialsConsumptionDetail[];
    productCode: string;
    productDescription: string;
    trackingDate: string;
    materialsConsumptionSite: string;
    numberProductsReleased: number;
};

export class MaterialsConsumptionDetailsRecordManager {
    constructor(
        private page: Page<GraphApi>,
        cleanupIncompleteSessionLines = false,
    ) {
        if (cleanupIncompleteSessionLines) this._cleanupSession();
    }

    private static readonly WORK_ORDER_KEY: string = 'workOrder';

    private _materialsConsumptionSession: MaterialsConsumptionSession | null;

    public get materialsConsumptionSession(): MaterialsConsumptionSession | null {
        if (this._materialsConsumptionSession) {
            return this._materialsConsumptionSession;
        }

        const storedSession = this.page.$.storage.get(
            MaterialsConsumptionDetailsRecordManager.WORK_ORDER_KEY,
        ) as string;
        if (storedSession) {
            const parsedSession = JSON.parse(storedSession) as MaterialsConsumptionSession;
            if (this._checkStorage(parsedSession)) {
                this._materialsConsumptionSession = parsedSession;
            } else {
                this.clearSession();
            }
        }
        return this._materialsConsumptionSession;
    }

    public set materialsConsumptionSession(materialsConsumptionSession: MaterialsConsumptionSession | null) {
        if (!materialsConsumptionSession) {
            return;
        }

        if (!this._checkStorage(materialsConsumptionSession)) {
            throw new Error('Invalid Materials Consumption session');
        }

        this.page.$.storage.set(
            MaterialsConsumptionDetailsRecordManager.WORK_ORDER_KEY,
            JSON.stringify(materialsConsumptionSession),
        );

        this._materialsConsumptionSession = materialsConsumptionSession;
    }

    private _checkStorage(sessionData: MaterialsConsumptionSession): boolean {
        if (!sessionData.transaction || !sessionData.workOrder.startDate || !sessionData.workOrder.productionSite) {
            return false;
        } else {
            return true;
        }
    }

    private _cleanupSession() {
        let returnFromDetailsPage = this.page.$.queryParameters['returnFromDetailsPage'] as string;
        this._materialsConsumptionSession = this.materialsConsumptionSession; // get the latest from session
        if (!this._materialsConsumptionSession) return;

        // if all lines have been removed, clear the session
        if (this._materialsConsumptionSession.detail.length === 0 || returnFromDetailsPage != 'true') {
            this.clearSession();
        } else {
            this.materialsConsumptionSession = this._materialsConsumptionSession;
        }
    }

    public clearSession(): void {
        this.page.$.storage.remove(MaterialsConsumptionDetailsRecordManager.WORK_ORDER_KEY);
        this._materialsConsumptionSession = null;
    }

    public loadMaterialsConsumptionDetails(
        page: {
            product: { value: string };
            description: { value: string };
            unit: { value: string };
            quantity: { value: number };
            status: { value: string };
            location: { value: string };
            lot: { value: { lot?: string } };
            sublot: { value: string };
            serialNumber: { value: { code?: string } };
            closeWoLine: { value: boolean };
            licensePlateNumber: { value: string };
            lineNumber: { value: integer };
            bomSequence: { value: integer };
            bomNumber: { value: string };
            bomAlternative: { value: integer };
            trackingDate: { value: string };
        },
        addNewDetail: boolean = false,
    ) {
        const detailLine: any = {
            workOrder: this._materialsConsumptionSession?.workOrder.number,
            product: this._materialsConsumptionSession?.productCode,
            description: this._materialsConsumptionSession?.productDescription,
            unit: page.unit.value ?? undefined,
            quantity: page.quantity.value ?? undefined,
            status: page.status.value ?? undefined, // this is always required
            lot: { code: page.lot.value?.lot ?? undefined }, // using undefined, omits the property entirely from JSON when stringified
            sublot: page.sublot.value ?? undefined,
            location: { code: page.location.value ?? undefined },
            serialNumber: { code: page.serialNumber.value?.code ?? undefined },
            licensePlateNumber: { code: page.licensePlateNumber.value ?? undefined },
            closeWoLine: page.closeWoLine.value ?? undefined,
            lineNumber: this._materialsConsumptionSession?.productsReleased.lineNumber,
            bomSequence: this._materialsConsumptionSession?.product.bomSequence,
            bomNumber: this._materialsConsumptionSession?.productsReleased.product,
            bomAlternative: this._materialsConsumptionSession?.productsReleased.bomCode,
            trackingDate: this._materialsConsumptionSession?.trackingDate,
        };

        // unplanned material has no bomSequence
        if (typeof detailLine.bomSequence === 'undefined') {
            detailLine.bomSequence = 0;
        }
        this._materialsConsumptionSession?.detail.push(detailLine);
        this.materialsConsumptionSession = this._materialsConsumptionSession;

        if (addNewDetail) {
            let TotalConsumedQTY: number = 0;
            if (this.materialsConsumptionSession?.detail && this.materialsConsumptionSession.detail.length > 0) {
                this.materialsConsumptionSession.detail.forEach(line => {
                    TotalConsumedQTY += Number(line.quantity);
                });
            }

            let remainingQuantity =
                Number(this.materialsConsumptionSession?.product.requiredQuantity) -
                Number(this._materialsConsumptionSession?.product.consumedQuantity);
            if (Number(remainingQuantity - TotalConsumedQTY) <= 0) {
                page.quantity.value = 0;
            } else {
                page.quantity.value = Number(remainingQuantity - TotalConsumedQTY);
            }

            page.status.value = '';
            page.location.value = '';
            page.lot.value.lot = '';
            page.sublot.value = '';
            page.licensePlateNumber.value = '';
            page.serialNumber.value.code = '';
        }

        console.log('🚀 ~ MaterialsConsumptionDetailsRecordManager ~ detailLine:', detailLine);
    }
}
