"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCrudBulkMutationNames = exports.isNodeDecorator = exports.saveAsJson = exports.parseObjectLiteralToObject = exports.enumNameToStringKey = exports.artifactReferenceToTitleStringKey = exports.menuItemIdToStringKey = exports.getLocaleFromHeader = exports.getDirName = exports.getFileName = exports.getPackageNameAndRoot = void 0;
const xtrem_log_1 = require("@sage/xtrem-log");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const parser = require("accept-language-parser");
const fs = require("fs");
const lodash_1 = require("lodash");
const path = require("path");
const ts = require("typescript");
const logger = new xtrem_log_1.Logger(__filename, 'utils');
/**
 * Gets the name of the current package
 * @param dir the starting directory
 * @returns the name of the package (closest ancestor)
 */
function getPackageNameAndRoot(start) {
    const isExisting = fs.existsSync(start);
    if (!isExisting) {
        throw new Error(`'${start}' doesn't exist`);
    }
    const root = path.parse(start).root;
    let dir = start;
    if (fs.lstatSync(start).isFile()) {
        dir = path.dirname(start);
    }
    const packageJson = path.join(dir, 'package.json');
    if (!fs.existsSync(packageJson)) {
        const newDir = path.join(dir, '..');
        if (newDir === root) {
            throw new Error("Couldn't find the name of the package");
        }
        return getPackageNameAndRoot(newDir);
    }
    try {
        const name = JSON.parse(fs.readFileSync(packageJson, { encoding: 'utf8' })).name;
        return { name, root: dir };
    }
    catch (error) {
        throw new Error(`Couldn't read the 'name' property from '${packageJson}'`);
    }
}
exports.getPackageNameAndRoot = getPackageNameAndRoot;
/**
 * Gets the base name of a file
 * @param file Typescript source file
 * @returns base name of the given file without any extension
 */
function getFileName(file) {
    return path.basename(file.fileName, '.ts');
}
exports.getFileName = getFileName;
/**
 * Gets the directory a file belongs to
 * @param file Typescript source file
 * @param relativeToLib Should we get directory relative to the lib folder in the path
 *                      or do we get that last directory in the path. Default to false
 * @returns name of th directory the given file belongs to
 */
function getDirName(file, relativeToLib = false) {
    const pathElements = path.normalize(path.dirname(file.fileName)).split(path.sep);
    const indexOfLib = pathElements.indexOf('lib');
    if (relativeToLib && indexOfLib >= 0 && indexOfLib < pathElements.length - 1) {
        return pathElements[indexOfLib + 1];
    }
    return pathElements.pop();
}
exports.getDirName = getDirName;
const getLocaleFromHeader = (headers) => {
    if (headers) {
        const acceptLanguage = headers['accept-language'];
        if (acceptLanguage && typeof acceptLanguage === 'string') {
            const language = parser.pick(xtrem_shared_1.supportedLocales, acceptLanguage);
            return language || 'base';
        }
    }
    else {
        logger.debug(() => 'Request headers missing');
    }
    return 'base';
};
exports.getLocaleFromHeader = getLocaleFromHeader;
const menuItemIdToStringKey = (menuItemId) => {
    const parts = menuItemId.split('/');
    const packageName = `${parts[0]}/${parts[1]}`;
    return `${packageName}/menu_item__${menuItemId.replace(`${packageName}/`, '')}`;
};
exports.menuItemIdToStringKey = menuItemIdToStringKey;
const artifactReferenceToTitleStringKey = (artifactReference, artifactType) => {
    const parts = artifactReference.split('/');
    const packageName = `${parts[0]}/${parts[1]}`;
    return `${packageName}/${artifactType}__${(0, lodash_1.snakeCase)(parts[2])}____title`;
};
exports.artifactReferenceToTitleStringKey = artifactReferenceToTitleStringKey;
const enumNameToStringKey = (enumName, memberName) => {
    const parts = enumName.split('/');
    return `${parts[0]}/${parts[1]}/enums__${(0, lodash_1.snakeCase)(parts[2])}__${memberName}`;
};
exports.enumNameToStringKey = enumNameToStringKey;
const parseObjectLiteralToObject = (objectLiteral) => {
    const targetObject = {};
    objectLiteral.properties.forEach(p => {
        if (ts.isPropertyAssignment(p) && p.name && p.initializer) {
            const name = p.name.getText();
            if (ts.isStringLiteral(p.initializer)) {
                targetObject[name] = (0, lodash_1.trim)(p.initializer.getText(), '"\'');
            }
            else if (ts.isNumericLiteral(p.initializer)) {
                targetObject[name] = Number(p.initializer.getText());
            }
            else if (p.initializer.kind === ts.SyntaxKind.TrueKeyword) {
                targetObject[name] = true;
            }
            else if (p.initializer.kind === ts.SyntaxKind.FalseKeyword) {
                targetObject[name] = false;
            }
            else if (ts.isObjectLiteralExpression(p.initializer)) {
                targetObject[name] = (0, exports.parseObjectLiteralToObject)(p.initializer);
            }
        }
    });
    return targetObject;
};
exports.parseObjectLiteralToObject = parseObjectLiteralToObject;
function saveAsJson(filename, obj) {
    fs.writeFileSync(filename, `${JSON.stringify(obj, null, 4)}\n`, 'utf-8');
}
exports.saveAsJson = saveAsJson;
function isNodeDecorator(node) {
    return !!(ts.isDecorator(node) &&
        ts.isCallExpression(node.expression) &&
        ts.isPropertyAccessExpression(node.expression.expression) &&
        node.expression.expression.getText() === 'decorators.node');
}
exports.isNodeDecorator = isNodeDecorator;
function getCrudBulkMutationNames(filename, node) {
    const names = [];
    (0, xtrem_shared_1.getDecoratorArguments)(filename, node).forEach(property => {
        if (ts.isPropertyAssignment(property) &&
            property.initializer &&
            property.initializer.kind === ts.SyntaxKind.TrueKeyword &&
            ts.isIdentifier(property.name)) {
            const propertyName = property.name;
            const m = /^canBulk(Delete|Update)$/.exec(propertyName.escapedText.toString());
            if (m) {
                names.push(`bulk${m[1]}`);
            }
        }
    });
    return names;
}
exports.getCrudBulkMutationNames = getCrudBulkMutationNames;
//# sourceMappingURL=utils.js.map