"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addDictionaryEntry = exports.writeStringLiteralsToBase = exports.merge = exports.getLocaleFilePath = exports.sortByKey = exports.excludeProperties = exports.isEnumFile = exports.isTestFile = exports.isDataTypeFile = exports.isActivityFile = exports.isServiceOptionFile = exports.isNodeArtifactFile = exports.isNodeLikeArtifactFile = exports.isClientArtifactFile = exports.createDictionaryKey = exports.convertDeepBindToPath = exports.literalsCache = void 0;
const fs = require("fs");
const lodash_1 = require("lodash");
const path = require("path");
const utils_1 = require("../utils");
exports.literalsCache = {};
/**
 * Turn nested bind property reference object to dot-notion reference. For example, it converts `{this:{is:{a:{nested:{bind:true}}}}}`
 * to `this__is__a__nested__bind`.
 * @param bind string or deep bind object
 * @returns dot notion deep reference string
 */
const convertDeepBindToPath = (bind) => {
    if (bind === null || bind === undefined || typeof bind === 'boolean') {
        return null;
    }
    if (typeof bind === 'string') {
        return bind;
    }
    if (typeof bind === 'object' && Object.keys(bind).length === 1) {
        const childKeys = Object.keys(bind);
        const childPath = (0, exports.convertDeepBindToPath)(bind[childKeys[0]]);
        if (childPath) {
            return `${childKeys[0]}__${childPath}`;
        }
        return childKeys[0];
    }
    throw new Error(`Unsupported bind value: ${JSON.stringify(bind)}`);
};
exports.convertDeepBindToPath = convertDeepBindToPath;
/**
 * Creates a dictionary key
 * @param prefix prefix for the generated key
 * @param key translation key
 * @returns dictionary key
 */
const createDictionaryKey = (prefix, key, decoratorContext) => {
    let dictionaryKey = `${prefix}__${key.replace(/[.-]/g, '_')}`;
    if (prefix.endsWith('columns') && decoratorContext?.bind) {
        dictionaryKey = `${dictionaryKey}__${(0, exports.convertDeepBindToPath)(decoratorContext.bind)}`;
        if (decoratorContext.valueField) {
            dictionaryKey = `${dictionaryKey}__${(0, exports.convertDeepBindToPath)(decoratorContext.valueField)}`;
        }
    }
    return dictionaryKey;
};
exports.createDictionaryKey = createDictionaryKey;
/**
 * Determines whether the given file is a page or a sticker
 * @param file
 * @returns true if file is a page or a sticker
 */
function isClientArtifactFile(file) {
    const fileName = path.basename(file.fileName, '.ts');
    return (fileName !== 'index' &&
        fileName !== '_index' &&
        Boolean(file.fileName.match(/(\/|\\)lib(\/|\\)(pages|stickers|page-extensions|page-fragments|sticker-extensions|widgets)/)));
}
exports.isClientArtifactFile = isClientArtifactFile;
/**
 * Determines whether the given file is a node or node-extension
 * @param file
 * @returns true if file is a node or node-extension
 */
function isNodeLikeArtifactFile(file) {
    const fileName = path.basename(file.fileName, '.ts');
    return (fileName !== 'index' &&
        fileName !== '_index' &&
        Boolean(file.fileName.match(/(\/|\\)lib(\/|\\)(nodes|node-extensions)/)));
}
exports.isNodeLikeArtifactFile = isNodeLikeArtifactFile;
function isNodeArtifactFile(file) {
    const fileName = path.basename(file.fileName, '.ts');
    return fileName !== 'index' && fileName !== '_index' && Boolean(file.fileName.match(/(\/|\\)lib(\/|\\)nodes/));
}
exports.isNodeArtifactFile = isNodeArtifactFile;
/**
 * Determines whether the given file is a service option
 * @param file
 * @returns true if file is a service option
 */
function isServiceOptionFile(file) {
    const fileName = path.basename(file.fileName, '.ts');
    return (fileName !== 'index' &&
        fileName !== '_index' &&
        Boolean(file.fileName.match(/(\/|\\)lib(\/|\\)service-options(\/|\\)/)));
}
exports.isServiceOptionFile = isServiceOptionFile;
/**
 * Determines whether the given file is an activity
 * @param file
 * @returns true if file is an activity
 */
function isActivityFile(file) {
    const fileName = path.basename(file.fileName, '.ts');
    return (fileName !== 'index' &&
        fileName !== '_index' &&
        Boolean(file.fileName.match(/(\/|\\)lib(\/|\\)activities(\/|\\)/)));
}
exports.isActivityFile = isActivityFile;
/**
 * Determines whether the given file is a dataType
 * @param file
 * @returns true if file is an dataType
 */
function isDataTypeFile(file) {
    return Boolean(file.fileName.match(/(\/|\\)lib(\/|\\)data-types(\/|\\)/)) || isEnumFile(file);
}
exports.isDataTypeFile = isDataTypeFile;
/**
 * Determines whether the given file is a test file
 * @param file
 * @returns true if file is a test file
 */
function isTestFile(file) {
    return (file.fileName.includes('test') && path.normalize(path.dirname(file.fileName)).split(path.sep).includes('test'));
}
exports.isTestFile = isTestFile;
/**
 * Determines whether the given file is an enum file
 * @param file
 * @returns true if file is an enum file
 */
function isEnumFile(file) {
    const fileName = path.basename(file.fileName, '.ts');
    return fileName !== 'index' && fileName !== '_index' && Boolean(file.fileName.match(/(\/|\\)lib(\/|\\)(enums)/));
}
exports.isEnumFile = isEnumFile;
/**
 * Returns a copy of the target without the properties received as second parameter
 * @returns a copy of target without the given properties
 */
const excludeProperties = (target, properties) => {
    return !properties.length ? target : (0, lodash_1.omit)(target, properties);
};
exports.excludeProperties = excludeProperties;
const sortByKey = (dict) => {
    const sortedDict = {};
    Object.keys(dict)
        .sort((a, b) => a.localeCompare(b))
        .forEach(key => {
        sortedDict[key] = dict[key];
    });
    return sortedDict;
};
exports.sortByKey = sortByKey;
const getLocaleFilePath = (dir, locale, isBuild = false) => {
    const targetDir = isBuild ? path.join(dir, 'build', 'lib', 'i18n') : path.join(dir, 'lib', 'i18n');
    if (!fs.existsSync(targetDir)) {
        fs.mkdirSync(targetDir);
    }
    return path.resolve(targetDir, `${locale}.json`);
};
exports.getLocaleFilePath = getLocaleFilePath;
function merge(target, source) {
    // eslint-disable-next-line no-restricted-syntax
    for (const [key, value] of Object.entries(source)) {
        if (value != null)
            target[key] = value;
    }
    return target;
}
exports.merge = merge;
// When the transformers are called from a ts-loader worker, we cannot use the literals cache because they are in different processes
// We may investigate a better way to identify this but for now, this is the easiest I found.
const mainFilename = (require.main?.filename || '').replace(/\\/g, '/');
const isWorker = /\/thread-loader\/dist\/worker.js$/.test(mainFilename);
const isTscp = /\/tspc.js$/.test(mainFilename);
function writeStringLiteralsToBase(currentLiterals, dir) {
    const basePath = (0, exports.getLocaleFilePath)(dir, `base.${process.pid}`);
    // we can use cache only for non-workers process
    if (!isWorker && !isTscp) {
        exports.literalsCache[dir] = exports.literalsCache[dir] ?? {};
        merge(exports.literalsCache[dir], currentLiterals);
        return;
    }
    const baseFileContent = fs.existsSync(basePath) ? JSON.parse(fs.readFileSync(basePath, 'utf-8')) : {};
    const fileContent = merge(baseFileContent, currentLiterals);
    // First write the updated base file.
    (0, utils_1.saveAsJson)(basePath, (0, exports.sortByKey)(fileContent));
}
exports.writeStringLiteralsToBase = writeStringLiteralsToBase;
/**
 * Adds dictionary entry
 * @param key the translation key
 * @param value the translation value
 * @returns the newly created key
 */
function addDictionaryEntry(dictionary, key, value) {
    if (dictionary[key] === undefined) {
        dictionary[key] = value;
        return key;
    }
    const latestDuplicateKey = Object.keys(dictionary)
        .filter(k => k.startsWith(key))
        .pop() || '';
    const counterRegexMatch = latestDuplicateKey.match(/__(\d*)$/);
    const counter = counterRegexMatch ? parseInt(counterRegexMatch[1], 10) + 1 : 2;
    const updatedKey = `${key}__${counter}`;
    dictionary[updatedKey] = value;
    return updatedKey;
}
exports.addDictionaryEntry = addDictionaryEntry;
//# sourceMappingURL=transformer-utils.js.map