"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.decoratorTransformer = exports.visitor = void 0;
const xtrem_shared_1 = require("@sage/xtrem-shared");
const ts = require("typescript");
const utils_1 = require("../utils");
const transformer_utils_1 = require("./transformer-utils");
const includedAstTypes = new Set([ts.SyntaxKind.Decorator]);
const excludedAstTypes = new Set([
    ts.SyntaxKind.ImportDeclaration,
    ts.SyntaxKind.TypePredicate,
    ts.SyntaxKind.TypeReference,
]);
const includedProperties = new Set([
    'addButtonText',
    'content',
    'createTunnelLinkText',
    'description',
    'emptyStateClickableText',
    'emptyStateText',
    'helperText',
    'infoMessage',
    'lookupDialogTitle',
    'objectTypePlural',
    'objectTypeSingular',
    'placeholder',
    'postfix',
    'prefix',
    'subtitle',
    'title',
    'warningMessage',
    'wizardNextButtonLabel',
    'wizardPreviousButtonLabel',
]);
/**
 * Creates a localization wrapper
 * @param key the translation key
 * @returns a Typescript Identifier, i.e. the translation wrapper
 */
function createLocalizeWrapper(key, value) {
    const newNode = ts.factory.createCallExpression(ts.factory.createPropertyAccessExpression(ts.factory.createIdentifier('ui'), ts.factory.createIdentifier('localize')), undefined, [ts.factory.createStringLiteral(key), ts.factory.createStringLiteral(value)]);
    return newNode;
}
/**
 * Gets a node's key, i.e. the node's text
 * @param node Typescript node
 * @returns node key or undefined
 */
function getNodeKey(node) {
    if (!node) {
        return undefined;
    }
    switch (node.kind) {
        case ts.SyntaxKind.PropertyAccessExpression: {
            const expression = node.expression;
            if (!expression) {
                return undefined;
            }
            const name = expression.text;
            if (!name) {
                return undefined;
            }
            return name.indexOf('\n') === -1 ? name : undefined;
        }
        case ts.SyntaxKind.Decorator:
            return '';
        case ts.SyntaxKind.MethodDeclaration:
            return node.name.getText();
        case ts.SyntaxKind.PropertyAssignment:
            return node.name.getText();
        case ts.SyntaxKind.PropertyDeclaration:
            return node.name.getText();
        case ts.SyntaxKind.Identifier:
            return node.text;
        case ts.SyntaxKind.StringLiteral:
            return node.text;
        case ts.SyntaxKind.NoSubstitutionTemplateLiteral:
            return node.text;
        default:
            return undefined;
    }
}
/**
 * Top-level TS visitor
 * @param prefix prefix for translation keys
 */
const visitor = (dictionary, prefix, ctx) => (node) => {
    if (excludedAstTypes.has(node.kind)) {
        return node;
    }
    const localKey = getNodeKey(node);
    // Note that undefined is there on purpose because an empty string is falsy
    const localPrefix = localKey !== undefined ? (0, transformer_utils_1.createDictionaryKey)(prefix, localKey) : prefix;
    return includedAstTypes.has(node.kind)
        ? // If type is of interest, recur on children with 'decoratorVisitor'
            ts.visitEachChild(node, decoratorVisitor(dictionary, localPrefix, ctx, (0, xtrem_shared_1.createDictionary)()), ctx)
        : // Otherwise, recur on children
            ts.visitEachChild(node, (0, exports.visitor)(dictionary, localPrefix, ctx), ctx);
};
exports.visitor = visitor;
/**
 * Decorator visitor
 * @param prefix prefix for translation keys
 */
const decoratorVisitor = (dictionary, prefix, ctx, decoratorContext) => (node) => {
    if (excludedAstTypes.has(node.kind) || ts.isFunctionLike(node)) {
        return node;
    }
    const localKey = getNodeKey(node);
    // If node is not of interest recur on children
    // Note that undefined is there on purpose because an empty string is falsy
    const nextDecoratorContext = ts.isObjectLiteralExpression(node)
        ? (0, utils_1.parseObjectLiteralToObject)(node)
        : decoratorContext;
    if (localKey === undefined) {
        return ts.visitEachChild(node, decoratorVisitor(dictionary, prefix, ctx, nextDecoratorContext), ctx);
    }
    const key = (0, transformer_utils_1.createDictionaryKey)(prefix, localKey, decoratorContext);
    return includedProperties.has(localKey)
        ? // If property is of interest visit it with 'stringVisitor'
            ts.visitEachChild(node, stringVisitor(dictionary, key, ctx, localKey, nextDecoratorContext, node.kind), ctx)
        : // Otherwise, recur on children
            ts.visitEachChild(node, decoratorVisitor(dictionary, key, ctx, nextDecoratorContext), ctx);
};
/**
 * String visitor
 * @param prefix prefix for translation keys
 */
const stringVisitor = (dictionary, prefix, ctx, parentKey, decoratorContext, parentKind) => (node) => {
    if (excludedAstTypes.has(node.kind) || ts.isFunctionLike(node)) {
        return node;
    }
    const localKey = getNodeKey(node);
    // If node is not of interest recur on children
    // Note that undefined is there on purpose because an empty string is falsy
    if (localKey === undefined) {
        return ts.visitEachChild(node, stringVisitor(dictionary, prefix, ctx, parentKey, decoratorContext, node.kind), ctx);
    }
    const key = (0, transformer_utils_1.createDictionaryKey)(prefix, localKey, decoratorContext);
    if (includedProperties.has(parentKey) &&
        ts.SyntaxKind.PropertyAssignment === parentKind &&
        (ts.isStringLiteral(node) || ts.isNoSubstitutionTemplateLiteral(node))) {
        // Discard if not a string
        const dictionaryKey = (0, transformer_utils_1.addDictionaryEntry)(dictionary, prefix, localKey);
        return createLocalizeWrapper(dictionaryKey, localKey);
    }
    // We need to keep the key of nested field node or event handlers (e.g. title for onClick)
    const correspondingKey = node.kind === ts.SyntaxKind.MethodDeclaration ? parentKey : localKey;
    return ts.visitEachChild(node, stringVisitor(dictionary, key, ctx, correspondingKey, decoratorContext, node.kind), ctx);
};
/**
 * This transformer is meant to be run BEFORE the 'message-transformer' and its purpose is
 * to wrap some page/sticker decorator properties (see 'includedProperties') with a call to the 'ui.localize' function.
 *
 * @param ctx transformation context
 * @returns the transformed file
 */
function decoratorTransformer(ctx) {
    return (file) => {
        if (!(0, transformer_utils_1.isClientArtifactFile)(file)) {
            return file;
        }
        try {
            const dictionary = {};
            const fileName = (0, utils_1.getFileName)(file);
            const dirName = (0, utils_1.getDirName)(file);
            const nameAndRoot = (0, utils_1.getPackageNameAndRoot)(file.fileName);
            const fileKey = `${nameAndRoot.name}/${(0, transformer_utils_1.createDictionaryKey)(dirName, fileName)}`;
            const result = ts.visitNode(file, (0, exports.visitor)(dictionary, fileKey, ctx));
            return result;
        }
        catch (err) {
            throw new Error(`Decorator transformer failed due to the following error: ${err}`);
        }
    };
}
exports.decoratorTransformer = decoratorTransformer;
//# sourceMappingURL=decorator-transformer.js.map