/** @packageDocumentation @module test */
import { AnyRecord, AnyValue, AsyncResponse } from '@sage/xtrem-async-helper';
import { AuthConfig, Config, Dict, LocalizeLocale } from '@sage/xtrem-shared';
import { CookieOptions, NextFunction, Request, Response } from 'express';
import { ExecutionResult, FormattedExecutionResult } from 'graphql';
import { Application, ApplicationCreateOptions, ServiceOption } from '../application';
import { Context, ContextOptions, ContextSource } from '../runtime/context';
import { Node } from '../ts-api';
import { ConditionVariable } from './condition-variable';
import { Plugin } from './mocker';
export { CookieOptions } from 'express';
interface TestContextOptions extends ContextOptions {
    /**
     * The (optional) tenantId to use for the context
     */
    tenantId?: string;
}
/**
 * Single interface used to populate and propagate test options
 */
export interface TestOptions {
    /**
     * The tenantId to use for the context. If not set, Test.defaultTenantId will be used
     */
    tenantId?: string;
    config?: any;
    source?: ContextSource;
    mocks?: string[];
    directory?: string;
    scenario?: string;
    today?: string;
    now?: string;
    user?: {
        email: string;
        userName?: string;
    };
    auth?: AuthConfig;
    locale?: LocalizeLocale;
    currentLegislationCode?: string;
    skipMocks?: boolean;
    plugins?: Plugin[];
    testAttributes?: AnyRecord;
    /**
     * the list of service options required for tests
     */
    testActiveServiceOptions?: ServiceOption[];
    /**
     * Disable all CRUD notifications from triggers
     */
    disableAllCrudNotifications?: boolean;
    /**
     * Disable tenant CRUD notifications from triggers
     */
    disableTenantCrudNotifications?: boolean;
    /** Do not use lazy loading of property values */
    noLazyLoading?: boolean;
    /** **DANGER**: This marks that the context will run SQL commands without a tenant filter regardless if they are shared tables or not. */
    unsafeApplyToAllTenants?: boolean;
    timeLimitAsTimestamp?: number;
    /**
     * An optional description for the context
     */
    description?: () => string;
}
export interface TestConfig {
    mocks?: string[];
    directory?: string;
    scenario?: string;
    testAttributes?: AnyRecord;
}
export interface UserTestOptions {
    user?: string;
    cookie?: boolean | ((name: string, val: string, options: CookieOptions) => void);
    tenantId?: string;
    auth?: AuthConfig;
}
export interface Cookie {
    value: string;
    options: CookieOptions;
}
interface ApiRouter {
    use(cb: (req: Request, res: Response, next: NextFunction) => void): void;
}
type State = {
    layers?: string[];
    nowMock?: string;
    application?: string;
};
export declare class Test {
    private static _application;
    private static _config;
    private static _timeStubs;
    static _schemaExists: Dict<boolean>;
    /**
     * The id of the default tenant used for tests
     */
    static readonly defaultTenantId: string;
    static readonly defaultEmail = "unit.test@acme.com";
    /**
     * The name of the default tenant used for tests
     */
    static readonly defaultTenantName = "Tenant for tests (automatic creation)";
    static get application(): Application;
    static set application(application: Application);
    /**
     * Creates the 777...777 tenant for tests if it does not exist.
     */
    static ensureTestTenantExists(context: Context): Promise<void>;
    /**
     * Creates a committed context, bound to the tenantId used for tests, unless a specific tenantId is provided in the options.
     */
    static withCommittedContext<T extends AnyValue | void>(body: (context: Context) => AsyncResponse<T>, options?: TestContextOptions): Promise<T>;
    /**
     * Creates a uncommitted context, bound to the tenantId used for tests, unless a specific tenantId is provided in the options.
     */
    static withUncommittedContext<T extends AnyValue | void>(body: (context: Context) => AsyncResponse<T>, options?: TestContextOptions): Promise<T>;
    /**
     * Creates a read-only context, bound to the tenantId used for tests, unless a specific tenantId is provided in the options.
     */
    static withReadonlyContext<T extends AnyValue | void>(body: (context: Context) => AsyncResponse<T>, options?: TestContextOptions): Promise<T>;
    static get state(): State;
    private static get options();
    static cliActiveServiceOptions: ServiceOption[];
    static isAllocationLoaded: boolean;
    /**
     * Update config with passed parameter
     * @param patch
     */
    static patchConfig(patch: AnyRecord): void;
    static configForTest(): Config;
    static fixRequestMiddleWare(req: Request, res: Response, next: NextFunction): void;
    /**
     * Convert the passed TestOptions to ContextOptions
     * @param options
     */
    static convertOptions(options: TestOptions | undefined): TestContextOptions;
    /**
     * Returns whether the status of a service option is enabled or not by the config.
     * It uses the test config which may be have been patched by Test.patchConfig.
     */
    static isServiceOptionEnabledByTestConfig(serviceOption: ServiceOption): boolean;
    /**
     * Returns the list of service options which should be activated by the test.
     *
     * If the testActiveServiceOptions option was not set and storage is external, we return undefined.
     * The serviceOptionManager will get the service option states from the external storage in this case.
     *
     * This is called by the Context constructor. The options returned by this call are set in the
     * context but are not persisted in the database. This is an efficient way to manage the service options
     * in all tests that use uncommitted contexts.
     */
    static getTestActiveServiceOptions(options?: TestOptions): ServiceOption[] | undefined;
    /**
     * set plugins from the passed TestOptions
     * @param options
     */
    static createPlugins(options: TestOptions): TestOptions;
    /** Preliminary steps */
    private static preliminarySteps;
    /**
     * Execute the passed body with a readonly transaction
     *  converting the passed test options to context options and applying any mocks passed
     * @param body
     * @param opts
     */
    static withContext<T extends AnyValue | void>(body: (context: Context) => AsyncResponse<T>, options?: TestOptions): Promise<T>;
    private static withMocks;
    /**
     * This method activates a list of service options, executes the test body, and then restores the service options
     * to their prior state.
     *
     * Unlike the service options returns by Test.getTestActiveServiceOptions(options), the changes of service option
     * states are persisted into the database.
     * This is used by test which query the database from other contexts to get the state of service options,
     * typically tests that activate the isDemoTenant service option.
     */
    static withCommittedServiceOptions<T extends AnyValue>(tenantId: string, body: () => AsyncResponse<T>, serviceOptions: ServiceOption[]): Promise<T>;
    static withUserContext<T extends AnyValue>(body: (context: Context) => AsyncResponse<T>, options: UserTestOptions, serviceOptions?: ServiceOption[]): Promise<T>;
    /**
     * Execute the passed body with a readonly transaction
     *  converting the passed test options to context options and applying any mocks passed
     * @param body
     * @param opts
     */
    /** @deprecated */
    static readonly<T extends AnyValue | void>(body: (context: Context) => AsyncResponse<T>, options?: TestOptions): Promise<T>;
    /**
     * Execute the passed body with a committed transaction
     *  converting the passed test options to context options and applying any mocks passed
     * @param body
     * @param opts
     */
    /** @deprecated */
    static committed<T extends AnyValue | void>(body: (context: Context) => AsyncResponse<T>, options?: TestOptions): Promise<T>;
    /**
     * Execute the passed body with a uncommitted transaction
     *  converting the passed test options to context options and applying any mocks passed
     * @param body
     * @param opts
     */
    /** @deprecated */
    static uncommitted<T extends AnyValue | void>(body: (context: Context) => AsyncResponse<T>, options?: TestOptions): Promise<T>;
    static setup(application: Application): Promise<void>;
    static getTestSchemaName: (dir: string) => string;
    static getDirectory: (dir: string, fileName: string) => string;
    static createCliApplication(options: ApplicationCreateOptions): Application;
    static createTestApplication(options: ApplicationCreateOptions): Promise<Application>;
    static setupApi(application: Application, apiApp: ApiRouter): Promise<Application>;
    static initializeManagers(context: Context): void;
    static cleanUp(application: Application): Promise<void>;
    static applyDateMocks(options: TestOptions): void;
    static restoreDateMocks: () => void;
    private static withContextMocks;
    /**
     * Creates a condition variable.
     * @param name the name of the condition variable (for logs)
     * @returns the condition variable.
     */
    static createConditionVariable(name: string): ConditionVariable;
    /**
     * **WARNING:**
     * This rolls back the transaction state cache and makes all previously cached node states stale. Please use wisely
     *
     * @param context
     */
    static rollbackCache(context: Context): Promise<void>;
    /**
     * Sends a graphql request to the test application.
     */
    static graphql<ResultT extends AnyValue>(context: Context, request: string): Promise<{
        data: ResultT;
    }>;
    /** Test hacks that we should eliminate */
    static hacks: {
        isNodeStateUpdatable(node: Node): boolean;
        setNodeStateUpdatable(node: Node): void;
    };
}
/**
 * graphql 16.1 preserves the error objects instead of converting them to plain objects.
 * We convert them here to get the same result in direct in-memory calls as with HTTP transport.
 * This facilitates error checking in unit tests.
 */
export declare function mapGraphQlResult<T extends AnyValue>(result: ExecutionResult<T>): FormattedExecutionResult<T>;
//# sourceMappingURL=test.d.ts.map