import { TableDefinition, TriggerDefinition, TriggerEventType, TriggerTimingType } from '@sage/xtrem-postgres';
export interface TriggerBuilderParams {
    when: TriggerTimingType;
    event: TriggerEventType;
    options?: string;
    functionParametersProvider?: (tableDef: TableDefinition) => string;
}
/**
 * A trigger builder is used to manage triggers in database (create/drop).
 */
export declare class TriggerBuilder {
    protected readonly name: string;
    private static _builders;
    protected readonly when: TriggerTimingType;
    protected readonly event: TriggerEventType;
    protected readonly options?: string;
    protected readonly functionParametersProvider?: (tableDef: TableDefinition) => string;
    protected constructor(name: string, params: TriggerBuilderParams);
    private static _getKey;
    get key(): string;
    /**
     * Create (and register) a table trigger (insert/update/delete base/...)
     */
    static createTableTriggerBuilder(name: string, params: TriggerBuilderParams): TriggerBuilder;
    /**
     * Register a trigger builder
     */
    static registerBuilder(triggerBuilder: TriggerBuilder): TriggerBuilder;
    /**
     * Get a trigger builder (already registered with registerBuilder() function)
     */
    static getBuilder(name: string, when: TriggerTimingType, event: TriggerEventType, options?: string): TriggerBuilder;
    /**
     * Returns the registered builder that can manage the provided trigger definition
     */
    static getBuilderForTrigger(trigger: TriggerDefinition): TriggerBuilder;
    /**
     * Returns the SQL command to execute to create the trigger.
     */
    getSqlToCreateTrigger(tableDef: TableDefinition): string;
    getFunctionParameters(tableDef: TableDefinition): string;
    /**
     * Return the definition of the trigger
     */
    getDefinition(tableDef: TableDefinition): TriggerDefinition;
    /**
     * Returns the full name (including the schema name) of the function that will be executed by the trigger.
     */
    getFunctionFullname(tableDef: TableDefinition): string;
    /**
     * Indicates whether the trigger uses a dedicated function (that should be created
     * when the trigger is created and dropped when the trigger is dropped).
     * Should return false if the trigger uses a common function (insert_table, ...)
     */
    get useDedicatedFunction(): boolean;
    /**
     * Returns the SQL command to execute to create the dedicated function that will be called by the trigger (if any)
     * Will return undefined when the trigger is using a common function (insert_table, ...).
     * Must be in sync with useDedicatedFunction.
     */
    getSqlToCreateDedicatedFunction(tableDef: TableDefinition): string | undefined;
    /**
     * The SQL command to drop the trigger and its dedicated function (if any)
     * @param tableDef
     * @returns
     */
    getSqlToDrop(tableDef: TableDefinition): string;
}
//# sourceMappingURL=trigger-builder.d.ts.map