/** @module @sage/xtrem-postgres */
import { AnyValue, AsyncReader, AsyncResponse } from '@sage/xtrem-async-helper';
import { Connection, ConnectionPool, PoolClient, PoolConfig, PoolLoggerInterface, PostgresPoolReaderOptions, PostgresSqlExecuteOptions, TableDefinition } from '@sage/xtrem-postgres';
import { SqlErrorMessageType } from './types';
interface ExecuteSqlParameters {
    cnx?: Connection;
    sql: string;
    args?: any[];
    opts?: PostgresSqlExecuteOptions;
}
type ContextConnectionPoolType = 'user' | 'sys';
export declare class SqlContext {
    #private;
    protected lockId?: number | undefined;
    readonly config: PoolConfig;
    /**
     * Creates a new instance of the SqlContext
     * @param lockId the lock number (must be unique and randomly generated)
     * @param poolType the pool type: 'sys' or 'user'
     */
    constructor(lockId?: number | undefined, poolType?: ContextConnectionPoolType);
    static get logger(): PoolLoggerInterface;
    protected withAdvisoryLock<T extends AnyValue>(body: (cnx: Connection) => AsyncResponse<T>, client?: Connection): Promise<T>;
    private withAdvisoryLockCnx;
    get connectionPool(): ConnectionPool;
    execute<T extends AnyValue | void>(cnx: Connection, sql: string, args?: any[], opts?: PostgresSqlExecuteOptions): Promise<T>;
    withConnection<T extends AnyValue | void>(body: (cnx: Connection) => AsyncResponse<T>, connection?: PoolClient): Promise<T>;
    createReader<T extends AnyValue>(cnx: Connection, sql: string, args: any[], opts?: PostgresPoolReaderOptions): AsyncReader<T>;
    executeSqlStatement<T extends AnyValue | void>(params: ExecuteSqlParameters): Promise<T>;
    get user(): string;
    /**
     * Returns string containing parameter with passed index i
     * @param i
     */
    static param(i: number): string;
    /**
     * Formats a column name with double quotes
     *
     * @see https://www.postgresql.org/docs/13/sql-syntax-lexical.html
     *
     * @param name
     */
    static escape(name: string): string;
    /**
     * Checks if passed error is an index not found error
     * @param err
     */
    static isIndexNotFound(ex: Error): boolean;
    /**
     * Checks if passed error is a table not found error
     * @param err
     */
    static isTableNotFound(ex: Error): boolean;
    /**
     * Checks if passed error is a unique contraint violation
     * @param err
     */
    static isUniqueViolated(ex: Error): boolean;
    /**
     * Checks if passed error is a locked record error
     * @param err
     */
    static isLocked(ex: Error): boolean;
    /**
     * Checks if passed error is a no data found error
     * @param err
     */
    static isNoRecord(ex: Error): boolean;
    /**
     * Return the full qualified table name based on the passed table definition
     * @param def
     */
    static getFullTableDefName(tableDef: TableDefinition): string;
    /**
     *  Return the full qualified table name based on the passed schema, table name and alias
     * @param schemaName
     * @param tableName
     * @param alias
     */
    static getFullTableName(schemaName: string, tableName: string, alias?: string): string;
    /**
     * Returning message error according to the type.
     * The type as will as its options should be defined in SqlErrorMessageType
     * @param type
     * @param options
     */
    static sqlErrorMessage(type: keyof SqlErrorMessageType, options: SqlErrorMessageType[keyof SqlErrorMessageType]): string;
}
export {};
//# sourceMappingURL=sql-context.d.ts.map