"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.orderedSystemColumns = exports.databaseComputedColumnNames = exports.nonSharedDatabaseComputedColumnNames = exports.sharedDatabaseComputedColumnNames = exports.SystemProperties = exports.defaultCodeLength = void 0;
/** @ignore */ /** */
const xtrem_date_time_1 = require("@sage/xtrem-date-time");
const crypto_1 = require("crypto");
const create_property_1 = require("../properties/create-property");
const schema_1 = require("../sql/schema");
const types_conversion_1 = require("../sql/statements/types-conversion");
const ts_api_1 = require("../ts-api");
const context_1 = require("./context");
const core_hooks_1 = require("./core-hooks");
const loggers_1 = require("./loggers");
const system_data_types_1 = require("./system-data-types");
exports.defaultCodeLength = 32;
/** @disabled_internal */
class SystemProperties {
    static { this.publishedSystemProperties = {}; }
    static { this.systemProperties = {}; }
    static { this.idProperties = {}; }
    static { this.idColumns = {}; }
    static { this.updateTickColumns = {}; }
    static { this.syncTickColumns = {}; }
    static { this.syncInfoColumns = {}; }
    static { this.customDataProperties = {}; }
    static { this.customDataColumns = {}; }
    static { this.tenantIdProperties = {}; }
    static { this.tenantIdColumns = {}; }
    static { this.sourceIdProperties = {}; }
    static { this.sourceIdColumns = {}; }
    static { this.sortValueProperties = {}; }
    static { this.sortValueColumns = {}; }
    static { this.updateActionProperties = {}; }
    static { this.updateActionColumns = {}; }
    static { this.constructorProperties = {}; }
    static { this.constructorColumns = {}; }
    static { this.createUserProperties = {}; }
    static { this.updateUserProperties = {}; }
    static { this.vendorProperties = {}; }
    static { this.updateTickProperties = {}; }
    static { this.syncTickProperties = {}; }
    static { this.syncInfoProperties = {}; }
    static { this.etagProperties = {}; }
    static { this.valuesHashProperties = {}; }
    static { this.valuesHashColumns = {}; }
    static { this.attachmentsProperties = {}; }
    static { this.factoryProperties = {}; }
    /** @internal */
    static getPublishedSystemProperties(factory) {
        const cached = SystemProperties.publishedSystemProperties;
        const name = factory.name;
        if (!cached[name]) {
            cached[name] = [this.etagProperty(factory)];
            if (['sql', 'external'].includes(factory.storage || '')) {
                const createStampProperty = (0, create_property_1.createProperty)(factory, createStamp, factory.rootFactory.package);
                const updateStampProperty = (0, create_property_1.createProperty)(factory, updateStamp, factory.rootFactory.package);
                if (name !== factory.rootFactory.name) {
                    createStampProperty.isInherited = true;
                    updateStampProperty.isInherited = true;
                }
                cached[name].push(createStampProperty);
                cached[name].push(updateStampProperty);
            }
            if (factory.isSynchronizable) {
                const syncTickProperty = (0, create_property_1.createProperty)(factory, syncTick, factory.rootFactory.package);
                if (!factory.nodeDecorator.isSynchronizable) {
                    syncTickProperty.isInherited = true;
                }
                cached[name].push(syncTickProperty);
            }
            if (factory.isSynchronized) {
                const syncInfoProperty = (0, create_property_1.createProperty)(factory, syncInfo, factory.rootFactory.package);
                if (!factory.nodeDecorator.isSynchronized) {
                    syncInfoProperty.isInherited = true;
                }
                cached[name].push(syncInfoProperty);
            }
            cached[name].forEach(p => {
                p.isSystemProperty = true;
            });
        }
        return cached[name];
    }
    /** @disabled_internal */
    static getSystemProperties(factory) {
        const cached = SystemProperties.systemProperties;
        const name = factory.name;
        if (!cached[name]) {
            const createUserDecorator = factory.isSharedByAllTenants
                ? createUserManagedProperty.integer
                : createUserManagedProperty.reference;
            const updateUserDecorator = factory.isSharedByAllTenants
                ? updateUserManagedProperty.integer
                : updateUserManagedProperty.reference;
            if (factory.storage === 'external') {
                createUserDecorator.isNullable = true;
                updateUserDecorator.isNullable = true;
            }
            cached[name] = [
                (0, create_property_1.createProperty)(factory, updateTick, factory.rootFactory.package),
                (0, create_property_1.createProperty)(factory, createUserDecorator, factory.rootFactory.package),
                (0, create_property_1.createProperty)(factory, updateUserDecorator, factory.rootFactory.package),
                ...SystemProperties.getPublishedSystemProperties(factory),
            ];
            if (name !== factory.rootFactory.name) {
                cached[name].forEach(p => {
                    if (p.name !== '_etag' && p.name !== '_syncTick' && p.name !== '_syncInfo')
                        p.isInherited = true;
                });
            }
            if (factory.storage === 'sql') {
                if (factory.hasVendorProperty)
                    cached[name].push((0, create_property_1.createProperty)(factory, vendorManagedProperty.reference, factory.package));
            }
            cached[name].forEach(p => {
                p.isSystemProperty = true;
            });
        }
        return cached[name];
    }
    static getSystemProperty(factory, name) {
        const systemProperties = SystemProperties.getSystemProperties(factory);
        return systemProperties.find(property => property.name === name);
    }
    /** @disabled_internal */
    static isSystemOrTechnicalColumn(factory, columnName) {
        switch (columnName) {
            case '_id':
            case '_tenant_id':
            case '_source_id':
            case '_sort_value':
            case '_custom_data':
            case '_layer':
            case '_vendor':
            case '_constructor': {
                return true;
            }
            default: {
                if (factory.table.columns
                    .filter(column => column.isInternalSystemProperty)
                    .some(column => column.columnName === columnName))
                    return true;
                return false;
            }
        }
    }
    /** @internal */
    static isSystemOrTechnicalProperty(propertyName) {
        switch (propertyName) {
            case '_id':
            case '_sourceId':
            case '_sortValue':
            case '_customData':
            case '_layer':
            case '_vendor':
            case '_etag':
            case '_constructor':
            case '_valuesHash':
            case '_updateTick':
            case '_syncTick':
            case '_syncInfo':
            case '_attachments':
            case '_factory':
            case '_createStamp':
            case '_updateStamp': {
                return true;
            }
            default: {
                return false;
            }
        }
    }
    /** @internal */
    static getPublishedInputSystemProperties() {
        return [etagTransient.name];
    }
    static getProperty(factory, cached, persistent, transient, ownerFactory = factory) {
        const name = factory.name;
        if (!cached[name]) {
            const isStored = factory.storage === 'sql';
            cached[name] = isStored
                ? (0, create_property_1.createProperty)(factory, persistent, ownerFactory.package)
                : (0, create_property_1.createProperty)(factory, transient, ownerFactory.package);
            cached[name].isSystemProperty = true;
            cached[name].isInherited = ownerFactory.name !== factory.name;
        }
        return cached[name];
    }
    static getColumn(factory, cached, property) {
        const name = factory.name;
        const isStored = factory.storage === 'sql';
        if (!isStored)
            throw new Error(`${name} not stored: no column can be created.`);
        if (!cached[name])
            cached[name] = new schema_1.Column(property);
        return cached[name];
    }
    /** @disabled_internal */
    static idProperty(factory) {
        const cached = SystemProperties.idProperties;
        return SystemProperties.getProperty(factory, cached, persistentId, transientId);
    }
    /** @disabled_internal */
    static idColumn(factory) {
        const cached = SystemProperties.idColumns;
        const property = SystemProperties.idProperty(factory);
        return SystemProperties.getColumn(factory, cached, property);
    }
    /** @internal */
    static updateTickProperty(factory) {
        const cached = SystemProperties.updateTickProperties;
        const name = factory.name;
        if (cached[name])
            return cached[name];
        cached[name] = (0, create_property_1.createProperty)(factory, updateTick, factory.rootFactory.package);
        if (factory.baseFactory)
            cached[name].isInherited = true;
        cached[name].isSystemProperty = true;
        return cached[name];
    }
    /** @internal */
    static updateTickColumn(factory) {
        const cached = SystemProperties.updateTickColumns;
        const property = SystemProperties.updateTickProperty(factory);
        return SystemProperties.getColumn(factory, cached, property);
    }
    /** @internal */
    static syncTickProperty(factory) {
        const cached = SystemProperties.syncTickProperties;
        const name = factory.name;
        if (cached[name])
            return cached[name];
        cached[name] = (0, create_property_1.createProperty)(factory, syncTick, factory.rootFactory.package);
        if (factory.baseFactory)
            cached[name].isInherited = true;
        return cached[name];
    }
    /** @internal */
    static syncTickColumn(factory) {
        const cached = SystemProperties.syncTickColumns;
        const property = SystemProperties.syncTickProperty(factory);
        return SystemProperties.getColumn(factory, cached, property);
    }
    /** @internal */
    static syncInfoProperty(factory) {
        const cached = SystemProperties.syncInfoProperties;
        const name = factory.name;
        if (cached[name])
            return cached[name];
        cached[name] = (0, create_property_1.createProperty)(factory, syncInfo, factory.rootFactory.package);
        if (factory.baseFactory)
            cached[name].isInherited = true;
        return cached[name];
    }
    /** @internal */
    static syncInfoColumn(factory) {
        const cached = SystemProperties.syncInfoColumns;
        const property = SystemProperties.syncInfoProperty(factory);
        return SystemProperties.getColumn(factory, cached, property);
    }
    /** @internal */
    static customDataProperty(factory) {
        const cached = SystemProperties.customDataProperties;
        const decorator = factory.hasVendorProperty ? persistentCustomDataWithVendor : persistentCustomData;
        return SystemProperties.getProperty(factory, cached, decorator, decorator);
    }
    /** @internal */
    static customDataColumn(factory) {
        const cached = SystemProperties.customDataColumns;
        const property = SystemProperties.customDataProperty(factory);
        return SystemProperties.getColumn(factory, cached, property);
    }
    /** @disabled_internal */
    static sourceIdProperty(factory) {
        const cached = SystemProperties.sourceIdProperties;
        return SystemProperties.getProperty(factory, cached, persistentSourceId, transientSourceId, factory.rootFactory);
    }
    /** @disabled_internal */
    static sourceIdColumn(factory) {
        const cached = SystemProperties.sourceIdColumns;
        const property = SystemProperties.sourceIdProperty(factory);
        return SystemProperties.getColumn(factory, cached, property);
    }
    /** @internal */
    static sortValueProperty(factory) {
        const cached = SystemProperties.sortValueProperties;
        const ownerFactory = factory.rootCollectionFactory ?? factory;
        const prop = SystemProperties.getProperty(factory, cached, persistentSortValue(factory), transientSortValue, ownerFactory);
        prop.isSystemProperty = false;
        return prop;
    }
    /** @internal */
    static sortValueColumn(factory) {
        const cached = SystemProperties.sortValueColumns;
        const property = SystemProperties.sortValueProperty(factory);
        return SystemProperties.getColumn(factory, cached, property);
    }
    /** @internal */
    static tenantIdProperty(factory) {
        const cached = SystemProperties.tenantIdProperties;
        const name = factory.name;
        if (!cached[name]) {
            const property = (0, create_property_1.createProperty)(factory, tenantId, factory.package);
            property.isSystemProperty = true;
            cached[name] = property;
        }
        return cached[name];
    }
    /** @disabled_internal */
    static tenantIdColumn(factory) {
        const cached = SystemProperties.tenantIdColumns;
        const name = factory.name;
        if (!cached[name]) {
            cached[name] = new schema_1.Column(SystemProperties.tenantIdProperty(factory));
        }
        return cached[name];
    }
    /** @internal */
    static updateActionProperty(factory) {
        const cached = SystemProperties.updateActionProperties;
        const name = factory.name;
        if (!cached[name]) {
            cached[name] = (0, create_property_1.createProperty)(factory, updateAction, factory.package);
            cached[name].isSystemProperty = true;
        }
        return cached[name];
    }
    /** @internal */
    static updateActionColumn(factory) {
        const cached = SystemProperties.updateActionColumns;
        const name = factory.name;
        if (!cached[name])
            cached[name] = new schema_1.Column(SystemProperties.updateActionProperty(factory));
        return cached[name];
    }
    /** @internal */
    static constructorProperty(factory) {
        const cached = SystemProperties.constructorProperties;
        const name = factory.name;
        if (!cached[name]) {
            cached[name] = (0, create_property_1.createProperty)(factory, constructorProperty, factory.package);
            cached[name].isSystemProperty = true;
        }
        return cached[name];
    }
    /** @internal */
    static constructorColumn(factory) {
        const cached = SystemProperties.constructorColumns;
        const name = factory.name;
        if (!cached[name])
            cached[name] = new schema_1.Column(SystemProperties.constructorProperty(factory));
        return cached[name];
    }
    /** @internal */
    static valuesHashProperty(factory) {
        const cached = SystemProperties.valuesHashProperties;
        const decorator = valuesHash;
        return SystemProperties.getProperty(factory, cached, decorator, decorator);
    }
    /** @internal */
    static valuesHashColumn(factory) {
        const cached = SystemProperties.valuesHashColumns;
        const property = SystemProperties.valuesHashProperty(factory);
        return SystemProperties.getColumn(factory, cached, property);
    }
    /** @internal */
    static attachmentsProperty(factory) {
        const cached = SystemProperties.attachmentsProperties;
        const name = factory.name;
        if (cached[name])
            return cached[name];
        const attachmentManager = core_hooks_1.CoreHooks.getAttachmentManager();
        const attachments = {
            name: '_attachments',
            type: 'collection',
            isMutable: true,
            isPublished: true,
            dependsOn: ['_id'],
            node: () => attachmentManager.getAttachmentNode(),
            join: {
                sourceNodeName: new ts_api_1.JoinLiteralValue(factory.name),
                sourceNodeId() {
                    return this._id;
                },
            },
            async saveBegin() {
                await this._attachments.forEach(async (attachment) => {
                    await attachment.$.set({ sourceNodeId: this._id, sourceNodeName: factory.name });
                });
            },
        };
        cached[name] = (0, create_property_1.createProperty)(factory, attachments, factory.systemPropertyDefiningPackage._attachments ?? factory.rootFactory.package);
        cached[name].isSystemProperty = true;
        if (factory.baseFactory)
            cached[name].isInherited = true;
        return cached[name];
    }
    /** @internal */
    static factoryProperty(factory) {
        const metaNodeFactoryConstructor = core_hooks_1.CoreHooks.metadataManager.getMetaNodeFactoryConstructor();
        if (!metaNodeFactoryConstructor)
            return null;
        if (factory.storage !== 'sql')
            return null;
        const cached = SystemProperties.factoryProperties;
        const name = factory.name;
        if (cached[name])
            return cached[name];
        const hasConstructorProperty = factory.isAbstract || !!factory.baseFactory;
        const join = hasConstructorProperty
            ? {
                name() {
                    return this._constructor;
                },
            }
            : {
                name: new ts_api_1.JoinLiteralValue(factory.name),
            };
        const decorator = {
            name: '_factory',
            type: 'reference',
            isPublished: true,
            canLookup: false,
            lookupAccess: true,
            node: () => metaNodeFactoryConstructor,
            join,
        };
        cached[name] = (0, create_property_1.createProperty)(factory, decorator, factory.systemPropertyDefiningPackage._factory ?? factory.rootFactory.package);
        cached[name].isSystemProperty = true;
        if (factory.baseFactory)
            cached[name].isInherited = true;
        return cached[name];
    }
    static getReferencedFactory(application, referenceDecorator) {
        try {
            const nodeConstructor = referenceDecorator.node?.();
            return nodeConstructor ? application.getFactoryByConstructor(nodeConstructor) : null;
        }
        catch (e) {
            loggers_1.loggers.runtime.error(`failed to get referenced factory ${referenceDecorator.name}: ${e.message}`);
            return null;
        }
    }
    static isReferencedFactory(factory, referenceFactoryName, referenceDecorator) {
        try {
            return factory.name === referenceDecorator.node?.().name;
        }
        catch (e) {
            return [referenceFactoryName, `Test${referenceFactoryName}`].includes(factory.name);
        }
    }
    /** @internal */
    static makeManagedReferenceProperty(factory, managedProperty, cached, ownerFactory = factory) {
        const name = factory.name;
        let isReference = managedProperty.isReference(factory);
        if (isReference) {
            // Handle special cases where the managed factory is not yet added to the application
            // For instance, Company.updateUser will be created as an integerProperty (instead of a referenceProperty)
            // Also disable foreign key for User table, because of the self reference.
            // TODO: Revisit later
            if (SystemProperties.getReferencedFactory(factory.application, managedProperty.reference) == null) {
                isReference = false;
            }
        }
        if (!cached[name] || cached[name].isReferenceProperty() !== isReference) {
            cached[name] = (0, create_property_1.createProperty)(factory, isReference ? managedProperty.reference : managedProperty.integer, ownerFactory.package);
            cached[name].isSystemProperty = true;
            cached[name].isInherited = ownerFactory.name !== factory.name;
        }
        return cached[name];
    }
    /** @internal */
    static createUserProperty(factory) {
        return this.makeManagedReferenceProperty(factory, createUserManagedProperty, SystemProperties.createUserProperties, factory.rootFactory);
    }
    /** @internal */
    static isUserManagementProperty(property) {
        return [createUserManagedProperty.reference.name, updateUserManagedProperty.reference.name].includes(property.name);
    }
    /** @internal */
    static isAttachmentsProperty(property) {
        return property.name === '_attachments';
    }
    /** @internal */
    static updateUserProperty(factory) {
        return this.makeManagedReferenceProperty(factory, updateUserManagedProperty, SystemProperties.updateUserProperties, factory.rootFactory);
    }
    /** @internal */
    static vendorProperty(factory) {
        const ownerFactory = factory.rootVendorFactory ?? factory;
        return this.makeManagedReferenceProperty(factory, vendorManagedProperty, SystemProperties.vendorProperties, ownerFactory);
    }
    /** @internal */
    static etagProperty(factory) {
        const cached = SystemProperties.etagProperties;
        const name = factory.name;
        if (!cached[name])
            cached[name] = (0, create_property_1.createProperty)(factory, etagTransient, factory.package);
        return cached[name];
    }
    static getInternalSystemProperties(factory) {
        const systemProperties = [];
        // _id property
        const idPropToAdd = SystemProperties.idProperty(factory);
        // For subclasses, _id can't be auto-incremented. We must be able to set _id's value when creating a new node.
        if (factory.decorators.superDecorators)
            idPropToAdd.isAutoIncrement = false;
        systemProperties.push(idPropToAdd);
        // _sourceId
        systemProperties.push(SystemProperties.sourceIdProperty(factory));
        // Non platform nodes can be have custom data.
        if (!factory.isPlatformNode && factory.storage === 'sql')
            systemProperties.push(SystemProperties.customDataProperty(factory));
        // Add _sortValue system property for vital child collections.
        if (factory.isVitalCollectionChild && !factory.isAssociationChild) {
            const sortValueProp = SystemProperties.sortValueProperty(factory);
            systemProperties.push(sortValueProp);
        }
        // abstract class : add the constructor property.
        if (factory.isAbstract)
            systemProperties.push(SystemProperties.constructorProperty(factory));
        if (!factory.isSharedByAllTenants && ['sql', 'external'].includes(factory.storage || '')) {
            systemProperties.push(SystemProperties.createUserProperty(factory));
            systemProperties.push(SystemProperties.updateUserProperty(factory));
        }
        // _updateTick
        if (['sql'].includes(factory.storage || '')) {
            systemProperties.push(SystemProperties.updateTickProperty(factory));
        }
        if (factory.hasVendorProperty)
            systemProperties.push(SystemProperties.vendorProperty(factory));
        if (factory.isContentAddressable)
            systemProperties.push(SystemProperties.valuesHashProperty(factory));
        if (factory.hasAttachments && !factory.isAbstract)
            systemProperties.push(SystemProperties.attachmentsProperty(factory));
        const metaNodeFactoryProperty = SystemProperties.factoryProperty(factory);
        if (metaNodeFactoryProperty)
            systemProperties.push(metaNodeFactoryProperty);
        return systemProperties;
    }
}
exports.SystemProperties = SystemProperties;
/** entityTag */
const etagTransient = {
    name: '_etag',
    type: 'string',
    isPublished: true,
    excludedFromPayload: true,
    dataType: () => system_data_types_1._etagDataType,
    async computeValue() {
        return (0, crypto_1.createHash)('sha256')
            .update((await this.$.updateStamp).toString())
            .digest('base64');
    },
};
const createStamp = {
    name: '_createStamp',
    type: 'datetime',
    isStored: true,
    excludedFromPayload: true,
    lookupAccess: true,
    defaultValue() {
        return xtrem_date_time_1.Datetime.now();
    },
    sqlAttributes: {
        default: 'now()', // lower case 'now()' as postgres converts it to lowercase and it will be a column consistency during the upgrade if we use NOW()
    },
};
const updateStamp = {
    name: '_updateStamp',
    type: 'datetime',
    isStored: true,
    excludedFromPayload: true,
    lookupAccess: true,
    defaultValue() {
        return xtrem_date_time_1.Datetime.now();
    },
    sqlAttributes: {
        default: 'now()', // lower case 'now()'  as postgres converts it to lowercase and it will be a column consistency during the upgrade if we use NOW()
    },
};
const updateTick = {
    name: '_updateTick',
    type: 'integer',
    isStored: true,
    isPublished: true,
    excludedFromPayload: true,
    defaultValue() {
        return 1;
    },
};
const syncTick = {
    name: '_syncTick',
    type: 'decimal',
    dataType: () => system_data_types_1._syncTickDataType,
    isStored: true,
    isPublished: true,
    excludedFromPayload: true,
    defaultValue() {
        return 0;
    },
    exportValue: 0,
};
const syncInfo = {
    name: '_syncInfo',
    type: 'json',
    isStored: true,
    isPublished: true,
    excludedFromPayload: true,
    exportValue: {},
};
function makeManagedProperty(options) {
    const { factoryName, propertyName, node, isNullable, defaultValue, isReference, ignoreIsActive, isPublished, lookupAccess, canLookup, } = options;
    return {
        factoryName,
        reference: {
            name: propertyName,
            type: 'reference',
            isStored: true,
            isNullable,
            node,
            defaultValue: defaultValue == null ? null : defaultValue,
            ignoreIsActive,
            isPublished,
            lookupAccess,
            canLookup,
        },
        integer: {
            name: propertyName,
            type: 'integer',
            isStored: true,
            isNullable,
            defaultValue: defaultValue == null ? null : defaultValue,
        },
        isReference,
    };
}
function getUserNode() {
    return core_hooks_1.CoreHooks.sysManager.getUserNode();
}
const createUserManagedProperty = makeManagedProperty({
    factoryName: 'User',
    propertyName: '_createUser',
    node: () => getUserNode(),
    isReference: (factory) => !factory.isSharedByAllTenants,
    ignoreIsActive: true,
    isPublished: true,
    lookupAccess: true,
    canLookup: false,
});
const updateUserManagedProperty = makeManagedProperty({
    factoryName: 'User',
    propertyName: '_updateUser',
    node: () => getUserNode(),
    isReference: (factory) => !factory.isSharedByAllTenants,
    ignoreIsActive: true,
    isPublished: true,
    lookupAccess: true,
    canLookup: false,
});
const vendorManagedProperty = makeManagedProperty({
    factoryName: 'SysVendor',
    propertyName: '_vendor',
    node: () => context_1.Context.dataSettingsManager.getSysVendorNode(),
    isPublished: true,
    isNullable: true,
    isReference: (factory) => !(factory.isSharedByAllTenants || factory.isPlatformNode),
});
const transientId = {
    name: '_id',
    type: 'string',
    isPublished: true,
};
const persistentId = {
    ...transientId,
    type: 'integer',
    isStored: true,
    isAutoIncrement: true,
    isTransientInput: false,
    lookupAccess: true,
};
const persistentCustomData = {
    name: '_customData',
    dataType: () => system_data_types_1._jsonDataType,
    type: 'json',
    isNullable: true,
    isTransientInput: false,
    defaultValue: () => ({}),
    isStored: true,
    isPublished: true,
    lookupAccess: true,
};
const persistentCustomDataWithVendor = {
    ...persistentCustomData,
    isOwnedByCustomer: true,
};
const tenantId = {
    name: '_tenantId',
    isStored: true,
    type: 'string',
    dataType: () => system_data_types_1.nanoIdDataType,
};
const transientSourceId = {
    name: '_sourceId',
    type: 'string',
    dataType: () => system_data_types_1._sourceIdDataType,
    isPublished: true,
    isTransientInput: true,
    dependencyIndex: 0,
};
const persistentSourceId = {
    ...transientSourceId,
    isStored: true,
    isTransientInput: false,
};
const transientSortValue = {
    name: '_sortValue',
    type: 'integer',
    isPublished: true,
};
const valuesHash = {
    name: '_valuesHash',
    type: 'string',
    dataType: () => system_data_types_1._valuesHashDataType,
    isStored: true,
    excludedFromPayload: true,
};
const persistentSortValue = (factory) => {
    const rootFactory = factory.rootFactory;
    return {
        ...transientSortValue,
        isStored: true,
        allowedInUniqueIndex: true,
        lookupAccess: true,
        /**
         * The SQL default of the sortValue is the currval of the _id sequence * 100
         * When defaulting on insert the currval will be the same as the _id.
         */
        sqlAttributes: {
            default: rootFactory.fullTableName
                ? `(${(0, types_conversion_1.getSqlCurrvalOfIdSequence)(rootFactory.fullTableName)} * 100)`
                : undefined,
            excludeFromInsertIfNull: true,
        },
    };
};
const updateAction = {
    name: '_action',
    type: 'enum',
    dataType: () => system_data_types_1._updateActionDataType,
    isNullable: true,
    excludedFromPayload: true,
    isPublished: true,
    isTransientInput: true,
};
const constructorProperty = {
    name: '_constructor',
    isStored: true,
    type: 'string',
    dataType: () => system_data_types_1._constructorDataType,
    isPublished: true,
};
// TODO: _update_tick should be in this list but we don't have any triggers for it in shared tables
exports.sharedDatabaseComputedColumnNames = Object.freeze(['_create_stamp', '_update_stamp']);
exports.nonSharedDatabaseComputedColumnNames = Object.freeze([
    '_create_user',
    '_update_user',
    '_create_stamp',
    '_update_stamp',
    '_update_tick',
]);
const databaseComputedColumnNames = (isSharedByAllTenants) => {
    return isSharedByAllTenants ? exports.sharedDatabaseComputedColumnNames : exports.nonSharedDatabaseComputedColumnNames;
};
exports.databaseComputedColumnNames = databaseComputedColumnNames;
// Explicit ordering of the system columns on a table.
exports.orderedSystemColumns = Object.freeze([
    '_tenant_id',
    '_id',
    '_constructor',
    '_sort_value',
    '', // This entry is important as it indicates the split of systems columns in a table. Column before will be created at the start of the table and after will be created at the end
    '_vendor',
    '_create_user',
    '_update_user',
    '_create_stamp',
    '_update_stamp',
    '_update_tick',
    '_source_id',
    '_custom_data',
]);
//# sourceMappingURL=system-properties.js.map