/// <reference types="node" />
/** @packageDocumentation @module runtime */
import { AnyRecord, AnyValue, AsyncArray, AsyncReader, AsyncResponse, UnPromised } from '@sage/xtrem-async-helper';
import { Datetime } from '@sage/xtrem-date-time';
import { Logger } from '@sage/xtrem-log';
import { ConnectionPool, SqlExecuteOptions } from '@sage/xtrem-postgres';
import { AccessStatus, AuthConfig, AuthorizationError, BusinessRuleError, Config, DataInputError, Diagnosis, Dict, InitialNotification, LocalizeLocale, MetaCustomFields, User, ValidationSeverity } from '@sage/xtrem-shared';
import { EventEmitter } from 'events';
import { CookieOptions, Response } from 'express';
import { Application, NotificationManager, ServiceOption } from '../application';
import { ServiceOptionManager } from '../application/service-option-manager';
import { GlobalCacheOptions } from '../cache/index';
import { IsolationLevel, NodeKey, StaticThis } from '../decorators';
import { LookupsArgs } from '../graphql/queries/lookup-query';
import { Property, ReferenceArrayProperty, ReferenceProperty } from '../properties';
import { TestConfig } from '../test';
import { Activity, AggregateGroups, AggregateValues, Diagnose, integer, Node, NodeCreateData, NodeQueryFilter, NodeQueryOptions, QueryAggregateOptions, QueryAggregateReturn, ReadAggregateOptions, ReadAggregateReturn } from '../ts-api';
import { NodeSelectOptions, NodeSelector, NodeSelectResult } from '../ts-api/node-select-types';
import { BinaryStream } from '../types';
import { ConfigurationService } from './configuration-service';
import { ContextVault } from './context-vault';
import { Introspection } from './introspection';
import { NodeFactory } from './node-factory';
import { Transaction, TransactionOptions } from './transaction';
export interface NodeReadOptions {
    forUpdate?: boolean;
    isTransient?: boolean;
    isOnlyForLookup?: boolean;
}
export interface NodeCreateOptions {
    isTransient?: boolean;
    isOnlyForDefaultValues?: boolean;
    isOnlyForDuplicate?: boolean;
    isOnlyForLookup?: boolean;
    writable?: boolean;
}
export interface NodeDeleteOptions {
    skipControls?: boolean;
    path?: string[];
}
export type UserAccess = {
    status: AccessStatus;
    sites: string[] | null;
    accessCodes: string[] | null;
};
export declare const standardOperations: readonly ["create", "update", "delete", "read", "lookup", "import"];
export type StandardOperation = (typeof standardOperations)[number];
/**
 * Extend AccessRightsManager to include Authorization functions from xtrem-authorization
 */
export interface AccessRightsManager {
    /**
     * getUserAccessFor
     * Returns a list of Site ids to which a user has access for the specified operation on a Node
     * Empty array means full access with no restriction on Site
     * Undefined means no access
     * @param context
     * @param nodeName
     * @param propertyOrOperation a graphQL operation or property name
     * @param options optional authorizationCode added for external applications(X3)
     */
    getUserAccessFor(context: Context, nodeName: string, propertyOrOperation: string, options?: {
        authorizationCode?: string;
    }): AsyncResponse<UserAccess>;
    isAccessCodeAvailable(context: Context, accessCode: string): AsyncResponse<boolean>;
    createAdminUser(context: Context, data: UserData, options?: CreateAdminUserOptions): AsyncResponse<void>;
    createRequiredUsers(context: Context): AsyncResponse<void>;
    ensureAdminPersonaCreated(context: Context): AsyncResponse<UserInfo>;
    getUser(context: Context, code: string): AsyncResponse<UserInfo>;
    getCurrentUser(context: Context): AsyncResponse<UserInfo>;
    getUserNode(): StaticThis<Node>;
    getPermissions(context: Context, activity: string): AsyncResponse<string[]>;
    createActivities(context: Context): AsyncResponse<void>;
    updateActivities(context: Context): AsyncResponse<void>;
    deleteActivities(context: Context): AsyncResponse<void>;
    getActivityNode(): StaticThis<Node>;
    getActivitiesInfo(context: Context): AsyncResponse<ActivityInfo[]>;
    supportsPersona(context: Context): AsyncResponse<boolean>;
    getPersonaUser(context: Context, email: string): AsyncResponse<UserInfo | null>;
    getDemoPersonas(context: Context): AsyncResponse<UserInfo[]>;
    getUserNavigation?(context: Context): AsyncResponse<UserNavigationInfo>;
    invalidateAuthorizationCache(context: Context): Promise<void>;
}
export interface LocalizationManager extends BaseManager {
    /**
     * @param context
     */
    getDefaultTenantLocale(context: Context): AsyncResponse<string>;
    isMasterLocale(context: Context): AsyncResponse<boolean>;
    createTenantLocale(context: Context, locale: string): AsyncResponse<void>;
}
export interface UserData {
    email: string;
    firstName: string;
    lastName: string;
    locale: string;
}
export interface CreateAdminUserOptions {
    skipWelcomeEmail?: boolean;
}
export interface UserNavigationInfo {
    history?: string[];
    bookmarks?: string[];
}
export interface UserInfo extends User<BinaryStream> {
    _id: string | number;
    email: string;
    firstName?: string;
    lastName?: string;
    isActive?: boolean;
    /** Is this an Admin user */
    isAdministrator?: boolean;
    /** Is this user required to be created per tenant */
    isRequired?: boolean;
    photo?: BinaryStream | null;
    userName?: string;
    isFirstAdminUser?: boolean;
    isDemoPersona?: boolean;
    isApiUser?: boolean;
    locale?: string;
    clientEncryptionKey?: string;
}
export interface ActivityInfo {
    _id: string | number;
    name: string;
}
export interface ApiApplication {
    /** API application Id */
    id: string | undefined;
    /** API application name */
    name: string | undefined;
}
export type TableExistsCallback = (folder: string, tableName: string) => boolean;
export interface CustomRecordInterface {
    bundleId: string;
    factoryName: string;
    _customData?: any;
}
export type PackVersionInterface = {
    name: string;
    version: string;
    isHidden: boolean;
    isReleased: boolean;
    sqlSchemaVersion?: string;
};
export interface PackVersionInterfaceWithId extends PackVersionInterface {
    _id: number;
}
export declare enum PackAllocationStatusEnum {
    off = 1,
    preparing = 2,
    on = 3
}
export type PackAllocationStatus = keyof typeof PackAllocationStatusEnum;
export type PackAllocationInterface = {
    package: Partial<PackVersionInterface>;
    isActive: boolean;
    status: PackAllocationStatus;
    isActivable: boolean;
};
export interface PackVersionOptions {
    isActivable?: boolean;
    isHidden?: boolean;
    isReleased?: boolean;
    sqlSchemaVersion?: string;
}
export type UpdateSetFunctionSet<This> = {
    [K in Exclude<keyof This, '$' | '_id' | '_sourceId' | '_sortValue'>]?: UnPromised<This[K]> | ((this: This) => This[K] | UnPromised<This[K]>);
};
type BulkWhereFilter<This extends Node> = NodeQueryFilter<This>;
/**

 * Options for bulk updates
 */
export interface BulkUpdateOptions<This extends Node> {
    /**
     * The 'set' object that describes the properties to be updated with their new value (or functions that return a value).
     *
     * Example:
     * ```
     *       set {
     *          description: 'new description',
     *          value() { return this.otherProperty * 2},
     *       }
     * ```
     */
    set: UpdateSetFunctionSet<This>;
    /**
     * The optional 'where' function to filter the update
     *
     * Example:
     * ```
     *       where() {
     *           return this.site === 'siteCode';
     *       },
     * ```
     * or:
     * ```
     *       where: {
     *           site: 'siteCode',
     *       },
     * ```
     */
    where?: BulkWhereFilter<This>;
}
export interface BulkDeleteOptions<This extends Node> {
    /**
     * The optional 'where' function to filter the update
     *
     * Example:
     * ```
     *       where() {
     *           return this.site === 'siteCode';
     *       },
     * ```
     * or:
     * ```
     *       where: {
     *           site: 'siteCode',
     *       },
     * ```
     */
    where?: BulkWhereFilter<This>;
}
export interface BaseManager {
    initializeManager(context: Context): void;
}
export interface CustomerInfo {
    id: string;
    name: string;
}
export interface TenantInfoBase {
    id: string;
    directoryName?: string;
}
export interface TenantInfo extends TenantInfoBase {
    name: string;
    directoryName: string;
    customer: CustomerInfo;
}
/** @interface */
export interface TenantManager extends BaseManager {
    ensureTenantExists(context: Context, options: {
        customer: {
            id: string;
            name: string;
        };
        tenant: {
            id: string;
            name: string;
        };
    }): AsyncResponse<void>;
    listTenantsIds(context: Context): AsyncResponse<string[]>;
    getTenantsInfo(context: Context, tenantId?: string): AsyncResponse<TenantInfo[]>;
    deleteTenant(application: Application, tenantId: string): AsyncResponse<void>;
}
export interface DataSettingsManager extends BaseManager {
    getSysVendorNode(): StaticThis<Node>;
    sageVendorId(context: Context): AsyncResponse<number>;
}
export interface ContextCacheOptions<T extends AnyValue> extends GlobalCacheOptions<T> {
}
interface ErrorParameters {
    key: string;
    message: string;
    data?: object | AnyValue[];
    innerError?: Error;
    diagnoses?: Diagnosis[];
}
interface CookieDefinition {
    name: string;
    options: CookieOptions;
}
export declare enum ContextGetConfigurationKeyEnum {
    serviceOptionsLevel = 0
}
export type ContextGetConfigurationKeyType = keyof typeof ContextGetConfigurationKeyEnum;
export declare const rootUserEmail = "root@localhost.domain";
export declare const supportUserEmail = "support@localhost.domain";
export declare const supportReadonlyUserEmail = "support.readonly@localhost.domain";
export declare const adminDemoPersonaEmail = "admin.persona@localhost.domain";
export declare const adminDemoPersona: UserInfo;
export declare const personaCookieDefinition: (tenantId: string, secure: boolean) => CookieDefinition;
export interface ContextInternal {
    userIp: string;
}
/**
 * The _context_
 *
 * The context carries important information about the request which is currently being executed,
 * like the user name and the current locale.
 *
 * The context also provides method to initiate transactions, query and create nodes.
 *
 * The context is created by the framework when the framework receives a GraphQL request,
 * and disposed just after the request completes and the response is sent.
 * Applicative code does not create contexts directly, except to run unit tests (@see [Test](test) API).
 */
export declare class Context extends EventEmitter {
    #private;
    readonly application: Application;
    private readonly options;
    currentLegislationCode: string;
    readonly globals: AnyRecord;
    readonly constants: AnyRecord;
    readonly customFields: MetaCustomFields;
    /**
     * The already logged messages.
     * - index is the message itself.
     * - value is the number of times the message should have been logged
     */
    private readonly _duplicatedLoggedMessages;
    /**
     * The nodes that were saved with the {deferred:true} option (node.$.save({deferred:true}))
     */
    private readonly _nodesWithDeferredSave;
    get source(): ContextSource;
    get isIsolated(): boolean;
    get withoutTransactionUser(): boolean;
    private _contextValues;
    private _lastTransientId;
    private _allocateExternalIds;
    private _lastExternalTransientId;
    testMode: boolean;
    testLayers?: string[];
    private _collectedDependencyPaths?;
    readonly testConfig?: TestConfig;
    testNowMock?: string;
    /**
     * Are we loading CSV files ?
     */
    private _inCsvLoading;
    /**
     * The child contexts
     */
    private children;
    /**
     * Context vault
     */
    readonly vault: ContextVault;
    readonly logger: Logger;
    readonly sqlFunctionCache: Dict<string>;
    mappedIds: Dict<number>;
    private init;
    static create(application: Application, options: ContextOptions, tenantId: string | null): Promise<Context>;
    close(): Promise<void>;
    get schemaName(): string;
    /**
     * Register a node that was saved with the {deferred:true} option (node.$.save({deferred:true}))
     * This node will only be saved to the database at the end of the transaction
     */
    queueDeferredSave(node: Node): void;
    /**
     * Returns the nodes that were saved with the {deferred:true} option (node.$.save({deferred:true}))
     */
    get nodesWithDeferredSave(): Readonly<Node[]>;
    /**
     * Log details for an expensive context
     */
    private _logDetailsForExpensiveContext;
    businessRuleError(params: ErrorParameters): BusinessRuleError;
    private integrityConstraintError;
    /**
     * Returns whether a message should be logged (only used when we want to avoid logging the same message multiple times)
     * This callback will only be invoked when the logger is called with a ignoreCallback option set
     */
    shouldIgnoreDuplicateLogs(message: string): boolean;
    authorizationError(params: ErrorParameters): AuthorizationError;
    dataInputError(params: ErrorParameters): DataInputError;
    isSecure(): boolean;
    isHttp(): this is {
        response: Response;
    };
    /**
     * subscribeToCacheNotifications subscribes to cache notifications sent by other containers
     */
    static subscribeToCacheNotifications(application: Application): Promise<void>;
    /**
     * Gets the source of the request
     */
    get cloudflareRayID(): string;
    /**
     * Gets the source of the request
     */
    get requestSource(): string;
    /**
     * Gets the cloudflare id or an internal id if not available
     */
    get customerRequestId(): string;
    get originId(): string;
    get isAborted(): boolean;
    private checkNotAborted;
    /**
     * Returns the current transaction
     */
    get transaction(): Transaction;
    /**
     * Returns the current tenantId
     * When null, the context can only be used to access to sharedTables
     */
    get tenantId(): string | null;
    /**
     * Returns the unsafeApplyToAllTenants flag
     */
    get unsafeApplyToAllTenants(): boolean;
    private resetUsers;
    /**
     * Set the current tenantId.
     * This function should not be used, please set the tenantId when creating the context instead
     */
    setTenantId(tenantId: string | null): Promise<void>;
    /** Returns the dependency paths for the specified property's rule (used by xtrem-cop) */
    collectDependencyPaths(property: Property, ruleName: 'defaultValue' | 'updatedValue'): string[];
    /**
     * @disabled_internal
     * The list of enabled package for the tenant
     */
    getActivePackageNames(): Promise<string[]>;
    /**
     * Check if current user has specified permission on an Activity
     */
    isAuthorized(activity: Activity, permission: string): Promise<boolean>;
    getActivities(): Dict<Activity>;
    isNodeAccessControlled(nodeName: string): boolean;
    /**
     * Indicates whether a package is enabled for the tenant
     */
    isPackageEnabled(packageName?: string): Promise<boolean>;
    supportsPersona(): AsyncResponse<boolean>;
    isDemoPersona(user: {
        isDemoPersona?: boolean;
    } | null): Promise<boolean>;
    /**
     * The list of active service options for the tenant
     */
    get activeServiceOptions(): Promise<ServiceOption[]>;
    /**
     * SYNC variant of the service options API
     * The API is duplicated to avoid disruption but this will be resolved after migration to node 16.
     * The Sync API is used by external storage. It will superseed the other one (and we will remove the Sync postfix)
     */
    /**
     * Returns `true` if the given service option is available and active
     * @param serviceOption
     */
    isServiceOptionActiveSync(serviceOption: ServiceOption): boolean;
    /** @disabled_internal */
    allocateTransientId(): number;
    /** The current login user */
    get loginUser(): Promise<UserInfo | null>;
    /**
     * The transaction user that will be used as createUser and updateUser in the DB transaction
     * Note: this user will be null when the context was declared as to be used with sharedTables
     */
    get transactionUser(): Promise<Pick<UserInfo, '_id' | 'email'> | null>;
    private resolveLoginUser;
    private verifiedUser;
    private resolvePersonaUser;
    private setPersonaCookie;
    setDemoPersona(email: string): Promise<boolean>;
    get rootUser(): AsyncResponse<UserInfo>;
    get userId(): string | number;
    /**
     * The current user
     * Note: this user will be null when the context was declared as to be used with sharedTables
     */
    get user(): Promise<UserInfo | null>;
    get currentLocaleLanguage(): string;
    get currentLocale(): string;
    setDefaultLocale(): Promise<void>;
    get defaultLocale(): string;
    get defaultLanguage(): string;
    get locales(): string[];
    get collation(): string | undefined;
    private static getLocalizedFallback;
    localize(key: string, template: string, data?: object | AnyValue[], locale?: LocalizeLocale): string;
    localizeEnumMember(enumName: string, memberName: string): string;
    getContextValue(key: string): string | undefined;
    get sqlPool(): ConnectionPool;
    private static addManager;
    static get managers(): BaseManager[];
    static get accessRightsManager(): AccessRightsManager;
    static set accessRightsManager(value: AccessRightsManager);
    static get localizationManager(): LocalizationManager;
    static set localizationManager(value: LocalizationManager);
    get serviceOptionManager(): ServiceOptionManager;
    static get tenantManager(): TenantManager;
    static set tenantManager(value: TenantManager);
    static get dataSettingsManager(): DataSettingsManager;
    static set dataSettingsManager(value: DataSettingsManager);
    static get notificationManager(): NotificationManager;
    static set notificationManager(value: NotificationManager);
    /**
     * Checks if the current user is authorized to perform an operation on a node.
     * Throws an exception if the user is not authorized.
     */
    checkThatNodeOperationIsAuthorized(nodeName: string, operationName: StandardOperation | string): Promise<void>;
    get isWritable(): boolean;
    get isAutoCommit(): boolean;
    private static overallSqlExecutionCount;
    /**  */
    private static canRelaxSqlInjectionTest;
    private static stripRaiseStatements;
    private static isSafe;
    private static preventSqlInjections;
    /**
     * Utility method used to identify sql statements that are executed more that once for the current context
     * The log will only be output if the log level of `sage/xtrem-core/performance` is set to `debug`
     * but the logic will also be executed at verbose level, to update the counts.
     */
    private logSql;
    /**
     * Method that prints the sql execution count when the context is closed
     * The log will be output if the log level of `sage/xtrem-core/performance` is set to `debug` or `verbose`
     */
    private logSqlSummary;
    /**
     * Log the previously ignored messages (duplicate messages)
     */
    private logIgnoredDuplicateMessages;
    /** @disabled_internal */
    executeSql<T extends AnyValue = AnyValue[]>(sql: string, args: AnyValue[], opts?: SqlExecuteOptions): Promise<T>;
    /**
     * Get the current database timestamp
     * @returns the current timestamp
     */
    getSqlTimestamp(): Promise<Datetime>;
    get mayCommit(): boolean;
    /**
     * Flushes the actions that have been queued for execution before the commit.
     * Call this when you need to force the evaluation of deferredDefaultValue rules
     * in the middle of a transaction, before the commit.
     * Use this method with care as it may increase contention between concurrent transactions.
     */
    flushDeferredActions(): Promise<void>;
    /** @disabled_internal */
    /**
     * DO NOT USE OUTSIDE OF FRAMEWORK
     *
     * executes `body` inside a new context.
     *
     * Important: this call leaves the context open because the SQL transaction may be needed
     * after the body has been exited, for example to produce the response to a graphQl query or mutation.
     * The child context will be closed when its parent is closed and the caller is responsible
     * for closing the parent context.
     * @param options
     * isDetachedContext is passed as true from runInWritableContext, and flags that the context is detached from the parent, i.e. it will manage itself.
     */
    withChildContext<T extends AnyValue | void>(body: (context: Context) => AsyncResponse<T>, options: TransactionOptions & {
        isDetachedContext?: boolean;
        source?: ContextSource;
        isIsolated?: boolean;
    }): Promise<T>;
    /**
     * Runs the provided function in an isolated context.
     *
     * @param body - The function to be executed in the isolated context.
     * @param options - Optional configuration for the isolated context.
     * @returns A promise that resolves to the result of the function execution.
     */
    runInIsolatedContext<T extends AnyValue | void>(body: (context: Context) => AsyncResponse<T>, options?: {
        isolationLevel?: IsolationLevel;
        isReadonly?: boolean;
        disableTenantCrudNotifications?: boolean;
    }): Promise<T>;
    /**
     * executes a `body` passing it a new writable context.
     * If the current context is writable or the source of the current context is not `listener` then an error is thrown
     * @returns a promise resolving the result of `body` in a new writable context
     */
    runInWritableContext<T extends AnyValue>(body: (context: Context) => AsyncResponse<T>, options?: {
        isolationLevel?: IsolationLevel;
        noCommit?: boolean;
        source?: ContextSource;
    }): Promise<T>;
    private diagnosesPrefixes;
    private _diagnoses;
    private _severity;
    get isolationLevel(): IsolationLevel | undefined;
    /** Returns the timestamp value from the context options. */
    get timeLimitAsTimestamp(): number;
    addDiagnoseAtPath(severity: ValidationSeverity, path: string[], message: string): void;
    resetDiagnoses(): void;
    get diagnoses(): Diagnose[];
    get severity(): ValidationSeverity;
    hasErrors(): boolean;
    debugString(): string;
    get introspection(): Introspection;
    /** specialized API to read configuration data (coming from xtrem-config.yml) */
    get configuration(): ConfigurationService;
    /**
     * Creates a node from given data.
     * @param clas
     * @param data
     */
    create<T extends Node>(clas: StaticThis<T>, data: NodeCreateData<T>, options?: NodeCreateOptions): Promise<T>;
    private readNodeState;
    tryRead<T extends Node>(clas: StaticThis<T>, key: NodeKey<T>, options?: NodeReadOptions): Promise<T | null>;
    read<T extends Node>(clas: StaticThis<T>, key: NodeKey<T>, options?: NodeReadOptions): Promise<T>;
    deleteMany<T extends Node>(clas: StaticThis<T>, filter: NodeQueryFilter<T>, options?: NodeDeleteOptions): Promise<number>;
    delete<T extends Node>(clas: StaticThis<T>, key: AnyRecord): Promise<void>;
    executeGraphql<T extends AnyValue>(query: string): Promise<T>;
    executeGraphqlStream<T extends AnyValue>(query: string): Promise<T>;
    query<T extends Node>(clas: StaticThis<T>, options?: NodeQueryOptions<T>): AsyncArray<T>;
    /**
     * Low level select
     *
     * Executes a query and returns an array of plain JavaScript objects.
     * The `selector` parameter specifies the properties included in the response.
     * The `options` are the same as for context.query.
     */
    select<NodeT extends Node, SelectorT extends NodeSelector<NodeT> = NodeSelector<NodeT>>(nodeConstructor: StaticThis<NodeT>, selector: SelectorT, options: NodeSelectOptions<NodeT>): Promise<NodeSelectResult<NodeT, SelectorT>[]>;
    /**
     * Low level select reader
     *
     * Executes a query and returns a reader of plain JavaScript objects.
     * The `selector` parameter specifies the properties included in the response.
     * The `options` are the same as for context.query.
     */
    getSelectReader<NodeT extends Node, SelectorT extends NodeSelector<NodeT> = NodeSelector<NodeT>>(nodeConstructor: StaticThis<NodeT>, selector: SelectorT, options: NodeSelectOptions<NodeT>): Promise<AsyncReader<NodeSelectResult<NodeT, SelectorT>>>;
    /**
     * Runs a bulk update. The options.set and options.where must be simple enough to be parsed into sql statement.
     * If not, an error will be raised
     * @param nodeConstructor
     * @param options
     */
    bulkUpdate<This extends Node>(nodeConstructor: StaticThis<This>, options: BulkUpdateOptions<This>): Promise<number>;
    /**
     * Runs a bulk update. The options.set and options.where must be simple enough to be parsed into sql statement.
     * If not, an error will be raised
     * @param nodeConstructor
     * @param options
     */
    bulkDeleteSql<This extends Node>(nodeConstructor: StaticThis<This>, options: BulkDeleteOptions<This>): Promise<number>;
    queryWithReader<NodeT extends Node, ResultT extends AnyValue>(clas: StaticThis<NodeT>, options: NodeQueryOptions<NodeT>, body: (reader: AsyncReader<NodeT>) => AsyncResponse<ResultT>): Promise<ResultT>;
    queryCount<T extends Node>(clas: StaticThis<T>, options?: NodeQueryOptions<T>): Promise<number>;
    /**
     * Queries an aggregate
     *
     * @param constructor the node constructor
     * @param options the query options
     *
     * Example:
     * ```
     * const results = context.queryAggregates(Invoice, {
     *     filter: { customer: { country: 'US' } },
     *     group: { date: { _by:  'month' } },
     *     values: { amount: { sum: true, avg: true } },
     * })
     * ```
     *
     * This query will return the monthly sum and average amounts of US invoices.
     * Typical result will be:
     *
     * ```
     * [{
     *     group: { date: '01-01-2020' },
     *     values: { amount: { sum: 5000, avg: 1250 } },
     * }, {
     *     group: { date: '01-02-2020' },
     *     values: { amount: { sum: 3800, avg: 950 } },
     * }, {
     *     ...
     * }]
     * ```
     */
    queryAggregate<T extends Node, GroupT extends AggregateGroups<T>, ValuesT extends AggregateValues<T>>(constructor: StaticThis<T>, options: QueryAggregateOptions<T, GroupT, ValuesT>): AsyncArray<QueryAggregateReturn<T, GroupT, ValuesT>>;
    /**
     * Reads an aggregate
     *
     * @param constructor the node constructor
     * @param options the read options
     *
     * Example:
     * ```
     * const results = context.readAggregates(Invoice, {
     *     filter: { customer: { country: 'US' } },
     *     values: { amount: { sum: true, avg: true } },
     * })
     * ```
     *
     * This query will return the sum and average amounts of all US invoices.
     * Typical result will be:
     *
     * ```
     * { amount: { sum: 385000, avg: 1439 } }
     * ```
     */
    readAggregate<T extends Node, ValuesT extends AggregateValues<T>>(clas: StaticThis<T>, options: ReadAggregateOptions<T, ValuesT>): Promise<ReadAggregateReturn<T, ValuesT>>;
    exists<T extends Node>(clas: StaticThis<T>, key: NodeKey<T>): Promise<boolean>;
    /** wrapper to include reference data in data payloads when creating nodes */
    static referenceData<T extends Node>(_clas: StaticThis<T>, data: Partial<T>): T;
    /**
     * Returns the cached value for a given key. If not found, the getValue callback will be invoked (if provided)
     * @param category a string that will identify the category of the cached value
     * @param key the key of the cached value
     * @param options
     */
    getCachedValue<T extends AnyValue>(options: ContextCacheOptions<T>): Promise<T>;
    /**
     * Returns the cache category for a given node.
     * Items that will be cached with this category will be automatically invalidated when the node's table changes.
     */
    getNodeCacheCategory(nodeConstructor: StaticThis<Node>): string;
    /**
     * Invalidates the cache entries for a given category.
     * @param category a string that will identify the category of the cached values
     * @param options
     *
     * This is called when records are inserted, updated or deleted in the database
     * Transactions are isolated from each other so we don't propagate the change to the global cache.
     * Instead, we just record the category in an array of modified categories, and we use this list
     * to invalidate the global cache later, when the transaction is committed.
     */
    invalidateCachedCategory(category: string, options?: {
        skipNotify?: boolean;
    }): Promise<void>;
    private buildFilter;
    private lookupQuery;
    private lookupQueryCount;
    private makeLookupQueryFilters;
    private makeLookupQueryParametersWithTransientScope;
    makeLookupQueryParameters(nodeConstructor: StaticThis<Node>, propertyName: string, args: LookupsArgs): Promise<LookupQueryParameters<Node>>;
    withLocalizedTextAsJson<T extends AnyValue | void>(body: () => AsyncResponse<T>): Promise<T>;
    withoutLocalizedTextAsJson<T extends AnyValue | void>(body: () => AsyncResponse<T>): Promise<T>;
    /**
     * Scope a body with inCsvLoading set to true
     * @param body
     * @returns
     */
    withCsvLoading<T>(body: () => T): T;
    /**
     * Returns whether we are in a withCsvLoading scope
     */
    get inCsvLoading(): boolean;
    /**
     * Activates the conversion of references to natural keys during the execution of `body`.
     * See context.convertReference.
     */
    withReferenceAsNaturalKey<T extends AnyValue | void>(body: () => AsyncResponse<T>): Promise<T>;
    /**
     * Converts a reference when mapping records returned by SQL queries.
     * The id is converted if this is called from a context.withReferenceAsNaturalKey body.
     * Otherwise the id is returned.
     */
    convertReference(factory: NodeFactory, id: integer): Promise<string | integer>;
    private activateTestServiceOptions;
    isServiceOptionEnabled(serviceOption: ServiceOption, options?: {
        noCache: boolean;
    }): Promise<boolean>;
    setServiceOptionsEnabledFlag(serviceOption: ServiceOption, isEnabled: boolean): void;
    initializeFactory<NodeT extends Node>(clas: StaticThis<NodeT> | undefined): Promise<void>;
    static getConfigurationValue(name: ContextGetConfigurationKeyType): string;
    /**
     * To be used when getting a property value and you do not want the getPropertyValue error logger to print
     */
    withoutGetPropertyValueErrorLogger<T extends AnyValue | void>(body: () => AsyncResponse<T>): Promise<T>;
    /**
     * Send notification to the client
     * @param category
     * @param payload
     */
    notifyUser(notification: InitialNotification): Promise<void>;
}
interface LookupQueryParameters<T extends Node> {
    node: T;
    property: ReferenceProperty | ReferenceArrayProperty;
}
interface ResponseExtra {
    [key: string]: any;
}
export type ContextSource = 'graphql' | 'workflow' | 'listener' | 'routing' | 'internal' | 'customMutation' | 'web-socket' | 'import' | 'rest';
export interface ContextOptions extends TransactionOptions {
    userEmail?: string;
    auth?: AuthConfig;
    parent?: Context;
    config?: Config;
    legislationCode?: string;
    locale?: string;
    source?: ContextSource;
    request?: any;
    response?: ResponseExtra | Response;
    /**
     * signal to abort this context and all its children, any transaction will be rolled back.
     * This uses the AbortController API available in node.js.
     * See https://nodejs.org/docs/latest-v20.x/api/all.html#all_globals_class-abortcontroller
     */
    signal?: AbortSignal;
    testMode?: boolean;
    testLayers?: string[];
    testNowMock?: string;
    testConfig?: TestConfig;
    /**
     * the list of service options required for the test
     */
    testActiveServiceOptions?: ServiceOption[];
    /**
     * the list of active package for tests
     */
    contextValues?: Dict<string>;
    /**
     * Disable all CRUD notifications from triggers
     */
    disableAllCrudNotifications?: boolean;
    /**
     * Disable tenant CRUD notifications from triggers
     */
    disableTenantCrudNotifications?: boolean;
    /**
     * Flag used by test contexts to force rollback instead of commit.
     */
    noCommit?: boolean;
    /** is the transaction in auto commit mode - overrides other options  */
    isAutoCommit?: boolean;
    /** Do not use lazy loading of property values */
    noLazyLoading?: boolean;
    /** Flag to indicate we are creating the root user
     * This will set transaction user to null to force the default self reference on user insert */
    withoutTransactionUser?: boolean;
    /** is this a system context */
    isSystem?: boolean;
    /** **DANGER**: This marks that the context will run SQL commands without a tenant filter regardless if they are shared tables or not. */
    unsafeApplyToAllTenants?: boolean;
    timeLimitAsTimestamp?: number;
    skipManagedExternalInit?: boolean;
    /** Bypass the isFrozen checks - only allowed in test */
    bypassFrozen?: boolean;
    /** SQL connection is not available - used to verify ts-to-sql conversions */
    withoutSqlConnection?: true;
    /** is this an isolated context */
    isIsolated?: boolean;
    /**
     * An optional description for the context
     */
    description?: () => string;
}
export {};
//# sourceMappingURL=context.d.ts.map