/** @packageDocumentation @module runtime */
import { AnyRecord, AnyValue, AsyncResponse } from '@sage/xtrem-async-helper';
import { Decimal } from '@sage/xtrem-decimal';
import { DataInputError, Dict, LogicError, SystemError, ValidationSeverity } from '@sage/xtrem-shared';
import { InternalAttachmentAssociationData } from '../application';
import { BaseCollection } from '../collections';
import { CollectionProperty, Property } from '../properties';
import { Context, NodeDeleteOptions, NodeReadOptions } from '../runtime/context';
import { NodeFactory } from '../runtime/node-factory';
import { NodeConstructOptions } from '../runtime/node-factory-types';
import { SyncInfo } from '../synchronization/sync-info';
import { AnyNode, Collection, Node, NodePayloadOptions, NodeSaveOptions, NodeStatus, NodeUpdateData, OrderBy, datetime } from '../ts-api';
interface ErrorParameters {
    key: string;
    message: string;
    data?: object | AnyValue[];
    innerError?: Error;
}
/**
 * NodeState instances manage the _state_ of a node instance.
 *
 * There is exactly one NodeState instance for each node instance, accessible with `node.$.state`.
 * The entire state of the node instance is tracked in its `node.$.state` instance.
 *
 * @disabled_internal
 */
export declare class NodeState {
    #private;
    readonly factory: NodeFactory;
    /** The node instance */
    readonly node: Node;
    /**
     * Was the node read for update
     */
    readonly forUpdate: boolean;
    /**
     * The values of the properties.
     */
    values: AnyRecord;
    statesToUpdate: [NodeState, string[]][];
    /**
     * The values of the reference properties, as node instances.
     * For _stored_ reference properties, state.values contains the `_id` of the referenced node,
     * whereas state.references contains the node instance.
     */
    readonly references: Map<string, Node | null>;
    /**
     * The values of the reference array properties, as node instance arrays.
     * For _stored_ reference array properties, state.values contains the array of `_id` of the referenced nodes,
     * whereas state.referenceArrays contains the node instance array.
     */
    readonly referenceArrays: Map<string, Node[] | null>;
    /**
     * The values of the collection properties, as collection instances.
     * The state.values does not contain any value for collection properties.
     */
    readonly collections: Map<string, BaseCollection>;
    /**
     * The internal status of the node.
     * This status is more detailed than the status returned by node.$.status (which is a NodeStatus, not a StateStatus)
     */
    status: StateStatus;
    /**
     * Is the state a thunk?
     * A thunk is a partial state in which `state.values` does not have values for all the _stored_ properties.
     * The missing values will be lazy loaded by reading the record from the database. This will reset the thunk flag.
     */
    isThunk: boolean;
    /**
     * Is the state for a transient node instance?
     */
    readonly isTransient: boolean;
    /**
     * Was this state created only to obtain default values?
     */
    readonly isOnlyForDefaultValues: boolean;
    /**
     * Is the state created only for a lookup query?
     */
    readonly isOnlyForLookup: boolean;
    /**
     * Is the state created only for a duplicate query?
     */
    readonly isOnlyForDuplicate: boolean;
    /**
     * Old state that we saved the first time this state was modified.
     * This is used by `node.$.old`.
     */
    oldState: NodeState | undefined;
    /**
     * Is this the _old_ state of a node
     */
    readonly isOld: boolean;
    /** Set of properties that have been invalidated because properties upon which they depend have been modified */
    readonly invalidProperties: Set<string>;
    /** Set of properties that have been invalidated because properties upon which they depend have been modified */
    readonly deferredProperties: Dict<boolean>;
    /** list of column names that have been modified and not saved yet */
    dirtyColumnNames: Dict<true>;
    /**
     * Index of the last property that we are allowed to access while executing an event.
     *
     * This limit is used to verify that we only access properties that P _depends upon_
     * when we execute rules attached to P (a property).
     */
    dependencyLimit: number;
    /** Unique debug id allocated by incrementing `NodeState.nextDebugId` */
    readonly debugId: number;
    /** Will this state be skipped during control and save */
    skipSave: boolean;
    /** Global counter to allocate `state.debugId` */
    private static nextDebugId;
    /** The state's constructor */
    constructor(context: Context, factory: NodeFactory, values: AnyRecord, status: StateStatus, options: NodeConstructOptions);
    /**
     * Unique key values for node interning
     * Format: <Root factory name>:<key property1 name>:<key property1 value>,<key property2 name>:<key property2 value>, ...
     * Examples:
     *      Currency:_id:1
     *      Currency:code:USD
     *      BaseDocument:_id:1
     *      BaseDocumentLine:_id:1
     *      BaseDocumentLine:_sortValue:100,document:1
     */
    get interningKeys(): Dict<string>;
    get interningKeyValues(): string[];
    /** The collection to which the node belongs. Only set if the node is loaded from a collection */
    private _collection?;
    /**
     * The collection to which the node belongs. Only set if the node is loaded from a collection
     */
    get collection(): BaseCollection | undefined;
    get context(): Context;
    /**
     * The node status
     *
     * This accessor maps the internal StateStatus values to NodeStatus values.
     * See node.$.status.
     */
    get nodeStatus(): NodeStatus;
    /** Is the state readonly? */
    get isReadonly(): boolean;
    /** Is the state writable? (opposite of isReadonly) */
    get isWritable(): boolean;
    /** Checks if status of the current state is created/constructed */
    get isNew(): boolean;
    /**
     * Is the state effectively readonly?
     *
     * Writable states are put in readonly mode during the execution of some rules
     * (defaultValue, updatedValue, control, controlDelete).
     * This accessor returns true while these rules are executed.
     */
    get isEffectivelyReadonly(): boolean;
    /** Is the state stale? */
    get isStale(): boolean;
    /** Returns the value of a property */
    getPropertyValue(property: Property): Promise<AnyValue>;
    /** Returns the value of a property */
    getPropertyValueSync(property: Property): AnyValue;
    /** Returns the value of a property, given the property name */
    getValue<T extends AnyValue>(propertyName: string): Promise<T>;
    /** Returns the value of a property */
    getValueSync<T extends AnyValue>(propertyName: string): T;
    /** Set the value of a property */
    setPropertyValue(property: Property, value: AnyValue): Promise<void>;
    /** Set the value of a property, given the property name */
    setValue(propertyName: string, value: AnyValue): Promise<void>;
    /** Set new values into the state. See node.$.set */
    set(data: NodeUpdateData<Node>, path?: string[]): Promise<void>;
    isPropertyValueDeferred(property: Property): boolean;
    /** The old state. See node.$.old */
    get old(): Promise<Node>;
    /** The _id of the node */
    get id(): number;
    /** The _sortValue of the node */
    get sortValue(): Promise<number>;
    /** The vendor of the node */
    get vendor(): Promise<Node | null>;
    /** Set the _sortValue of the node */
    setSortValue(value: number): Promise<void>;
    /** The _createUser of the node */
    get createdBy(): Promise<Node | number>;
    /** The _updateUser of the node */
    get updatedBy(): Promise<Node | number>;
    /** The _createStamp of the node */
    get createStamp(): Promise<datetime>;
    /** The _updateStamp of the node */
    get updateStamp(): Promise<datetime>;
    /** The _syncTick of the node (sync source only) */
    get syncTick(): Promise<Decimal>;
    /** The _syncInfo of the node (sync target only) */
    get syncInfo(): Promise<SyncInfo>;
    /** The _etag entity tag of the node (updateStamp as a string)  */
    get etag(): Promise<string>;
    /** The custom property data of the node */
    get customData(): Promise<object>;
    /** The custom property data of the node */
    get valuesHash(): Promise<string>;
    /** attachments collection */
    get attachments(): Collection<InternalAttachmentAssociationData & Node>;
    /** Adds a diagnose to the state's context */
    addDiagnose(severity: ValidationSeverity, path: string[], message: string): void;
    /** Validates the state. See node.$validate */
    control(): Promise<boolean>;
    /** Tries to save the state. See node.$.trySave */
    trySave(options?: NodeSaveOptions): Promise<boolean>;
    /** Saves the state. See node.$.save */
    save(options?: NodeSaveOptions): Promise<void>;
    /** Tries to delete the state. See node.$.tryDelete */
    tryDelete(options?: NodeDeleteOptions): Promise<boolean>;
    /** Deletes the state.  See node.$.delete */
    delete(options?: NodeDeleteOptions): Promise<void>;
    getKeyValues(): AnyRecord;
    /** The key token, a string that uniquely identifies a node within its factory */
    get keyToken(): string;
    getNaturalKeyValue(): Promise<string>;
    /** The cursor value for a given ordering. Used by paging API */
    getCursorValue(orderBy: OrderBy<Node>): Promise<string>;
    /** Returns the join values. Used by collections */
    getJoinValues(property: CollectionProperty): AsyncResponse<AnyRecord>;
    payload(options?: NodePayloadOptions<AnyNode>): Promise<AnyRecord>;
    duplicate(data?: AnyRecord): Promise<NodeState>;
    /** Returns an error prefixed by the factory and property name */
    propertySystemError(property: Property, message: string): SystemError;
    /** Returns an error prefixed by the factory and property name */
    propertyDataInputError(property: Property, params: ErrorParameters): DataInputError;
    /** Returns an error prefixed by the factory name */
    systemError(message: string): SystemError;
    /** Returns an error prefixed by the factory name */
    logicError(message: string): LogicError;
    /** Returns an error prefixed by the factory name */
    dataInputError(params: ErrorParameters): DataInputError;
    /** Returns a new NodeState for a record created with context.create */
    static newFromContextCreate(context: Context, factory: NodeFactory, path: string[], data: AnyRecord, options?: {
        isTransient?: boolean;
        isOnlyForDefaultValues?: boolean;
        isOnlyForDuplicate?: boolean;
        writable?: boolean;
        isOnlyForLookup?: boolean;
        collection?: BaseCollection;
    }): Promise<NodeState>;
    /** Returns a new NodeState for a record read individually from the database */
    static newFromRead(context: Context, factory: NodeFactory, key: AnyValue, options: NodeReadOptions & {
        dontThrow?: boolean;
    }): Promise<NodeState | null>;
    /** Returns a new NodeState for a record read by a database query */
    static newFromQuery(context: Context, factory: NodeFactory, record: AnyRecord, forUpdate: boolean): NodeState;
    /** Returns whether the node is frozen
     */
    isNodeFrozen(): AsyncResponse<boolean>;
    /**
     * Returns whether the property is frozen
     */
    isPropertyFrozen(property: Property): AsyncResponse<boolean>;
}
export {};
//# sourceMappingURL=node-state.d.ts.map