import { Datetime } from '@sage/xtrem-date-time';
import { Logger } from '@sage/xtrem-log';
import { AnyValue, AppConfig, Dict } from '@sage/xtrem-shared';
import { AxiosRequestConfig, AxiosResponse } from 'axios';
import { ExecutionPatchResult, ExecutionResult } from 'graphql';
import { Context } from '../runtime/context';
export declare const logger: Logger;
/** Options passed to the App constructor. */
export interface AppConstructOptions {
    /** The name of the app */
    name: string;
    /** The version of the app (semver) */
    version: string;
}
export interface AppInteropToken {
    tenantId: string;
    appName: string;
    scope: string;
    sourceUserEmail: string;
}
export interface AppInteropClaims extends AppInteropToken {
    iat: number;
    exp: number;
}
export interface GraphQlResponse<ResponseDataT = unknown, RequestDataT = {}> extends AxiosResponse<ResponseDataT, RequestDataT> {
}
export interface GraphQlRequestOptions {
    appName: string;
    scope: string;
    query: string;
    variables?: Dict<AnyValue>;
    config?: AxiosRequestConfig;
}
export interface AppHealthInfo {
    /** The timestamp of the health check */
    timestamp: Datetime;
    /** The duration of the health check */
    duration: number;
    /** The status of the health check */
    status: number;
    /** The status text of the health check */
    statusText?: string;
    /** Whether the app is alive or not */
    isAlive: boolean;
}
/**
 * InteropGraphqlClient is a client to send GraphQL (or ping) requests to other apps.
 */
export declare class InteropGraphqlClient {
    private static getSignatureKey;
    /**
     * Returns a bearer jwt token for a tenant id
     * @param payload the payload of the token
     * @returns the bearer token
     * @throws an error if the tenant id is missing
     */
    private static getBearerToken;
    /**
     * Parse a bearer token and return the claims
     * @param token the bearer token
     * @returns the claims
     * @throws an error if the token is invalid
     */
    static parseBearerToken(token: string): Promise<AppInteropToken>;
    /**
     * Returns the email of the user for the given scope
     * @param scope the scope
     * @returns the email of the user
     */
    static getScopeUserEmail(scope: string): string;
    /**
     * Returns the url of the app for the given service url key (appUrl, interopUrl, metricsUrl)
     * @param appName the name of the app
     * @param key the service url key
     * @returns the url of the app
     */
    static getUrl(appName: string, key: keyof AppConfig): string;
    /**
     * Returns the url of the app
     * @param appName the name of the app
     * @returns the url of the app
     */
    static getAppUrl(appName: string): string;
    /**
     * Returns the url of the app for interop
     * @param appName the name of the app
     * @returns the url of the app for interop
     */
    static getInteropUrl(appName: string): string;
    private static getRequestConfig;
    /**
     * Sends a GraphQL request to this app.
     * @param context the context which provides the tenant id and the user's email
     * @param options a { query, variables } object
     * @returns the GraphQl response
     */
    private static sendAnyGraphqlRequest;
    private static interopError;
    /**
     * Sends a GraphQL request to this app.
     * @param context the context which provides the tenant id and the user's email
     * @param options the graphql request options
     * @returns the GraphQl response
     */
    static sendGraphqlRequest<ResponseDataT>(context: Context, options: GraphQlRequestOptions): Promise<GraphQlResponse<ExecutionResult<ResponseDataT>>>;
    /**
     * Sends a GraphQL stream request to this app.
     * @param context the context which provides the tenant id and the user's email
     * @param options the graphql request options
     * @returns the GraphQl response
     */
    static sendStreamGraphqlRequest<FirstResultT, PatchResultT>(context: Context, options: GraphQlRequestOptions): Promise<GraphQlResponse<AsyncGenerator<ExecutionResult<FirstResultT> | ExecutionPatchResult<PatchResultT>>>>;
    /**
     * Get the health info of an app
     * @param appName the name of the app
     * @returns the health info of the app
     */
    static getAppHealth(appName: string): Promise<AppHealthInfo>;
}
//# sourceMappingURL=interop-graphql-client.d.ts.map