import { Application } from '../application';
type S3Configuration = {
    bucket: string;
    folder: string;
    s3Key?: string;
    /**
     * Returns the full folder (S3 prefix) to be used for this configuration
     * When not set, the folder will be used
     */
    getFullFolder?: (app: Application) => string;
};
declare const s3Configurations: {
    /**
     * The S3 bucket used to store the reference database backups (the backups used for the generation of SQL files)
     */
    forSqlFiles: S3Configuration;
    /**
     * The S3 bucket used to store cluster-cu backups
     * Used to restore a backup of cluster-cu, from the begining of the month
     */
    clusterCuBackup: S3Configuration;
    /**
     * The S3 bucket used to store cluster-ci backups
     * Used to restore a backup of cluster-ci, after the last successful upgrade
     */
    clusterCiBackup: S3Configuration;
    /**
     * The S3 bucket used to store cluster-ci/sdmo backups
     * Used to restore a backup of cluster-ci/sdmo, after the last successful upgrade
     */
    clusterCiSdmo: S3Configuration;
    /**
     * Cluster reference
     */
    clusterReference: S3Configuration;
    /**
     * Cluster release (dev)
     */
    clusterDevRelease: S3Configuration;
    /**
     * Cluster release (qa)
     */
    clusterQaRelease: S3Configuration;
    /**
     * Glossary
     * To regenerate the S3 backup : Use https://sage-liveservices.visualstudio.com/X3%20XTREM/_build?definitionId=2311
     *    - Name of cluster to dump : glossary
     *    - Env of cluster to dump : dev-eu
     *    - withDate : unchecked
     *    - suffix : latestWorking
     */
    glossary: S3Configuration;
    /**
     * Showcase
     * To regenerate the S3 backup : Use https://sage-liveservices.visualstudio.com/X3%20XTREM/_build?definitionId=2311
     *    - Name of cluster to dump : showcase
     *    - Env of cluster to dump : dev-eu
     *    - withDate : unchecked
     *    - suffix : latestWorking
     */
    showcase: S3Configuration;
    /**
     * Shopfloor
     */
    shopfloor: S3Configuration;
    /**
     * X3 Connector
     */
    x3_connector: S3Configuration;
};
export type S3ConfigurationType = keyof typeof s3Configurations;
export declare abstract class S3Manager {
    private static _postgreSqlTools;
    /**
     * Dump the schema of an application to a S3 bucket (and returns the S3 URI of the written object)
     * @param app the application to dump the schema from
     */
    static dumpSchemaToS3Bucket(app: Application): Promise<string>;
    /**
     * Returns the full folder (S3 prefix) to be used for a given S3 configuration
     */
    private static _getFullS3Folder;
    /**
     * Resolve the version of the backup to be used to generate a new SQL file
     * @param app
     * @param version
     * @returns
     */
    private static _resolveVersionToRestoreForSqlFiles;
    /**
     * Raise an error when a specific version is not part of the available backups used to generate SQL files
     * @param app
     * @param version
     * @param versions
     */
    private static _raiseErrorVersionNotFoundForSqlFile;
    /**
     * Restore a schema version from a S3 bucket.
     * If the version is 'latest', the latest available dump for the application will be restored
     * @param app the application to restore the schema for
     * @param versionOrS3Uri either a version to restore, or a full S3 URI of the backup to restore
     * @param s3ConfigurationType the type of the S3 configuration to restore from
     * @param options.checkSingleSchema if true, will raise an error if there are more than 2 schemas in the database
     */
    static restoreSchemaFromS3Bucket(app: Application, versionOrS3Uri: string, s3ConfigurationType?: S3ConfigurationType, options?: {
        checkSingleSchema: boolean;
    }): Promise<void>;
    private static _listSchemas;
    /**
     * Upload a local db dump to a S3 bucket and returns the S3 URL of the written object
     * @param localFolder the local folder that contains the file to copy
     * @param filename the name of the file to copy
     * @param dumpKey the key to use to store the dump into S3
     */
    static uploadDumpToS3(app: Application, localFolder: string, filename: string, dumpKey: string): Promise<string>;
    /**
     * Upload a local file to a S3 bucket and returns the S3 URL of the written object
     * @param s3Configuration see s3Configurations
     * @param localFilename the (full) filename of the local file to copy
     * @param objectKey the key to use to store the file into S3
     */
    private static _uploadLocalFileToS3;
    /**
     * Returns whether an objectKey matches a db backup that can be used to generate SQL files for a given application
     */
    private static _isBackupForSqlFile;
    /**
     * List the versions hosted in the S3 bucket for a given application.
     * Will only return versions for backups that can be used to generate SQL files.
     * Will only return the versions with the same major version as the application.
     * @param app
     */
    static getAvailableVersions(app: Application): Promise<string[]>;
    /**
     * Extracts the version from a S3 key.
     * @sage/xtrem-services-main@6.0.13.zip -> 6.0.13
     *
     * @param s3Key
     * @returns
     */
    private static _getVersionFromS3Key;
    /**
     * Returns the list of backups that can be used to generate SQL files for a given application
     * @param app
     * @param bucketName
     * @returns
     */
    private static _listS3BackupsForSqlFile;
    /**
     * Download a file from a S3ObjectInfo.
     * The function will return the filename of the written file (short filename, without the folder)
     *
     * @param sourceS3Info the information of the file to download
     * @param targetLocalFolder the local folder in which the file must be written
     */
    private static _downloadDumpFileFromS3ObjectInfo;
    /**
     * Download a file from a (S3Key, S3ConfigurationType)
     * and returns the filename of the written file (short filename, without the folder)
     *
     * @param localFolder the local folder in which the file must be written
     * @param s3Key the key of the file to download
     * @param s3ConfigurationType the configuration that will be used to compute the prefix of the file to download
     */
    private static _downloadDumpFileFromS3Key;
    private static _tailLogLine;
    /**
     * Pipes a readable stream to a writable stream.
     * Transformations are applied according to the provided mappings.
     */
    private static _pipeStreamWithTransformations;
    private static _pgClientToolCommand;
    /**
     * Dumps the current schema (described by the current configuration) to a local dump file.
     * Note: if the filename is a zip file, the dump will be a valid zip archive.
     *
     * Note: the created file will be anonymized: it will contain no schema/user name, they will be replaced with tags:
     * - schema -> [SCHEMA_NAME]
     * - user -> [USER_NAME]
     *
     * @param folder the folder that contains the dump file
     * @param filename the name of the dump file
     */
    static dumpSchemaToFile(folder: string, filename: string, schemaName: string): Promise<void>;
    /**
     * Returns the mappings that should be applied to a sql file before it get restored
     * @param schemaName
     * @param dumpFullFilename
     * @param sqlCfg
     */
    private static _getMappingsForDumpFiles;
    /**
     * Restore a schema from a dump file
     * @param tempFolder the 'temp' folder where to create the 'sql.mapped.dump' file
     * @param dumpFullFilename the FULL filename of the 'sql.dump' file
     * @param schemaName
     */
    static restoreFromDumpFile(folder: string, dumpFullFilename: string, schemaName: string): Promise<void>;
    /**
     * Restore from a zip file, extract the zip and restore the dump file contained
     * @param tempFolder the 'temp' folder to use to extract the archive
     * @param archiveFullFilename the FULL filename of the archive
     */
    private static _restoreSchemaFromZip;
    /**
     * Restore from a gz dump file
     * @param tempFolder the 'temp' folder to use
     * @param dumpFullFilename the FULL filename of the dump to restore
     */
    private static _restoreSchemaFromBackup;
    /**
     * Process the table of content of a postgres dump.
     * Mainly : some entries related to extensions will be filtered and the content will be parsed to retrieve the name of schemas
     */
    private static _processTableOfContentsForRestoration;
    /**
     * Drops a bunch of schemas (if they exist) before restoring a dump
     */
    private static _dropSchemasBeforeRestoration;
    /**
     * Restores the current schema (described by the current configuration) from a local dump file.
     * Note: if the filename is a zip file, it will be extracted first.
     *
     * @param tempFolder the 'temp' folder where the archive will be extracted
     * @param fullFilename the full filename of the dump file
     */
    private static _restoreSchemaFromFile;
    /** Generic asynchronous handler for child process stdout, stderr and close event listeners */
    private static _childProcessHandler;
    private static _normalizeApplicationName;
}
export {};
//# sourceMappingURL=s3-manager.d.ts.map