import { AsyncResponse, Dict } from '@sage/xtrem-shared';
import { Context } from '../runtime';
import { Application } from './application';
import { ServiceOption, ServiceOptionStatus } from './service-option';
/**
 * Base class for service option managers
 */
export declare abstract class ServiceOptionManager {
    #private;
    readonly application: Application;
    constructor(application: Application);
    /** Returns a toposorted array of all the application's service options */
    protected get sortedServiceOptions(): ServiceOption[];
    /** Returns the service options indexed by name */
    get serviceOptionsByName(): Dict<ServiceOption>;
    /** Returns whether a service option status is enabled or not by the config file */
    static isServiceOptionStatusEnabled(status: ServiceOptionStatus): boolean;
    /**
     * Returns whether a service option is enabled or not.
     * Enabled means enabled by configuration and by tenant.
     * @param context the context
     * @param ServiceOption the service option
     */
    abstract isServiceOptionEnabled(context: Context, serviceOption: ServiceOption): AsyncResponse<boolean>;
    /**
     * Gets the array of enabled service options
     * @param context: the context
     */
    abstract getEnabledServiceOptions(context: Context): Promise<ServiceOption[]>;
    /** Low-level method to set and persist the isActive state of a service option */
    abstract setServiceOptionActiveState(context: Context, serviceOption: ServiceOption, isActive: boolean): Promise<void>;
    /** Activates a single service option */
    private activateServiceOption;
    /** Get the array of service options that activate a given option */
    private static getActivatingActiveOptions;
    /** Deactivates a single service option */
    private deactivateServiceOption;
    /** Activates an array of service options */
    activateServiceOptions(context: Context, serviceOptions: ServiceOption[]): Promise<ServiceOption[]>;
    /** Deactivates an array of service options */
    deactivateServiceOptions(context: Context, serviceOptions: ServiceOption[]): Promise<void>;
    /** Activates or deactivates a service option (and its dependencies) */
    setServiceOptionActive(context: Context, serviceOption: ServiceOption, isActive: boolean): Promise<void>;
    /**
     * Creates missing service options (existing in the code but not in the database) and update existing ones (existing both in code and in db).
     */
    createOrUpdateServiceOptions(_context: Context): Promise<void>;
    /**
     * Deletes the obsolete service options (existing in the database but not in the code).
     */
    deleteObsoleteServiceOptions(_context: Context): Promise<void>;
    /**
     * Creates or upgrades the service option states of a tenant (the context's tenant).
     * Called at the end of schema creation and upgrade.
     */
    createOrUpgradeServiceOptionStates(_context: Context, _serviceOptionActiveFlags?: Dict<boolean>): Promise<void>;
}
//# sourceMappingURL=service-option-manager.d.ts.map