/** @packageDocumentation @module runtime */
import { AnyValue, AsyncResponse } from '@sage/xtrem-async-helper';
import { Dict, RootMenuItem, SubMenuItem } from '@sage/xtrem-shared';
import { StaticThis } from '../decorators/decorator-utils';
import { NotificationTopic } from '../interop';
import { NodeFactory } from '../runtime';
import { ExternalEnumDecorator, ExternalNodeDecorator, ExternalNodeExtensionDecorator, Node } from '../ts-api';
import { Activity } from '../ts-api/activity';
import { ActivityExtension } from '../ts-api/activity-extension';
import { DataType, DataTypeOptions, EnumDataType } from '../types';
import { WorkflowStepInterface } from '../workflow/workflow-step-descriptor';
import { Application } from './application';
import { ServiceOption } from './service-option';
export interface PackageApi {
    nodes?: Dict<any>;
    nodeExtensions?: Dict<any>;
    enums?: Dict<any>;
    enumExtensions?: Dict<any>;
    serviceOptions?: Dict<ServiceOption>;
    menuItems?: Dict<SubMenuItem | RootMenuItem>;
    activities?: Dict<Activity>;
    activityExtensions?: Dict<ActivityExtension>;
    dataTypes?: Dict<DataType<AnyValue, DataTypeOptions>>;
    notificationTopics?: Dict<NotificationTopic>;
    workflowSteps?: Dict<WorkflowStepInterface>;
    startService?: (application: Application) => AsyncResponse<void>;
}
export interface PackageCreateOptions {
    buildDir?: string;
    api?: PackageApi;
    isAddOn?: boolean;
}
export interface XtremOptions {
    options?: string[];
    isHidden?: boolean;
    isReleased?: boolean;
    sqlSchemaVersion?: string;
    /**
     * Marker for packages that should be inactive by default.
     * We use a negative marker here as the default is to be active by default and we want to mark the exceptions.
     */
    isInactiveByDefault?: boolean;
    /**
     * Can the nodes in this package be extended ?
     */
    isSealed?: boolean;
    /** Is the package the main package of an application? */
    isMain?: boolean;
    /** The app name. Must be set in the main package */
    appName?: string;
    /**
     * Is the package a platform package (one of the platform/system packages)
     */
    isPlatform?: boolean;
    /**
     * Is the package a service with its own queue.
     */
    isService?: boolean;
    /**
     * The package is a service or add-on or has the hasListeners xtrem attribute supplied as true in the package.json
     */
    hasListeners?: boolean;
    /** The default SQS queue name for this package */
    queue?: string;
    isFrontEndApp?: boolean;
}
/**
 * @disabled_internal
 */
export interface PackageJsonAbout {
    name: string;
    version: string;
    author: string;
    license: string;
    description: string;
    buildStamp?: string;
    appName?: string;
}
/**
 * @disabled_internal
 */
export interface PackageJsonFile extends PackageJsonAbout {
    dependencies: Dict<string>;
    peerDependencies: Dict<string>;
    devDependencies: Dict<string>;
    /**
     * Xtrem internal attributes.
     * Every package that contains nodes must have this attribute, even declared as an empty object {}
     */
    xtrem?: XtremOptions;
    xtremPlugin?: boolean;
    main: string;
    isMainUnitTest: boolean;
}
export declare function getPackageName(packageJson: PackageJsonFile): string;
export declare function getPackageQueueName(packageJson: PackageJsonFile): string;
export declare class Package {
    #private;
    readonly application: Application;
    options: PackageCreateOptions;
    private buildDir;
    private _dir;
    get dir(): string;
    api: PackageApi;
    private _packageJson;
    factories: NodeFactory[];
    _nodeExtensions: StaticThis<Node>[];
    constructor(application: Application, options: PackageCreateOptions);
    get isSealed(): boolean;
    /**
     * @disabled_internal
     * indicate if the package is hidden (technical) or not. Hidden packages cannot be deactivated
     */
    get isHidden(): boolean;
    /**
     * Indicate if the package should be active by default.
     */
    get isActiveByDefault(): boolean;
    /**
     * indicate if the package is released.
     */
    get isReleased(): boolean;
    /**
     * indicate if the package is released.
     */
    get sqlSchemaVersion(): string | undefined;
    /**
     * indicate if the package is a main package.
     */
    get isMain(): boolean;
    /**
     * Is the packages a service which handles SQS notifications and/or messages.
     * Only service packages have SQS queues.
     */
    get isService(): boolean;
    /**
     * Is the packages from a third party extension
     */
    get isAddOn(): boolean;
    /**
     * Indicates if the package can have listeners registered
     */
    get hasListeners(): boolean;
    get queue(): string;
    get dependencies(): Dict<string>;
    get directDependencies(): Package[];
    get apiDependencies(): string[];
    /** Returns a list of all the dependencies of the current package, and their dependencies, etc.
     * @param app: application object used to retrieve the full package list.
     */
    get allDependencies(): Package[];
    private checkXtremAttribute;
    get packageJson(): PackageJsonFile;
    get name(): string;
    get version(): string;
    get packageName(): string;
    static getPackageJson(dir: string): PackageJsonFile;
    get xtremOptions(): XtremOptions;
    /** Require the code of the Application and loads the application api attribute. */
    loadDecorators(): void;
    /**
     * Returns the list of nodeExtensions created by the package
     */
    get nodeExtensions(): StaticThis<Node>[];
    get activities(): Dict<Activity>;
    get activitiesExtensions(): Dict<ActivityExtension>;
    get serviceOptions(): Dict<ServiceOption>;
    static decoratorFromFactory(factory: NodeFactory): ExternalNodeDecorator;
    getNodeDecorators(): ExternalNodeDecorator[];
    getEnumDescriptors(): ExternalEnumDecorator[];
    private enumDataTypes;
    getEnumDataTypes(): EnumDataType[];
    getExtensionDecorators(): ExternalNodeExtensionDecorator[];
    getLocalizedTitleKey(): string;
}
export declare abstract class XtremAboutHelper {
}
export declare class PackageValidationContext {
    warnings: Set<String>;
}
//# sourceMappingURL=package.d.ts.map