"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFilterObject = exports.timeFrameValues = exports.geFilterQuery = void 0;
const xtrem_date_time_1 = require("@sage/xtrem-date-time");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const lodash_1 = require("lodash");
const geFilterQuery = ({ filterType, filterValue, parameter, }) => {
    const parameterPlaceholder = `{{${filterValue}}}`;
    const escapedParameterPlaceholder = `{{${filterValue}__ESCAPED}}`;
    switch (filterType) {
        case xtrem_shared_1.CONTAINS:
            if (parameter) {
                return { _regex: escapedParameterPlaceholder, _options: 'i' };
            }
            return { _regex: (0, lodash_1.escapeRegExp)(filterValue), _options: 'i' };
        case xtrem_shared_1.STARTS_WITH:
            if (parameter) {
                return { _regex: `^${escapedParameterPlaceholder}`, _options: 'i' };
            }
            return { _regex: `^${(0, lodash_1.escapeRegExp)(filterValue)}`, _options: 'i' };
        case xtrem_shared_1.ENDS_WITH:
            if (parameter) {
                return { _regex: `${escapedParameterPlaceholder}$`, _options: 'i' };
            }
            return { _regex: `${(0, lodash_1.escapeRegExp)(filterValue)}$`, _options: 'i' };
        case xtrem_shared_1.EQUALS:
            return { _eq: parameter ? parameterPlaceholder : filterValue };
        case xtrem_shared_1.MATCHES:
            return { _eq: parameter ? parameterPlaceholder : filterValue };
        case xtrem_shared_1.NOT_EQUALS:
            return { _ne: parameter ? parameterPlaceholder : filterValue };
        case xtrem_shared_1.MULTI_NOT_EQUALS:
            return { _nin: parameter ? parameterPlaceholder : filterValue };
        case xtrem_shared_1.GREATER_THAN:
            return { _gt: parameter ? parameterPlaceholder : filterValue };
        case xtrem_shared_1.GREATER_THAN_EQUAL:
            return { _gte: parameter ? parameterPlaceholder : filterValue };
        case xtrem_shared_1.LESS_THAN:
            return { _lt: parameter ? parameterPlaceholder : filterValue };
        case xtrem_shared_1.LESS_THAN_EQUAL:
            return { _lte: parameter ? parameterPlaceholder : filterValue };
        case xtrem_shared_1.RANGE: {
            if (parameter) {
                const parameters = filterValue.toString().split(xtrem_shared_1.RANGE_DIVIDER);
                return {
                    _gte: `{{${parameters[0]}}}`,
                    _lte: `{{${parameters[1]}}}`,
                };
            }
            const values = filterValue.toString().split(xtrem_shared_1.RANGE_DIVIDER);
            return {
                _gte: values[0],
                _lte: values[1],
            };
        }
        case xtrem_shared_1.SET: {
            return {
                _in: filterValue,
            };
        }
        default:
            return {};
    }
};
exports.geFilterQuery = geFilterQuery;
exports.timeFrameValues = [
    'same-month',
    'same-year',
    'previous-month',
    'previous-year',
    'last-7-days',
    'last-30-days',
];
function getFilterObject(filters) {
    return filters?.reduce((acc, { filterType, parameter, filterValue: filterVal, id, data: { type } }) => {
        if (Array.isArray(filterVal) &&
            ((filterVal.length !== 2 && filterType === xtrem_shared_1.RANGE) ||
                (filterType !== xtrem_shared_1.SET && filterType !== xtrem_shared_1.MULTI_NOT_EQUALS && filterType !== xtrem_shared_1.RANGE))) {
            throw new Error(`Unsupported filter. Filter type: ${filterType}, filter value: ${JSON.stringify(filterVal)}`);
        }
        let filterValue = filterVal;
        let filter;
        if (filterType === 'timeFrame') {
            if (type !== 'Date' && type !== 'DateTime') {
                throw new Error(`Invalid filter value for filter type "timeFrame": expected one of ['Date', 'DateTime'], got '${type}'`);
            }
            if (!exports.timeFrameValues.includes(filterValue)) {
                throw new Error(`Invalid filter value for type "timeFrame": expected one of ${JSON.stringify(exports.timeFrameValues)}, got '${filterValue}'`);
            }
            const range = xtrem_date_time_1.DateRange.getDateRange({ date: new Date().toISOString().slice(0, 10), range: filterValue });
            filter = {
                _gte: range.start,
                _lte: range.end,
            };
        }
        else {
            if (type === 'Boolean') {
                filterValue = filterValue === true || filterValue === 'true';
            }
            if ((type === 'Date' || type === 'DateTime') && !parameter) {
                filterValue = Array.isArray(filterValue)
                    ? filterValue.map(g => g.rawValue).join(xtrem_shared_1.RANGE_DIVIDER)
                    : filterValue.rawValue;
            }
            filter = (0, exports.geFilterQuery)({
                filterType,
                filterValue,
                parameter,
            });
        }
        return (0, xtrem_shared_1.deepMerge)(acc, (0, lodash_1.set)({}, id, { _and: [filter] }));
    }, {});
}
exports.getFilterObject = getFilterObject;
//# sourceMappingURL=index.js.map