/** @module @sage/xtrem-x3-sql */
import { AnyValue, AsyncReader, AsyncResponse } from '@sage/xtrem-core';
import { DateValue } from '@sage/xtrem-date-time';
import { LoggerInterface } from '@sage/xtrem-shared';
import { Hint, PoolConfig, PoolReaderOptions, SqlColumnInfo, SqlExecuteOptions, SqlReadTableSchemaOptions, TableDefinition, TableFilter } from './types';
export type AnyLob = AnyValue;
export declare abstract class Pool<ConnectionT extends AnyValue = {}> {
    protected logger?: LoggerInterface | undefined;
    private _user;
    usePool: boolean;
    config: PoolConfig;
    protected connections: ConnectionT[];
    protected initialCounters: {
        alloc: number;
        used: number;
        discarded: number;
        warned: number;
    };
    protected counters: {
        alloc: number;
        used: number;
        discarded: number;
        warned: number;
    };
    constructor(config: PoolConfig, logger?: LoggerInterface | undefined);
    get user(): string;
    /**
     * Getter method for dialect
     */
    abstract get dialect(): string;
    abstract get nullDate(): DateValue;
    /**
     * Get the record locking clause when selecting with lock
     */
    abstract lockClause0(): string;
    /**
     * Returns string containing parameter with passed index i
     * @param i
     */
    abstract param(i: number): string;
    /**
     * Checks if passed error is a unique contraint violation
     * @param err
     */
    abstract isUniqueViolated(err: Error): boolean;
    /**
     * Checks if passed error is a locked record error
     * @param err
     */
    abstract isLocked(err: Error): boolean;
    /**
     * Checks if passed error is a index not found error
     * @param err
     */
    abstract isIndexNotFound(err: Error): boolean;
    /**
     * Checks if passed error is a table not found error
     * @param err
     */
    abstract isTableNotFound(err: Error): boolean;
    /**
     * Checks if passed error is a no data found error
     * @param err
     */
    abstract isNoRecord(err: Error): boolean;
    /**
     * Verifies that the supplied table name exists in the supplied schema
     * @param name
     * @param schema
     */
    abstract tableExists(name: string, schema: string): Promise<boolean>;
    /**
     * Returns the database data type for a tiny int
     */
    abstract tinyIntType(): string;
    /**
     * Returns the database data type for a short
     */
    abstract shortIntType(): string;
    /**
     * Returns the database data type for a integer
     */
    abstract intType(): string;
    /**
     * Returns the database data type for a decimal
     */
    abstract decimalType(): string;
    /**
     * Returns the database data type for a float
     */
    abstract floatType(): string;
    /**
     * Returns the database data type for a double
     */
    abstract doubleType(): string;
    /**
     * Returns the database data type for a string with the specified length
     * @param len
     */
    abstract stringType(len: number): string;
    /**
     * Returns the database data type for a date
     */
    abstract dateType(): string;
    /**
     * Returns the database data type for a BLOB
     */
    abstract blobType(): string;
    /**
     * Returns the relevant database syntax for an empty BLOB
     */
    abstract emptyBlob(): string;
    /**
     * Returns the database data type for a CLOB
     */
    abstract clobType(): string;
    /**
     * Returns the relevant database syntax for an empty CLOB
     */
    abstract emptyClob(): string;
    /**
     * Returns the database data type for a datetime
     */
    abstract datetimeType(): string;
    /**
     * Returns the database data type for a uuid
     */
    abstract uuidType(): string;
    /**
     * Function that returns callback function withConnection to execute supplied SQL command, callback takes a ConnectionT as a parameter
     * @param sql SQL command
     * @param args SQL arguments
     * @param opts SQL execution options
     */
    abstract execute<T extends AnyValue>(sql?: string, args?: AnyValue[], opts?: SqlExecuteOptions): Promise<T>;
    /**
     * Function that execute SQL command on the provided connection and with the provided arguments and options.
     * @param cnx Database connection
     * @param sql SQL command
     * @param args SQL arguments
     * @param opts SQL options
     */
    abstract execute<T extends AnyValue>(cnx?: ConnectionT, sql?: string, args?: AnyValue[], opts?: SqlExecuteOptions): Promise<T>;
    /**
     * Release all database connections
     */
    abstract release(): void;
    /**
     * Returns hint clause for supplied hint object, if Hint is null return default (only for databases that support SQL hints)
     * @param hint
     * @param def
     */
    abstract hintClause0(hint: Hint, def: boolean): string;
    /**
     * Returns hint clause for supplied hint object, if Hint is null return default (only for databases that support SQL hints)
     * @param hint
     * @param def
     */
    abstract hintClause1(hint: Hint, def: boolean): string;
    /**
     * Get the record locking clause when selecting for update
     */
    abstract lockClause1(): string;
    /**
     * Returns the clause to select the first x number of records (top of query)
     * @param first
     */
    abstract firstClause0(first: number): string;
    /**
     * Returns the clause to select the first x number of records (bottom of query)
     * @param first
     */
    abstract firstClause1(first: number): string;
    /**
     * Returns SQL command to convert provide parameter from a ROWID to a character string
     * @param str
     */
    abstract rowIdGet(str: string): string;
    /**
     * Returns the SQL command to get the length of a character string
     */
    abstract lengthString(): string;
    /**
     * Returns the relevant SQL command to convert the passed columnName to a character string in a select clause, aliasing with the provide alias
     * @param columnName name of column to convert
     * @param length length of string
     */
    asString(columnName: string): string;
    /**
     * Returns the SQL function to retrieve the modulus for the provided dividend and divisor
     * @param dividend
     * @param divisor
     */
    abstract modString(dividend: string, divisor: string): string;
    /**
     * Returns the SQL command for the substring of the provided str, from the beg position to the end position
     * @param str
     * @param begin
     * @param end
     */
    abstract subString(str: string, begin: string, end: string): string;
    /**
     * Returns the SQL command to convert the provide arg value to a ROWID
     * @param arg
     */
    abstract rowIdPut(arg: string): string;
    /**
     * Returns the SQL command to get the next sequence number value for the provided table name
     * @param arg
     */
    abstract uniqid(arg: string): string;
    /**
     * Returns the column info for the provided SQL select statement in the provided schema
     * @param sql
     * @param schema
     */
    abstract describeQuery(sql: string, schema: string): Promise<SqlColumnInfo[]>;
    /**
     * Function to execute passed callback function passing the current allocated connection to the callback and then releasing the connection
     * @param body
     */
    withConnection<T extends AnyValue>(body: (cnx: ConnectionT) => AsyncResponse<T>): Promise<T>;
    /**
     * Return the full qualified table name based on the passed table definition
     * @param def
     */
    abstract getFullTableDefName(def: TableDefinition): string;
    /**
     *  Return the full qualified table name based on the passed schema, table name and alias
     * @param schemaName
     * @param tableName
     * @param alias
     */
    abstract getFullTableName(schemaName: string, tableName: string, alias?: string): string;
    /**
     * Execute a piece of sql in a transaction and either commit or rollback the transaction based on the options passed
     * @param sql
     * @param options
     */
    abstract executeWithTransaction<T extends AnyValue>(sql: string, options: {
        rollback: boolean;
        args?: any[];
    }): Promise<T>;
    /**
     * Escapes the string passed in for the relevant database
     */
    abstract escape(name: string): string;
    /**
     * Checks the connection passed in, if not connected then returns undefined
     * else returns the connection
     * @param cnx
     */
    protected abstract checkConnection(cnx: ConnectionT | undefined): ConnectionT | undefined;
    /**
     * Closes all connections
     * TODO: Review: this method is not called anywhere
     */
    closeConnections(): Promise<void>;
    /**
     * Gets the current database connection, if one does not exists then a connection is created
     */
    allocConnection(): Promise<ConnectionT>;
    /**
     * Closes supplied database connection
     * @param cnx
     * @param err
     */
    releaseConnection(cnx: ConnectionT, err?: Error): Promise<undefined>;
    /**
     * Creates a new connection
     */
    protected abstract createConnection(): Promise<ConnectionT>;
    /**
     * Closes supplied connection
     * @param cnx
     */
    protected abstract closeConnection(cnx: ConnectionT): AsyncResponse<void>;
    /**
     * Makes a backup of database
     * @param dbName
     * @param filename
     */
    abstract backupDatabase(dbName: string, filename: string): Promise<void>;
    /**
     * restores a database from a backup
     * @param dbName
     * @param filename
     */
    abstract restoreDatabase(dbName: string, filename: string): Promise<void>;
    /**
     * Returns all table definitions for the supplied schema
     * @param schemaName
     */
    abstract readTables(schemaName: string): Promise<TableDefinition[]>;
    /**
     * Returns the relevant SQL command to convert date colunm to an ISO character string - YYYY-MM-DD
     * @param dateColumn
     */
    abstract dateToChar(dateColumn: string, format?: string, dateTime?: boolean): string;
    /**
     * Returns the relevant SQL command to convert date colunm to an ISO character string - YYYY-MM-DD
     * @param dateColumn
     */
    abstract charToDate(dateColumn: string, dateTime?: boolean): string;
    /**
     * Returns the relevant SQL command to convert a colunm to a character string
     * @param column
     */
    abstract toChar(column: string): string;
    /**
     * Returns the relevant SQL concatenator to convert a colunm to a character string
     *
     */
    abstract charConcatenator(): string;
    /**
     * Returns the relevant SQL command for regex or like
     * @param column
     * @param regex
     * @param options
     */
    abstract regexOrLike(column: string, type: string, regex: string, options?: string): string;
    /**
     * Returns the relevant SQL command to limit the clause
     * @param limit
     */
    abstract limitClause(limit: number): string;
    /**
     * Returns the relevant SQL command for the fetch only clause
     * @param limit
     */
    abstract fetchOnlyClause(limit: number): string;
    /**
     * Returns the relevant SQL command for the fetch only clause
     * @param colName
     * @param name
     */
    abstract driverFunctions(colName: string, name: 'year' | 'month' | 'day'): string;
    /**
     * Returns the string passed in quotes
     * @param s
     */
    quote(s: string): string;
    /**
     * Returns the regex passed resolved for SQL
     * @param s
     * @param str
     */
    sanitizeSqlPattern(s: string, str: string): string;
    /**
     * Returns the regex passed resolved for SQL
     * @param str
     */
    regexToSql(str: string): string;
    /**
     * Returns the relevant SQL command to cast a column to a type
     * @param columnName
     * @param type
     */
    cast(columnName: string, type: string): string;
    /**
     * Tests regular expression
     * @param arg
     * @param param
     * @param flags
     */
    regexpLike?(arg: string, param: string, flags: string): string;
    /**
     * Returns table definition for the passed in schema and table
     * @param schemaName
     * @param tableName
     * @param trackerOrOptions
     */
    abstract readTableSchema(schemaName: string, tableName: string, trackerOrOptions?: SqlReadTableSchemaOptions): Promise<TableDefinition>;
    /**
     * Returns the SQL clause to extract year, month and day from a date/datetime column (sth like YYYY-MM-DD ou YYYY/MM/DD)
     * @param colName
     * @param forGroupBy whether the clause will be used in the 'groupBy' part of a query
     */
    abstract extractYearAndMonthAndDayFromDate(colName: string, forGroupBy: boolean): string;
    /**
     * Returns the SQL clause to extract year and month from a date/datetime column (sth like YYYY-MM-01 ou YYYY/MM/01)
     * @param colName
     * @param forGroupBy whether the clause will be used in the 'groupBy' part of a query
     */
    abstract extractYearAndMonthFromDate(colName: string, forGroupBy: boolean): string;
    /**
     * Returns the SQL clause to extract year from a date/datetime column (sth like YYYY-01-01 ou YYYY/01/01)
     * @param colName
     * @param forGroupBy whether the clause will be used in the 'groupBy' part of a query
     */
    abstract extractYearFromDate(colName: string, forGroupBy: boolean): string;
    /**
     * Returns a reader that could be used to read all the records of the table
     * @param tableDef is an object that has been created by getTableDef() or readTables() and updated by readTableSchema()
     * @param filters
     */
    createTableReader<T extends AnyValue>(tableDef: TableDefinition, filters: TableFilter[]): AsyncReader<T>;
    /**
     * Creates a reader for the provided connection and SQL statement
     * @param cnx
     * @param sql
     * @param args
     * @param opts
     */
    abstract createReader<T extends AnyValue>(cnx: ConnectionT, sql: string, args?: AnyValue[], opts?: PoolReaderOptions): AsyncReader<T>;
}
//# sourceMappingURL=pool.d.ts.map