import { getInternalPathFromExternal } from '@sage/xtrem-ui-components';
import axios from 'axios';
import { getUserDetails } from '../service/auth-service';
import { fetchChatbotConfiguration } from '../service/chatbot-service';
import { fetchMetaData, fetchNotifications, fetchStandaloneConfig, notificationDelete, notificationMarkAllRead, notificationMarkRead, notificationMarkUnread, } from '../service/metadata-service';
import { initiateSocketConnection } from '../service/socket-service';
import { getNewRelicApi, getPendoApi } from '../service/telemetry-service';
import { fetchTenants } from '../service/tenants-service';
import { ActionType } from './action-types';
import { isDevMode } from './store';
export const setPath = (path) => ({ type: ActionType.SetPath, value: path });
export const setSitemap = (sitemap) => ({ type: ActionType.SetSitemap, value: sitemap });
export const setNavigationOpen = (isOpen) => ({ type: ActionType.SetNavigationOpen, value: isOpen });
export const setChatbotOpen = (isOpen) => {
    getPendoApi()?.track(isOpen ? 'chatbotOpened' : 'chatbotClosed');
    window.GmsChatUi.setDisplayState(isOpen ? 'popup-right' : 'hidden');
    return { type: ActionType.SetChatbotOpen, value: isOpen };
};
export const setNotificationCenterOpen = (isOpen) => ({
    type: ActionType.SetNotificationCenterOpen,
    value: isOpen,
});
export const setMenuItems = (menuItems) => ({ type: ActionType.SetMenuItems, value: menuItems });
export const setTenantsList = (tenantsList) => ({
    type: ActionType.SetTenantsList,
    value: tenantsList,
});
export const actionStub = () => {
    throw new Error('Action stub called, did you forget to override the the action?');
};
export const onDirtyStatusChange = (isApplicationDirty, preNavigationConfirmation) => ({
    type: ActionType.SetDirtyState,
    value: {
        isApplicationDirty,
        preNavigationConfirmation,
    },
});
export const onInternalNavigationChange = (path, doDirtyCheck = false) => async (dispatch, getState) => {
    const state = getState();
    if (doDirtyCheck && state.isApplicationDirty && state.preNavigationConfirmation) {
        try {
            await state.preNavigationConfirmation();
            dispatch({ type: ActionType.SetDirtyState, value: null });
        }
        catch {
            return Promise.reject();
        }
    }
    const pathDifferent = path !== state.path;
    if (pathDifferent) {
        const nextUrl = `/${path}`;
        window.history.pushState({ id: path }, `XTreeM ${path}`, nextUrl);
        dispatch(setPath(path));
        if (path) {
            dispatch(setSitemap(state.sitemap));
        }
    }
    return Promise.resolve();
};
export const goHome = () => async (dispatch, getState) => {
    const path = '';
    const pathDifferent = path !== getState().path;
    window.document.title = `Sage ${getState().config?.productName || ''}`;
    if (pathDifferent) {
        await dispatch(onInternalNavigationChange('', true));
    }
};
export const updateNavStateBasedOnLocation = () => (dispatch, getState) => {
    const path = getInternalPathFromExternal();
    const pathDifferent = path !== getState().path;
    if (pathDifferent) {
        dispatch(setPath(path));
    }
};
export const init = () => async (dispatch) => {
    let loginServiceUrl = '';
    try {
        const response = await axios.get('/login-service');
        loginServiceUrl = response.data;
        const userDetailsFromToken = getUserDetails();
        if (!userDetailsFromToken.id && !loginServiceUrl) {
            // eslint-disable-next-line no-console
            console.warn('Security config is missing, user context could not be identified.');
            // eslint-disable-next-line no-console
            console.warn('Please add a security config file for your deployment in order to access this page.');
        }
        else {
            if (userDetailsFromToken.locale) {
                document.documentElement.setAttribute('lang', userDetailsFromToken.locale.substr(0, 2));
            }
            const standaloneConfig = await fetchStandaloneConfig(userDetailsFromToken);
            dispatch({ type: ActionType.SetConfig, value: standaloneConfig });
            dispatch({ type: ActionType.SetLoginService, value: loginServiceUrl });
            initiateSocketConnection(standaloneConfig.app);
            const metadata = await fetchMetaData(userDetailsFromToken.locale);
            const tenantsList = await fetchTenants(loginServiceUrl);
            const chatbotConfig = await fetchChatbotConfiguration(loginServiceUrl);
            if (isDevMode()) {
                // eslint-disable-next-line no-console
                console.log('Chatbot config:');
                // eslint-disable-next-line no-console
                console.log(chatbotConfig);
            }
            // Bitwise variable, convert it to binary so individual parts can be
            const consentOptions = Number(userDetailsFromToken.pref || 0)
                .toString(2)
                .padStart(3, '0');
            const { kind = '', countryCode = '', subscriptionType = '', tenantId = '', tenantName = '', } = tenantsList.tenantList.find(tenant => tenant.current === true) || {};
            let pendoHelpOnClickOutsideHandler = null;
            getPendoApi()?.initialize({
                visitor: {
                    hasanalyticsconsent: !!Number(consentOptions.charAt(1)),
                    hasmarketingconsent: !!Number(consentOptions.charAt(2)),
                    id: metadata.userInfo.uniqueUserId,
                    localeCode: userDetailsFromToken.locale,
                },
                account: {
                    countryCode,
                    kind,
                    id: metadata.userInfo.uniqueTenantId,
                    pendoClusterTag: standaloneConfig.pendoClusterTag || null,
                    subscriptionType,
                    tenantId,
                    tenantName,
                },
                guides: {
                    // @ts-expect-error: @types/pendo-io-browser typings are incomplete
                    globalScripts: [
                        {
                            script(_, guide) {
                                pendoHelpOnClickOutsideHandler = e => {
                                    if (e?.target &&
                                        !e.target.closest('.xe-header #pendo-help') &&
                                        !e.target.closest('#pendo-resource-center-container')) {
                                        getPendoApi()?.onGuideDismissed(guide.id);
                                    }
                                };
                                // @ts-expect-error: @types/pendo-io-browser typings are incomplete
                                getPendoApi()?.attachEvent(document, 'click', pendoHelpOnClickOutsideHandler);
                            },
                            unmounted() {
                                // @ts-expect-error: @types/pendo-io-browser typings are incomplete
                                getPendoApi()?.detachEvent(document, 'click', pendoHelpOnClickOutsideHandler);
                                pendoHelpOnClickOutsideHandler = null;
                            },
                            test(step) {
                                return step?.element?.matches('.xe-header #pendo-help');
                            },
                        },
                    ],
                },
            });
            dispatch({
                type: ActionType.SetMetadata,
                value: {
                    translations: metadata.translations,
                    notifications: metadata.notifications ? metadata.notifications : [],
                    sitemap: metadata.sitemap,
                    userDetails: { ...userDetailsFromToken, ...metadata.userInfo },
                    tenantsList,
                    chatbotConfig,
                },
            });
        }
    }
    catch (e) {
        // eslint-disable-next-line no-console
        console.error(e);
        getNewRelicApi()?.noticeError(e);
        if (loginServiceUrl) {
            window.location.replace(`${loginServiceUrl}/login?fromUrl=${document.location.href}`);
        }
    }
};
export const setPageTitle = (title) => (dispatch, getState) => dispatch({ type: ActionType.SetPageTitle, value: title || `Sage ${getState().config?.productName || ''}` });
export const refreshNotifications = () => async (dispatch, getState) => {
    const locale = getState()?.user?.locale || 'en-US';
    const notifications = await fetchNotifications(locale);
    dispatch({ type: ActionType.SetNotifications, value: notifications });
};
export const markNotificationAsRead = (notificationId) => async (_, getState) => {
    getPendoApi()?.track('markSingleNotificationsRead');
    try {
        const locale = getState()?.user?.locale || 'en-US';
        await notificationMarkRead(notificationId, locale);
    }
    catch (e) {
        // eslint-disable-next-line no-console
        console.error(e);
        getNewRelicApi()?.noticeError(e);
        return Promise.reject();
    }
    return Promise.resolve();
};
export const markAllNotificationsAsRead = () => async (_, getState) => {
    try {
        getPendoApi()?.track('markAllNotificationsRead');
        const locale = getState()?.user?.locale || 'en-US';
        await notificationMarkAllRead(locale);
    }
    catch (e) {
        // eslint-disable-next-line no-console
        console.error(e);
        getNewRelicApi()?.noticeError(e);
        return Promise.reject();
    }
    return Promise.resolve();
};
export const markNotificationAsUnread = (notificationId) => async (_, getState) => {
    try {
        getPendoApi()?.track('markAllNotificationsUnread');
        const locale = getState()?.user?.locale || 'en-US';
        await notificationMarkUnread(notificationId, locale);
    }
    catch (e) {
        // eslint-disable-next-line no-console
        console.error(e);
        getNewRelicApi()?.noticeError(e);
        return Promise.reject();
    }
    return Promise.resolve();
};
export const deleteNotification = (notificationId) => async (_, getState) => {
    try {
        getPendoApi()?.track('deleteNotification');
        const locale = getState()?.user?.locale || document.documentElement.getAttribute('lang') || 'en-US';
        await notificationDelete(notificationId, locale);
    }
    catch (e) {
        // eslint-disable-next-line no-console
        console.error(e);
        getNewRelicApi()?.noticeError(e);
        return Promise.reject();
    }
    return Promise.resolve();
};
//# sourceMappingURL=actions.js.map