import * as tokens from '@sage/design-tokens/js/base/common';
import { ActionPopover, ActionPopoverDivider, ActionPopoverItem } from 'carbon-react/esm/components/action-popover';
import ButtonMajor from 'carbon-react/esm/components/button';
import ButtonMinor from 'carbon-react/esm/components/button-minor';
import Icon from 'carbon-react/esm/components/icon';
import IconButton from 'carbon-react/esm/components/icon-button';
import formatDistance from 'date-fns/formatDistance';
import arSa from 'date-fns/locale/ar-SA';
import deDe from 'date-fns/locale/de';
import enGb from 'date-fns/locale/en-GB';
import enUs from 'date-fns/locale/en-US';
import esEs from 'date-fns/locale/es';
import frFr from 'date-fns/locale/fr';
import itIT from 'date-fns/locale/it';
import plPl from 'date-fns/locale/pl';
import ptPt from 'date-fns/locale/pt';
import ptBr from 'date-fns/locale/pt-BR';
import zhCn from 'date-fns/locale/zh-CN';
import { kebabCase, noop } from 'lodash';
import React from 'react';
import { connect } from 'react-redux';
import { useExternalLinkConfirmation } from '../../hooks/use-external-link-confirmation';
import { isSkipExternalLinkConfirmation } from '../../hooks/use-local-storage';
import { localize } from '../../service/standalone-i18n-service';
import * as actions from '../../standalone-redux/actions';
import { getNotificationActions } from './get-notification-actions';
import { isDownloadUrl, isExternalUrl } from './is-external-url';
import './notification-center.scss';
import { ExternalLinkConfirmationDialog } from '../external-link-confirmation/exterrnal-link-confirmation-dialog';
const dateLocales = {
    'en-GB': enGb,
    'en-US': enUs,
    'es-ES': esEs,
    'pl-PL': plPl,
    'zh-CN': zhCn,
    'fr-FR': frFr,
    'pt-PT': ptPt,
    'pt-BR': ptBr,
    'ar-SA': arSa,
    'de-DE': deDe,
    'it-IT': itIT,
};
const levels = {
    success: tokens.colorsSemanticPositive500,
    warning: tokens.colorsSemanticCaution400,
    error: tokens.colorsSemanticNegative450,
    info: tokens.colorsSemanticInfo500,
    approval: tokens.colorsSemanticFocus500,
};
export function NotificationCenterItem({ deleteNotification, locale, markNotificationAsRead, markNotificationAsUnread, notification, onClose, onInternalNavigationChange, setNotificationCenterOpen, translations, }) {
    const { getConfirmation, isOpen, onCancel, onConfirm } = useExternalLinkConfirmation({
        onCancel: noop,
        onConfirm: noop,
    });
    const startDate = typeof notification._createStamp === 'string'
        ? new Date(parseInt(notification._createStamp, 10))
        : notification._createStamp;
    const distance = formatDistance(startDate, new Date(), {
        addSuffix: true,
        locale: dateLocales[locale] ? dateLocales[locale] : enUs,
    });
    const getNotificationActionTestId = (title) => {
        return `xe-notification-action-${kebabCase(title)}`;
    };
    const notificationActions = getNotificationActions(notification.isRead, notification.actions, localize('@sage/xtrem-standalone/notification-action-mark-read', 'Mark as read', {}, translations, locale), localize('@sage/xtrem-standalone/notification-action-mark-unread', 'Mark as unread', {}, translations, locale));
    const markUnreadNotificationAsRead = async () => {
        if (!notification.isRead) {
            await markNotificationAsRead(notification._id);
        }
    };
    const onDeleteNotification = async () => {
        await deleteNotification(notification._id);
    };
    const onHandleButtonAction = (action) => async () => {
        if (action._id === 'notification-action-mark-read') {
            await markNotificationAsRead(notification._id);
            return;
        }
        if (action._id === 'notification-action-mark-unread') {
            await markNotificationAsUnread(notification._id);
        }
    };
    const renderTopRightButton = () => {
        if (onClose) {
            return (React.createElement(IconButton, { "aria-label": localize('@sage/xtrem-standalone/notification-action-close', 'Close', {}, translations, locale), "data-testid": "xe-notification-action-close", onClick: onClose },
                React.createElement(Icon, { tooltipMessage: localize('@sage/xtrem-standalone/notification-action-close', 'Close', {}, translations, locale), type: "close" })));
        }
        return (React.createElement(IconButton, { "aria-label": localize('@sage/xtrem-standalone/notification-action-delete', 'Delete', {}, translations, locale), "data-testid": "xe-notification-action-delete", onClick: onDeleteNotification },
            React.createElement(Icon, { color: tokens.colorsSemanticNegative500, tooltipMessage: localize('@sage/xtrem-standalone/notification-action-delete', 'Delete', {}, translations, locale), type: "delete" })));
    };
    const onHandleAnchorAction = (action, isExternalLink, isDownloadLink, href) => async (event) => {
        if (event.ctrlKey || event.metaKey) {
            await markUnreadNotificationAsRead();
            return;
        }
        event.preventDefault();
        if (isExternalLink || isDownloadLink) {
            await markUnreadNotificationAsRead();
            try {
                if (!isSkipExternalLinkConfirmation() && !isDownloadLink) {
                    await getConfirmation();
                }
                window.open(href, '_blank', 'noreferrer');
            }
            catch {
                // eslint-disable-next-line no-console
                console.warn(`Navigation to external link "${href}" cancelled by user.`);
            }
            return;
        }
        await markUnreadNotificationAsRead();
        onInternalNavigationChange(action.link);
        setNotificationCenterOpen(false);
    };
    return (React.createElement("div", { className: `xe-notification-card xe-notification-card--${notification.isRead ? 'read' : 'unread'}`, "data-testid": "xe-notification-card", "data-status": notification.isRead ? 'read' : 'unread' },
        React.createElement("div", { className: "xe-notification-icon-container" },
            React.createElement(Icon, { className: "xe-notification-icon", "data-testid": "xe-notification-icon", type: notification.icon, bg: levels[notification.level], color: tokens.colorsYang100, role: "img" })),
        React.createElement("div", { className: "xe-notification-body-container" },
            React.createElement("div", { className: "xe-notification-body-header", "data-testid": "xe-notification-body-header" },
                React.createElement("span", { "data-testid": "xe-notification-time-ago", className: "xe-notification-time-ago" }, distance),
                !onClose && notificationActions.menu.length ? (React.createElement(ActionPopover, { "data-testid": "xe-notification-menu-actions" },
                    notificationActions.menu.map((action) => {
                        if (action._id === 'notification-action-mark-read' ||
                            action._id === 'notification-action-mark-unread') {
                            return (React.createElement(ActionPopoverItem, { "data-testid": getNotificationActionTestId(action.title), icon: action.icon || 'none', key: action._id, onClick: onHandleButtonAction(action) }, action.title));
                        }
                        const isExternalLink = isExternalUrl(action.link);
                        const isDownloadLink = isDownloadUrl(action.link);
                        return (React.createElement(ActionPopoverItem, { "data-testid": getNotificationActionTestId(action.title), href: action.link, icon: action.icon || 'none', key: action._id, onClick: onHandleAnchorAction(action, isExternalLink, isDownloadLink, action.link) }, action.title));
                    }),
                    notificationActions.menu.length ? React.createElement(ActionPopoverDivider, null) : null,
                    React.createElement(ActionPopoverItem, { "data-testid": "xe-notification-action-delete", "data-is-destructive": "true", icon: "delete", onClick: onDeleteNotification }, localize('@sage/xtrem-standalone/notification-action-delete', 'Delete', {}, translations, locale)))) : (renderTopRightButton())),
            React.createElement("div", { className: "xe-notification-body-content" },
                React.createElement("h1", { className: "xe-notification-body-title", "data-testid": "xe-notification-title" }, notification.title),
                React.createElement("p", { className: "xe-notification-body-description", "data-testid": "xe-notification-description" }, notification.description),
                React.createElement("div", { className: "xe-notification-body-actions", "data-testid": "xe-notification-actions" }, notificationActions.buttons.map((action) => {
                    if (action._id === 'notification-action-mark-read' ||
                        action._id === 'notification-action-mark-unread') {
                        // Rendering mark as read/unread as buttons to avoid re-renders.
                        return (React.createElement(ButtonMinor, { buttonType: "tertiary", "data-testid": getNotificationActionTestId(action.title), key: action._id, onClick: onHandleButtonAction(action) }, action.title));
                    }
                    const isExternalLink = isExternalUrl(action.link);
                    const isDownloadLink = isDownloadUrl(action.link);
                    const buttonProps = {
                        key: action._id,
                        'data-testid': getNotificationActionTestId(action.title),
                        href: action.link,
                        target: isExternalLink || isDownloadLink ? '_blank' : '_self',
                        onClick: onHandleAnchorAction(action, isExternalLink, isDownloadLink, action.link),
                        children: (React.createElement(React.Fragment, null,
                            action.title,
                            isExternalLink && React.createElement(Icon, { type: "link" }),
                            isDownloadLink && React.createElement(Icon, { type: "download" }))),
                    };
                    return action.style === 'primary' || action.style === 'secondary' ? (React.createElement(ButtonMajor, { buttonType: action.style, ...buttonProps })) : (React.createElement(ButtonMinor, { buttonType: "tertiary", ...buttonProps }));
                })))),
        isSkipExternalLinkConfirmation() ? null : (React.createElement(ExternalLinkConfirmationDialog, { isOpen: isOpen, locale: locale, onCancel: onCancel, onConfirm: onConfirm, translations: translations }))));
}
const mapStateToProps = (state, props) => ({
    ...props,
    deleteNotification: actions.actionStub,
    locale: state.user?.locale || 'en-US',
    notification: state.notifications.find((notification) => notification._id === props.id),
    markNotificationAsRead: actions.actionStub,
    markNotificationAsUnread: actions.actionStub,
    onInternalNavigationChange: actions.actionStub,
    setNotificationCenterOpen: actions.actionStub,
    translations: state.translations,
});
const mapDispatchToProps = (dispatch) => ({
    deleteNotification: (id) => dispatch(actions.deleteNotification(id)),
    markNotificationAsRead: (id) => dispatch(actions.markNotificationAsRead(id)),
    markNotificationAsUnread: (id) => dispatch(actions.markNotificationAsUnread(id)),
    onInternalNavigationChange: (path) => dispatch(actions.onInternalNavigationChange(path, true)),
    setNotificationCenterOpen: (isOpen) => dispatch(actions.setNotificationCenterOpen(isOpen)),
});
export const ConnectedNotificationCenterItem = connect(mapStateToProps, mapDispatchToProps)(NotificationCenterItem);
//# sourceMappingURL=notification-center-item.js.map