import { GraphApi } from '@sage/x3-master-data-api';
import { Destination, User } from '@sage/x3-system-api';
import { getDestinations } from '@sage/x3-system/lib/client-functions/user-print-destinations-values';
import * as ui from '@sage/xtrem-ui';
@ui.decorators.sticker<MobilePrintDestinations>({
    title: 'Print destinations',
    icon: 'print',
    category: 'mobile',
    isActive() {
        return true;
    },
    async onLoad() {
        let sessionLabelDestination = this.$.storage.get('mobile-label-destination');
        let sessionDocumentDestination = this.$.storage.get('mobile-document-destination');

        if (!sessionLabelDestination && !sessionDocumentDestination) {
            const user = await this.readUser(this.$.userCode);
            if (!user) {
                return;
            }
            const defaultDestinations = await getDestinations(user);
            if (!defaultDestinations || defaultDestinations.length === 0 || !defaultDestinations[0].destination) {
                return;
            }
            const userDefaultDestination = defaultDestinations[0];

            if (['printer', 'zplPrinter'].includes(userDefaultDestination.destination)) {
                this.$.storage.set('mobile-label-destination', userDefaultDestination.code);
                this.$.storage.set('mobile-document-destination', userDefaultDestination.code);
                sessionLabelDestination = sessionDocumentDestination = userDefaultDestination.code;
            }
        }
        this.labelDestination.value = { code: sessionLabelDestination };
        this.documentDestination.value = { code: sessionDocumentDestination };
    },
    businessActions() {
        return [this.saveButton];
    },
})
export class MobilePrintDestinations extends ui.Sticker<GraphApi> {
    @ui.decorators.section<MobilePrintDestinations>({
        title: 'Print destinations',
    })
    section: ui.containers.Section;

    @ui.decorators.block<MobilePrintDestinations>({
        parent() {
            return this.section;
        },
    })
    block: ui.containers.Block;

    @ui.decorators.referenceField<MobilePrintDestinations, Destination>({
        node: '@sage/x3-system/Destination',
        filter() {
            return {
                isActive: true,
                destination: { _in: ['printer', 'zplPrinter'] },
            };
        },
        valueField: 'code',
        columns: [
            ui.nestedFields.text({ bind: 'code', title: 'Code', canFilter: true }),
            ui.nestedFields.text({ bind: 'description', title: 'Description', canFilter: true }),
            ui.nestedFields.text({ bind: 'destination', title: 'Type', canFilter: true }),
            ui.nestedFields.text({ bind: 'printerName', title: 'Name', canFilter: true }),
            ui.nestedFields.checkbox({ bind: 'isActive', isHidden: true }),
            ui.nestedFields.text({ bind: 'destination', isHidden: true }),
        ],
        isFullWidth: true,
        isAutoSelectEnabled: true,
        title: 'Label destination',
        helperTextField: 'description',
        canFilter: false,
        parent() {
            return this.block;
        },
        onChange() {
            this.saveButton.isDisabled = !this.labelDestination.value && !this.documentDestination.value;
        },
    })
    labelDestination: ui.fields.Reference;

    @ui.decorators.referenceField<MobilePrintDestinations, Destination>({
        node: '@sage/x3-system/Destination',
        filter() {
            return {
                isActive: true,
                destination: { _in: ['printer', 'zplPrinter'] },
            };
        },
        valueField: 'code',
        columns: [
            ui.nestedFields.text({ bind: 'code', title: 'Code', canFilter: true }),
            ui.nestedFields.text({ bind: 'description', title: 'Description', canFilter: true }),
            ui.nestedFields.text({ bind: 'destination', title: 'Type', canFilter: true }),
            ui.nestedFields.text({ bind: 'printerName', title: 'Name', canFilter: true }),
            ui.nestedFields.checkbox({ bind: 'isActive', isHidden: true }),
            ui.nestedFields.text({ bind: 'destination', isHidden: true }),
        ],
        isFullWidth: true,
        isAutoSelectEnabled: true,
        title: 'Document destination',
        helperTextField: 'description',
        canFilter: false,
        parent() {
            return this.block;
        },
        onChange() {
            this.saveButton.isDisabled = !this.labelDestination.value && !this.documentDestination.value;
        },
    })
    documentDestination: ui.fields.Reference;

    @ui.decorators.pageAction<MobilePrintDestinations>({
        title: 'Set destinations',
        buttonType: 'primary',
        isTransient: true,
        isDisabled: true,
        onClick() {
            if (this.labelDestination.value?.code) {
                this.$.storage.set('mobile-label-destination', this.labelDestination.value.code); // Save the destination in the session storage
            }
            if (this.documentDestination.value?.code) {
                this.$.storage.set('mobile-document-destination', this.documentDestination.value.code); // Save the destination in the session storage
            }
            this.$.finish(); // This is used to automatically close this sticker
        },
    })
    saveButton: ui.PageAction;

    async readUser(userCode: string | null): Promise<User | null> {
        const user = (await this.$.graph
            .node('@sage/x3-system/User')
            .read(
                {
                    code: true,
                    name: true,
                    userDestinations: {
                        query: {
                            edges: {
                                node: {
                                    code: true,
                                    destination: {
                                        code: true,
                                        description: true,
                                        destination: true,
                                    },
                                },
                            },
                        },
                    },
                },
                `${userCode?.toUpperCase()}`,
            )
            .execute()) as User | null;
        return user;
    }
}
