"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.querySelector = exports.edgesSelector = exports.convertPagingOptions = exports.extractEdges = exports.withoutEdges = void 0;
/**
 * Recursively transforms "edges":
 *
 * Array<{edges: { node: T }}> => T[]
 *
 * "query", "queryAggregate" & "readAggregate" are PRESERVED.
 *
 * @export
 * @template T
 * @param {Edges<T>} edges
 * @returns {WithoutEdges<T>[]}
 */
function withoutEdges(edges) {
    const transform = (val) => {
        if (!val || typeof val !== 'object')
            return val;
        // if (val.query) return transform(val.query);
        if (Array.isArray(val))
            return val.map(transform);
        if (Array.isArray(val.edges))
            return val.edges.map((v) => transform(v.node));
        return Object.keys(val).reduce((r, k) => {
            r[k] = transform(val[k]);
            return r;
        }, {});
    };
    return transform(edges);
}
exports.withoutEdges = withoutEdges;
/**
 * Recursively extract "edges" from queries:
 *
 * * {
 *     query: Array<{edges: { node: T }}>;
 *     readAggregate: AggregateResultValues<T>;     =>      T[]
 *     queryAggregate: AggregateEdges<T>;
 * }
 *
 * "query", "queryAggregate" & "readAggregate" are REMOVED.
 * Use "withoutEdges" if you want to keep them.
 * @export
 * @template T
 * @param {Edges<T>} edges
 * @returns {ExtractEdges<T>[]}
 */
function extractEdges(edges) {
    const transform = (val) => {
        if (!val || typeof val !== 'object')
            return val;
        if (val.query)
            return transform(val.query);
        if (Array.isArray(val))
            return val.map(transform);
        if (Array.isArray(val.edges))
            return val.edges.map((v) => transform(v.node));
        return Object.keys(val).reduce((r, k) => {
            r[k] = transform(val[k]);
            return r;
        }, {});
    };
    return transform(edges);
}
exports.extractEdges = extractEdges;
/** @internal */
function convertPagingOptions(args) {
    const converted = {};
    if (args.first)
        converted.first = args.first;
    if (args.after)
        converted.after = args.after;
    if (args.last)
        converted.last = args.last;
    if (args.before)
        converted.before = args.before;
    if (args.filter)
        converted.filter = JSON.stringify(args.filter);
    if (args.orderBy)
        converted.orderBy = JSON.stringify(args.orderBy);
    return Object.keys(converted).length > 0 ? converted : undefined;
}
exports.convertPagingOptions = convertPagingOptions;
function edgesSelector(selector, args) {
    return {
        __args: convertPagingOptions(args || {}),
        edges: { node: selector, cursor: true },
        pageInfo: { hasNextPage: true, endCursor: true, hasPreviousPage: true, startCursor: true },
    };
}
exports.edgesSelector = edgesSelector;
function querySelector(selector, args) {
    return {
        query: {
            __args: convertPagingOptions(args || {}),
            edges: { node: selector, cursor: true },
            pageInfo: { hasNextPage: true, endCursor: true, hasPreviousPage: true, startCursor: true },
        },
    };
}
exports.querySelector = querySelector;
//# sourceMappingURL=paging.js.map