import { MutableRefObject, Dispatch, SetStateAction, LegacyRef, ChangeEvent, useEffect, RefObject, Reducer, ReducerWithoutAction, DispatchWithoutAction } from 'react';
import { DebounceSettings, DebouncedFunc } from 'lodash';

/**
 * useBoundingclientRect hook
 *
 * @param ref The React ref whose ClientRect is needed
 * @returns ClientRect
 */
declare function useBoundingclientrect(ref: MutableRefObject<HTMLElement | null>): ClientRect | DOMRect | null;

declare type HTMLElementOrNull = HTMLElement | null;
declare type RefElementOrNull<T> = T | null;
declare type CallbackRef<T extends HTMLElement | null = HTMLElementOrNull> = (node: T) => any;
declare type AnyRef<T extends HTMLElement | null = HTMLElementOrNull> = CallbackRef<T> | MutableRefObject<T>;

/**
 * useBoundingclientrectRef hook
 *
 * @returns [CallbackRef | null, ClientRect | DOMRect | null, () => void]
 */
declare function useBoundingclientrectRef(): [
    CallbackRef | null,
    ClientRect | DOMRect | null,
    () => void
];

declare type CountdownOptions = {
    interval?: number;
    onDown?: Function;
    onEnd?: Function;
};
/**
 *
 * useCountdown
 * Easy way to countdown until a given endtime in intervals
 *
 * @param endTime Time to countdown
 * @param options  Countdown options
 */
declare function useCountdown(endTime: Date, options?: CountdownOptions): number;

declare type CounterHandler = {
    value: number;
    increment: () => void;
    decrement: () => void;
    incrementBy: (amount: number) => void;
    decrementBy: (amount: number) => void;
    reset: () => void;
};
/**
 *
 * @typedef handler
 * @type {Object}
 * @property {number} value The value of the counter
 * @property {Function}  increment Increment counter value by 1
 * @property {Function} decrement Decrement counter value by 1
 * @property {Function} incrementBy Increment counter by incrAmount
 * @property {Function} decrementBy Decrement counter by decrAmount
 * @property {Function} reset Reset counter to initialValue
 */
/**
 * Counter hook
 *
 * @param {number} initialValue The initial value of the counter
 * @returns {handler} A handler to interact with the counter
 */
declare function useCounter(initialValue: number): CounterHandler;

/**
 * Debounce hook
 * Debounces a function
 *
 * @param callback The callback to debounce
 * @param wait The duration to debounce
 * @param options The options object.
 * @param options.leading Specify invoking on the leading edge of the timeout.
 * @param options.maxWait The maximum time func is allowed to be delayed before it’s invoked.
 * @param options.trailing Specify invoking on the trailing edge of the timeout.
 * @returns Returns the new debounced function.
 */
declare function useDebounce<T extends (...args: any[]) => unknown>(callback: T, wait?: number, options?: DebounceSettings): DebouncedFunc<T>;

declare type UseDebouncedValueOptions = Partial<{
    initializeWithNull: boolean;
}>;
declare type UseDebouncedValueReturnType<T> = [
    debouncedValue: T | null,
    immediatelyUpdateDebouncedValue: Dispatch<SetStateAction<T | null>>
];
declare const useDebouncedValue: <T = unknown>(value: T, timeout: number, options?: UseDebouncedValueOptions) => UseDebouncedValueReturnType<T>;

/**
 * useDidMount hook
 * Calls a function on mount
 *
 * @param {Function} callback Callback function to be called on mount
 *
 */
declare function useDidMount(callback: () => void): void;

/**
 *  useDidUpdate hook
 *
 *  Fires a callback on component update
 *  Can take in a list of conditions to fire callback when one of the
 *  conditions changes
 *
 * @param {Function} callback The callback to be called on update
 * @param {Array} conditions The list of variables which trigger update when they are changed
 */
declare function useDidUpdate(callback: () => void, conditions?: unknown[]): void;

declare type UseDimensionsRefReturn = {
    bottom: number;
    height: number;
    left: number;
    right: number;
    top: number;
    width: number;
    x: number;
    y: number;
} | null;
declare type UseDimensionsHook = [
    LegacyRef<HTMLDivElement> | undefined,
    UseDimensionsRefReturn,
    HTMLElement | null
];
declare type UseDimensionsRefArgs = {
    updateOnResize?: boolean;
    updateOnScroll?: boolean;
};
declare const useDimensionsRef: ({ updateOnScroll, updateOnResize, }?: UseDimensionsRefArgs) => UseDimensionsHook;

declare type DeepNullable<T> = {
    [K in keyof T]: DeepNullable<T[K]> | null;
};
declare type ListenerOptions = boolean | {
    capture?: boolean;
    once?: boolean;
    passive?: boolean;
    signal?: AbortSignal;
};
declare type ExcludeFunction<T> = Exclude<T, Function>;

/**
 *  useDocumentEventListener hook
 *
 *  A react hook to an event listener to the document
 *
 * @param {string} eventName The event to track
 * @param {Function} callback The callback to be called on event
 * @param {ListenerOptions} listenerOptions The options to be passed to the event listener
 * @param {boolean} isLayoutEffect Should it use layout effect. Defaults to false
 * @returns {undefined}
 */
declare function useDocumentEventListener(eventName: string, callback: (...args: any) => void, listenerOptions?: ListenerOptions, isLayoutEffect?: boolean): void;

/**
 * useEffectOnceWhen hook
 *
 * It fires a callback once when a condition is true or become true.
 * Fires the callback at most one time.
 *
 * @param callback The callback to fire
 * @param when The condition which needs to be true
 */
declare function useEffectOnceWhen(callback: () => void, when?: boolean): void;

/**
 *  useEventListenerRef hook
 *
 *  A react hook to an event listener to an element
 *  Returns a ref
 *
 * @param {string} eventName The event to track
 * @param {Function} callback The callback to be called on event
 * @param {object} conditions The options to be passed to the event listener
 * @param {boolean} isLayoutEffect Should it use layout effect. Defaults to false
 * @returns {Function} A callback ref that can be used as ref prop
 */
declare function useEventListenerRef(eventName: string, callback: (...args: any) => void, listenerOptions?: any, isLayoutEffect?: boolean): (refElement: RefElementOrNull<HTMLElement>) => void;

/**
 * useFreshRef
 *
 * @param value The value which needs to be fresh at all times. Probably
 * best used with functions
 * @param preferLayoutEffect Should the value be updated using a layout effect
 * or a passive effect. Defaults to false.
 * @returns A ref containing the fresh value
 */
declare function useFreshRef<T>(value: T, preferLayoutEffect?: boolean): MutableRefObject<T>;

declare type CallbackType<T> = (...args: T[]) => void;
declare function useFreshTick<T>(callback: CallbackType<T>): CallbackType<T>;

declare type EventCallback = (this: Document, event_: any) => any;
declare type OnChangeEventCallback = (this: Document, event_: any, isOpen: boolean) => any;
declare type NoopFunction = () => void;
declare type FullscreenApi = {
    element: HTMLElement | null | undefined;
    exit: NoopFunction | (() => Promise<unknown>);
    isEnabled: boolean;
    isFullscreen: boolean;
    /**
     * @deprecated Please use useFullScreen({onChange : function() {}}) instead.
     */
    onChange: NoopFunction | ((callback: OnChangeEventCallback) => void);
    /**
     * @deprecated Please use useFullScreen({onError : function() {}}) instead.
     */
    onError: NoopFunction | ((callback: EventCallback) => void);
    request: NoopFunction | ((element?: HTMLElement) => Promise<unknown>);
    toggle: NoopFunction | ((element?: HTMLElement) => Promise<unknown>);
};
declare type RequestFullscreenOptions = {
    navigationUI?: string | "auto" | "hide" | "show";
};
declare type FullScreenOptions = {
    onChange?: OnChangeEventCallback;
    onError?: EventCallback;
    requestFullscreenOptions?: RequestFullscreenOptions;
};
/**
 * useFullscreen
 * A hook that helps make the document fullscreen
 */
declare function useFullscreen(options?: FullScreenOptions): FullscreenApi;

/**
 * useForkRef
 * Joins refs together and returns a combination of the two as a new ref
 *
 * @param refA
 * @param refB
 */
declare function useForkRef<T extends HTMLElement | null = HTMLElementOrNull>(refA: AnyRef<T> | null, refB: AnyRef<T> | null): CallbackRef<T> | null;

declare type UseGetIsMounted = () => () => boolean;
/**
 * @description useGetIsMounted hook checks if a component is mounted or not at the time.
 * Useful for async effects. Returns a callback that returns a boolean representing if the component
 * is mounted at the time.
 * @returns () => boolean
 * @see https://react-hooks.org/docs/useGetIsMounted
 */
declare const useGetIsMounted: UseGetIsMounted;

declare type IGetGeoLocation = {
    lat?: number;
    lng?: number;
    isError: boolean;
    message: string;
};
declare type IOptions = {
    enableHighAccuracy?: boolean;
    timeout?: number;
    maximumAge?: number;
    when?: boolean;
};
/**
 * useGeolocation
 * Gets the geolocation data as a hook
 *
 * @param geoLocationOptions Geolocation options
 */
declare function useGeolocation(geoLocationOptions?: IOptions): IGetGeoLocation | null;

declare type InputChangeEvent = ChangeEvent<HTMLInputElement>;
declare type InputHandler<T> = {
    /**
     * Function to handle onChange of an input element
     *
     * @param event The input change event
     */
    onChange: (event: InputChangeEvent) => void;
    /**
     * The current value of the input
     */
    value: T;
};
declare type Options<T> = {
    /**
     * validate
     *
     * Validator function which can be used to prevent updates
     *
     * @param {any} New value
     * @param {any} Current value
     * @returns {boolean} Whether an update should happen or not
     */
    validate?: (newValue: T, currentValue: T) => boolean;
};
/**
 *
 * useInput Hook
 *
 * Handles an input's value and onChange props internally to
 * make text input creation process easier
 *
 * @param {unknown} [initialValue] Initial value of the input
 * @param {Options} [options] Options object
 * @returns {InputHandler} Input handler with value and onChange
 */
declare function useInput<T extends number | string | readonly string[] | undefined = string>(initialValue?: T, options?: Options<T>): InputHandler<T>;

declare type IntervalHandlerAsObject = {
    /**
     * IntervalId of the interval
     */
    intervalId: ReturnType<typeof setTimeout> | null;
    /**
     * Function to start the interval
     */
    start: () => void;
    /**
     * Function to stop the interval
     */
    stop: () => void;
};
declare type IntervalHandlerAsArray = Array<ReturnType<typeof setTimeout> | (() => void) | null> & {
    0: () => void;
    1: () => void;
    2: ReturnType<typeof setTimeout> | null;
};
declare type IntervalHandler = IntervalHandlerAsArray & IntervalHandlerAsObject;
/**
 *
 * useInterval hook
 *
 * Declaratively creates a setInterval to run a callback after a fixed
 * amount of time
 *
 *@param {Function} callback - Callback to be fired
 *@param {number} intervalId - Interval duration in milliseconds after which the callback is to be fired
 *@param {boolean} startImmediate - Whether the interval should start immediately on initialise
 *@returns {IntervalHandler}
 */
declare function useInterval(callback: () => void, intervalDuration: number | null, startImmediate?: boolean): IntervalHandler;

/**
 * A setInterval hook that calls a callback after a interval duration
 * when a condition is true
 *
 * @param cb The callback to be invoked after interval
 * @param intervalDurationMs Amount of time in ms after which to invoke
 * @param when The condition which when true, sets the interval
 * @param startImmediate If the callback should be invoked immediately
 */
declare function useIntervalWhen(callback_: () => void, intervalDurationMs?: number, when?: boolean, startImmediate?: boolean): void;

/**
 *
 * useIntersectionObserverRef hook
 *
 * Returns a mutation observer for a React Ref and fires a callback
 *
 * @param {IntersectionObserverCallback} callback Function that needs to be fired on mutation
 * @param {IntersectionObserverInit} options
 */
declare function useIntersectionObserverRef(callback: IntersectionObserverCallback | undefined, options?: IntersectionObserverInit): [CallbackRef];

/**
 *
 * useInViewRef hook
 *
 * Returns a mutation observer for a React Ref and true/false when element enters/leaves the viewport. Also fires a callback.
 *
 * @param {IntersectionObserverCallback} callback Function that needs to be fired on mutation
 * @param {IntersectionObserverInit} options
 */
declare function useInViewRef(callback?: IntersectionObserverCallback, options?: IntersectionObserverInit): [CallbackRef, boolean];

/**
 * useIsomorphicEffect
 * Resolves to useEffect when "window" is not in scope and useLayout effect in the browser
 *
 * @param {Function} callback Callback function to be called on mount
 */
declare const useIsomorphicEffect: typeof useEffect;

declare type TrackedKeyEvents = "keydown" | "keypress" | "keyup";
declare type Options$1 = {
    /**
     * Keyboardevent types to listen for. Valid options are keyDown, keyPress and keyUp
     */
    eventTypes?: TrackedKeyEvents[];
    /**
     * target mutable ref on which the events should be listened. Doesn't work with callback refs.
     * Please use useKeyRef instead if you want to use with callback refs.
     * If no target is specified, events are listened to on the window. Defaults to window.
     */
    target?: RefObject<HTMLElement>;
    /**
     * Condition which if true, will enable the event listeners
     */
    when?: boolean;
};
/**
 * useKey hook
 *
 * Fires a callback on keyboard events like keyDown, keyPress and keyUp
 *
 * @param {TrackedKeyEvents} keys List of keys to listen for. Eg: ["a", "b"]
 * @param {Callback} callback  Callback to fire on keyboard events
 * @param {Options} options Options
 * @see {@link https://react-hooks.org/docs/useKey}
 */
declare function useKey(keys: Array<number | string> | number | string, callback: (event: KeyboardEvent) => void, options?: Options$1): void;

declare type TrackedKeyEvents$1 = "keydown" | "keypress" | "keyup";
declare type Options$2 = {
    /**
     * Keyboardevent types to listen for. Valid options are keyDown, keyPress and keyUp
     */
    eventTypes?: TrackedKeyEvents$1[];
    /**
     * target ref on which the events should be listened. If no target is specified,
     * events are listened to on the window. Only works with object refs. If you want to use with callback refs,
     * please use useKeyRef instead.
     */
    target?: RefObject<HTMLElement>;
    /**
     * Condition which if true, will enable the event listeners
     */
    when?: boolean;
};
declare type KeyBindings = {
    [key: string]: (event: KeyboardEvent) => void;
};
/**
 * useKeyBindings
 *
 * useKeyBindings binds pairs of keyboard events and handlers
 *
 * @param { KeyBindings } keyBindings
 * @param {Options} options
 * @see {@link https://react-hooks.org/docs/useKeyBindings}
 */
declare const useKeyBindings: (keyBindings: KeyBindings, options?: Options$2 | undefined) => void;

declare type TrackedKeyEvents$2 = "keydown" | "keypress" | "keyup";
declare type Callback = (event: KeyboardEvent) => void;
declare type Options$3 = {
    /**
     * Keyboardevent types to listen for. Valid options are keyDown, keyPress and keyUp
     */
    eventTypes?: TrackedKeyEvents$2[];
    /**
     * Condition which if true, will enable the event listeners
     */
    when?: boolean;
};
/**
 * useKeyRef hook
 *
 * Fires a callback on keyboard events like keyDown, keyPress and keyUp
 *
 * @param {[string|number]} keys List of keys to listen for. Eg: ["a", "b"]
 * @param {Function} callback Callback to fire on keyboard events
 * @param {Options} options Options
 * @returns {CallbackRef} CallbackRef
 * @see {@link https://react-hooks.org/docs/useKeyRef}
 */
declare function useKeyRef(keys: Array<number | string> | number | string, callback: Callback, options?: Options$3): CallbackRef;

declare type Options$4 = {
    /**
     * when boolean to enable and disable events, when passed false
     * remove the eventlistener if any
     */
    when?: boolean;
    /**
     * should the event logging be continuous
     */
    continuous?: boolean;
    /**
     * target ref on which the events should be listened. If no target is specified,
     * events are listened to on the document
     */
    target?: MutableRefObject<Document> | MutableRefObject<HTMLElement | null>;
};
/**
 * useKeys hook
 *
 * @param keysList
 * @param callback
 * @param opts
 */
declare function useKeys(keysList: string[], callback: (e: KeyboardEvent) => any, options_?: Options$4): void;

/**
 * useLifecycleLogger hook
 * logs parameters as component transitions through lifecycles
 *
 * @param componentName Name of the component
 * @param rest
 */
declare const useLifecycleLogger: (componentName?: string, ...otherArgs: unknown[]) => void;

declare type StorageHandlerAsObject = {
    value: any;
    set: (newValue: any) => void;
    remove: () => void;
};
declare type StorageHandlerAsArray = [any, (newValue: any) => void, () => void];
declare type StorageHandler = StorageHandlerAsArray & StorageHandlerAsObject;
/**
 * useLocalstorage hook
 * Tracks a value within localStorage and updates it
 *
 * @param {string} key - Key of the localStorage object
 * @param {any} defaultValue - Default initial value
 */
declare function useLocalstorage(key: string, defaultValue?: any): StorageHandler;

declare type UseLocalstorageStateReturnValue<S> = [
    S,
    Dispatch<SetStateAction<S>>,
    () => void
];
/**
 * useLocalstorageState hook
 * Tracks a value within localStorage and updates it
 *
 * @param {string} key - Key of the localStorage object
 * @param {any} initialState - Default initial value
 */
declare function useLocalstorageState<S>(key: string, initialState?: S | (() => S)): UseLocalstorageStateReturnValue<S>;

/**
 * useMapState hook
 * A hook to manage state in the form of a map or object.
 *
 * @param initialValue Initial value of the map
 */
declare function useMapState<T, K extends keyof T>(initialValue: T): [
    T,
    {
        has: (key: K) => boolean;
        remove: (key: K) => void;
        removeAll: () => void;
        removeMultiple: (...keys: K[]) => void;
        set: (key: K, value: any) => void;
        setMultiple: (next: Partial<T>) => void;
    }
];

/**
 * useMediaMatch
 *
 * A react hook that signals whether or not a media query is matched.
 *
 * @param query The media query to signal on. Example, `"print"` will signal
 * `true` when previewing in print mode, and `false` otherwise.
 * @returns Whether or not the media query is currently matched.
 */
declare function useMediaMatch(query: string): boolean;

/**
 * useMergeRefs
 * Merges multiple refs into a single function ref.
 * Takes any number of refs.
 * Refs can be mutable refs or function refs.
 *
 * @param refs
 */
declare function useMergeReferences(...references: AnyRef[]): CallbackRef | null;

declare type MouseData = {
    x: number | null;
    y: number | null;
    screenX: number | null;
    screenY: number | null;
    pageX: number | null;
    pageY: number | null;
    clientX: number | null;
    clientY: number | null;
    movementX: number | null;
    movementY: number | null;
    offsetX: number | null;
    offsetY: number | null;
};
/**
 * useMouse hook
 *
 * Retrieves current mouse position and information about the position like
 * screenX, pageX, clientX, movementX, offsetX
 */
declare function useMouse(): MouseData;

declare type OptionalIndexValue<T> = {
    index?: number;
    value?: T;
};
declare type OptionalIndicesValues<T> = {
    indices?: number[];
    values?: T[];
};

declare type UseMultiSelectableListReturnType<T> = [
    Array<number[] | T[]>,
    {
        matchSelection: (parameters: OptionalIndexValue<T>) => boolean;
        toggleSelection: (parameters: OptionalIndexValue<T>) => () => void;
        updateSelections: ({ indices, values, }: OptionalIndicesValues<T>) => () => void;
    }
];
/**
 * useMultiSelectableList
 * A custom hook to easily select multiple values from a list
 *
 * @param list - The list of values to select from
 * @param initialSelectIndices - The indices of the initial selections
 * @param allowUnselected - Whether or not to allow unselected values
 * @see {@link https://react-hooks.org/docs/useMultiSelectableList}
 */
declare function useMultiSelectableList<T>(list?: T[], initialSelectIndices?: number[], allowUnselected?: boolean): UseMultiSelectableListReturnType<T>;

/**
 *
 * useMutationObserver hook
 *
 * Returns a mutation observer for a React Ref and fires a callback
 *
 * @param {MutableRefObject<HTMLElement | null>} ref React ref on which mutations are to be observed
 * @param {MutationCallback} callback Function that needs to be fired on mutation
 * @param {MutationObserverInit} options
 */
declare function useMutationObserver(ref: MutableRefObject<HTMLElement | null>, callback: MutationCallback, options?: MutationObserverInit): void;

/**
 *
 * useMutationObserverRef hook
 *
 * Returns a mutation observer for a React Ref and fires a callback
 *
 * @param {MutationCallback} callback Function that needs to be fired on mutation
 * @param {MutationObserverInit} options
 */
declare function useMutationObserverRef(callback: MutationCallback, options?: MutationObserverInit): [CallbackRef];

declare type Language = string | null;
/**
 * useNavigatorLanguage hook
 * Returns the language of the navigator
 *
 * @returns {Language}
 * @see {@link https://react-hooks.org/docs/useNavigatorLanguage}
 */
declare function useNavigatorLanguage(): Language;

/**
 *
 * useOnWindowResize hook
 *
 * Fires a callback when window resizes
 *
 * @param {Function} callback Callback to be called before unmount
 * @param {boolean} when When the handler should be applied
 * @param {boolean} isLayoutEffect Should it use layout effect. Defaults to false
 */
declare function useOnWindowResize(callback: (event: any) => void, when?: boolean, isLayoutEffect?: boolean): void;

/**
 *
 * useOnWindowScroll hook
 * Fires a callback when window scroll
 *
 * @param {Function} callback Callback to be called before unmount
 * @param {boolean} when When the handler should be applied
 * @param {boolean} isLayoutEffect Should it use layout effect. Defaults to false
 */
declare function useOnWindowScroll(callback: (event: any) => void, when?: boolean, isLayoutEffect?: boolean): void;

/**
 * useOnline hook
 *
 * Returns true if navigator is online, false if not.
 *
 * @returns {boolean} The value of navigator.onLine
 */
declare function useOnline(): boolean | null;

/**
 *  useOutsideClick hook
 * Checks if a click happened outside a Ref. Handy for dropdowns, modals and popups etc.
 *
 * @param ref Ref whose outside click needs to be listened to
 * @param handler Callback to fire on outside click
 * @param when A boolean which which activates the hook only when it is true. Useful for conditionally enable the outside click
 */
declare function useOutsideClick(ref: MutableRefObject<HTMLElement | null>, handler: (e: MouseEvent) => any, when?: boolean): void;

/**
 * useOutsideClickRef hook
 * Checks if a click happened outside a Ref. Handy for dropdowns, modals and popups etc.
 *
 * @param handler Callback to fire on outside click
 * @param when A boolean which which activates the hook only when it is true. Useful for conditionally enable the outside click
 * @returns An array with first item being ref
 */
declare function useOutsideClickRef(handler: (e: MouseEvent) => any, when?: boolean): [CallbackRef];

/**
 * usePrevious hook for React
 *
 * @param currentValue The value whose previous value is to be tracked
 * @returns The previous value
 */
declare function usePrevious<T>(currentValue: T): T | null;

/**
 * usePreviousDifferent hook for React
 * It returns the past value which was different from the current one.
 *
 * @param currentValue The value whose previously different value is to be tracked
 * @returns The previous value
 */
declare function usePreviousDifferent<T>(currentValue: T): T | null;

/**
 * usePreviousImmediate hook for React
 *
 * @param currentValue The value whose previous value is to be tracked
 * @returns The previous value
 */
declare function usePreviousImmediate<T>(currentValue: T): T | null;

declare function useQueueState<T>(initialList: T[]): [
    T[],
    {
        enqueue: (item: T) => number;
        dequeue: () => T | undefined;
        peek: () => T | undefined;
        length: number;
    }
];

/**
 *
 * useRaf
 * Uses a polyfilled version of requestAnimationFrame
 *
 * @param {Function} callback The callback function to be executed
 * @param {boolean} [isActive] The value which while true, keeps the raf running infinitely
 * @see {@link https://react-hooks.org/docs/useRaf}
 */
declare function useRaf(callback: (timeElapsed: number) => void, isActive: boolean): void;

/**
 * useRefElement hook for React
 * Helps bridge gap between callback ref and state
 * Manages the element called with callback ref api using state variable
 */
declare function useRefElement<T>(): [
    (refElement: RefElementOrNull<T>) => void,
    RefElementOrNull<T>
];

declare type SelectHandler<T> = {
    index: number;
    item: T;
    setIndex: (newIndex: number) => void;
    setItem: (newItem: T) => void;
};
/**
 * useSelect hook
 * Helps easily select a value from a list of values
 *
 * @param list List of values to select a value from
 * @param [initialIndex=0] Initial index which is selected
 * @returns handler
 */
declare function useSelect<T>(list: T[], initialIndex?: number): SelectHandler<T>;

declare type Selection<T> = [number, T];
declare type UseSelectableListReturnType<T> = [
    Selection<T>,
    {
        matchSelection: (parameters: OptionalIndexValue<T>) => boolean;
        toggleSelection: (parameters: OptionalIndexValue<T>) => () => void;
        updateSelection: (parameters: OptionalIndexValue<T>) => () => void;
    }
];
/**
 * useSelectableList
 * Easily select a single value from a list of values. very useful for radio buttons, select inputs  etc.
 *
 * @param list - The list of values to select from
 * @param initialIndex  - The index of the initial selection
 * @param allowUnselected
 * @see {@link https://react-hooks.org/docs/useSelectableList}
 */
declare function useSelectableList<T>(list?: T[], initialIndex?: number, allowUnselected?: boolean): UseSelectableListReturnType<T>;

declare type StorageHandlerAsObject$1<T> = {
    remove: () => null | undefined;
    set: (newValue: T) => void;
    value: T;
};
declare type StorageHandlerAsArray$1<T> = [T, (newValue: T) => void, () => void];
declare type StorageHandler$1<T> = StorageHandlerAsArray$1<T> & StorageHandlerAsObject$1<T>;
/**
 * useSessionstorage
 * Tracks a value within sessionStorage and updates it
 *
 * @param key Key of the value to be stored
 * @param defaultValue Default value of the stored item
 * @see {@link https://react-hooks.org/docs/useSessionstorage}
 */
declare function useSessionstorage<T>(key: string, defaultValue?: T | null): StorageHandler$1<T>;

declare type UseSessionstorateStateReturnValue<S> = [
    S,
    Dispatch<SetStateAction<S>>,
    () => void
];
/**
 * useSessionstorageState hook
 * Tracks a value within sessionStorage and updates it
 *
 * @param {string} key - Key of the sessionStorage object
 * @param {any} initialState - Default initial value
 */
declare function useSessionstorageState<S>(key: string, initialState?: S | (() => S)): UseSessionstorateStateReturnValue<S>;

declare function useStackState<T>(initialList: T[]): [
    T[],
    {
        clear: () => void;
        isEmpty: () => boolean;
        push: (item: T) => number;
        pop: () => T | undefined;
        peek: () => T | undefined;
        length: number;
    },
    T[]
];

declare type Callback$1<T> = (...args: T[]) => void;
/**
 * useThrottle
 * Throttles a function with a timeout and ensures
 * that the callback function runs at most once in that duration
 *
 * @param callback The callback to throttle
 * @param timeout Throttle timeout
 * @see {@link https://react-hooks.org/docs/useThrottle}
 */
declare function useThrottle<T>(callback: Callback$1<T>, timeout?: number): [Callback$1<T>, boolean];

declare type UseTimeoutHandler = {
    start: () => any;
    clear: () => any;
    stop: () => any;
    isActive: boolean;
};
/**
 * A setTimeout hook that calls a callback after a timeout duration
 *
 * @param cb The callback to be invoked after timeout
 * @param timeoutDelayMs Amount of time in ms after which to invoke
 */
declare function useTimeout(callback_: () => void, timeoutDelayMs?: number): UseTimeoutHandler;

/**
 * A setTimeout hook that calls a callback after a timeout duration
 * when a condition is true
 *
 * @param cb The callback to be invoked after timeout
 * @param timeoutDelayMs Amount of time in ms after which to invoke
 * @param when The condition which when true, sets the timeout
 */
declare function useTimeoutWhen(callback_: () => void, timeoutDelayMs?: number, when?: boolean): void;

/**
 * Use toggle hook helps you easily toggle a value.
 *
 * @param initialValue Initial value of the toggle, which will be false if not provided.
 * @example
 * const [boolean, toggle] = useToggle();
 * // value is false
 * // toggle() will change value to true.
 */
declare function useToggle<S = boolean>(initialValue?: boolean): [S, Dispatch<unknown>];
/**
 * Use toggle hook helps you easily toggle a value
 *
 * @param initialValue Initial value of the toggle, which will be false if not provided.
 * @param toggleFunction A toggle function. This allows for non boolean toggles
 * @example
 * const [value, toggle] = useToggle("on", _value => _value === "on" ? "off" : "on");
 * // value is "on"
 * // toggle() will change value to "off". Calling it again will change value to "on".
 */
declare function useToggle<S>(initialValue: S, toggleFunction?: Reducer<S, unknown>): [S, Dispatch<unknown>];
/**
 * Use toggle hook helps you easily toggle a value
 *
 * @param initialValue Initial value of the toggle, which will be false if not provided.
 * @param toggleFunction A toggle function. This allows for non boolean toggles
 * @example
 * const [value, toggle] = useToggle("on", _value => _value === "on" ? "off" : "on");
 * // value is "on"
 * // toggle() will change value to "off". Calling it again will change value to "on".
 */
declare function useToggle<S>(initialValue: S, toggleFunction: ReducerWithoutAction<S>): [S, DispatchWithoutAction];

declare type UseUndoStateOptions = {
    maxSize: number;
};
declare type UseUndoStatePushFunctionArgumentsCallback<T> = (currentValue: T) => T;
declare type UseUndoStatePushFunction<T> = (argument: T | UseUndoStatePushFunctionArgumentsCallback<T>) => void;
declare type UndoFunction = () => void;
declare type UseUndoStateReturnValue<T> = [
    ExcludeFunction<T>,
    UseUndoStatePushFunction<ExcludeFunction<T>>,
    UndoFunction
];

/**
 * useUndoState hook
 * Drop in replacement for useState hook but with undo functionality.
 *
 * @typedef UndoStateOptions
 * @type {object}
 * @property {number} maxSize - Maximum number of states to keep in the undo stack.
 * @param {any} defaultValue - Default value to use for the state. This will be the first value in the undo stack.
 * @param {UseUndoStateOptions} options - Options for the undo state. Currently takes the maxSize option.
 * @returns {UseUndoStateReturnValue}
 * @see {@link https://react-hooks.org/docs/useUndoState}
 */
declare const useUndoState: <T>(defaultValue: Exclude<T, Function>, options?: UseUndoStateOptions | undefined) => UseUndoStateReturnValue<T>;

/**
 *  useDidUpdate hook
 *
 *  Fires a callback on component update
 *  Can take in a list of conditions to fire callback when one of the
 *  conditions changes
 *  Will fire callback's cleanup function on update
 *
 * @param {Function} callback The callback and its cleanup to be called on update
 * @param {Array} conditions The list of variables which trigger update when they are changed
 * @returns {}
 */
declare function useUpdateEffect(callback: () => void, conditions?: unknown[]): void;

/**
 * useVisibilitySensor hook
 * Tracks the visibility of a ref
 *
 * @param ref The ref to track visibility of
 * @param opts Options
 * @see {@link https://react-hooks.org/docs/useVisibilitySensor}
 */
declare function useVisibilitySensor(ref: any, options: any): any;

declare type Callback$2 = () => void;
/**
 * useWillUnmount hook
 * Fires a callback just before component unmounts
 *
 * @param {Function} callback Callback to be called before unmount
 */
declare function useWillUnmount(callback: Callback$2): void;

/**
 *  useWindowEventListener hook
 *
 *  A react hook to an event listener to the window
 *
 * @param {string} eventName The event to track
 * @param {Function} callback The callback to be called on event
 * @param {ListenerOptions} listenerOptions The options to be passed to the event listener
 * @param {boolean} isLayoutEffect Should it use layout effect. Defaults to false
 * @returns {undefined}
 */
declare function useWindowEventListener(eventName: string, callback: (...args: any) => void, listenerOptions?: ListenerOptions, isLayoutEffect?: boolean): void;

declare type ScrollPosition = {
    scrollX: Window["scrollX"];
    scrollY: Window["scrollY"];
};
/**
 *
 * useWindowScrollPosition hook
 * A React hook to get the scroll position of the window
 *
 * @returns an object containing scrollX and scrollY values
 */
declare function useWindowScrollPosition(): ScrollPosition;

declare type WindowDimensions = DeepNullable<Pick<Window, "innerHeight" | "innerWidth" | "outerHeight" | "outerWidth">>;
/**
 * useWindowSize hook
 * A hook that provides information of the dimensions of the window
 *
 * @returns Dimensions of the window
 */
declare function useWindowSize(): WindowDimensions;

export { useBoundingclientrect, useBoundingclientrectRef, useCountdown, useCounter, useDebounce, useDebouncedValue, useDidMount, useDidUpdate, useDimensionsRef, useDocumentEventListener, useEffectOnceWhen, useEventListenerRef, useForkRef, useFreshRef, useFreshTick, useFullscreen, useGeolocation, useGetIsMounted, useInViewRef, useInput, useIntersectionObserverRef, useInterval, useIntervalWhen, useIsomorphicEffect, useKey, useKeyBindings, useKeyRef, useKeys, useLifecycleLogger, useLocalstorage, useLocalstorageState, useMapState, useMediaMatch, useMergeReferences as useMergeRefs, useMouse, useMultiSelectableList, useMutationObserver, useMutationObserverRef, useNavigatorLanguage, useOnWindowResize, useOnWindowScroll, useOnline, useOutsideClick, useOutsideClickRef, usePrevious, usePreviousDifferent, usePreviousImmediate, useQueueState, useRaf, useRefElement, useSelect, useSelectableList, useSessionstorage, useSessionstorageState, useStackState, useThrottle, useTimeout, useTimeoutWhen, useToggle, useUndoState, useUpdateEffect, useVisibilitySensor, useWillUnmount, useWindowEventListener, useWindowScrollPosition, useWindowSize };
