"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.useKeyRef = void 0;
var react_1 = require("react");
var doesIdentifierMatchKeyboardEvent_1 = require("../utils/doesIdentifierMatchKeyboardEvent");
var noop_1 = require("../utils/noop");
var defaultOptions = {
    eventTypes: ["keydown"],
    when: true,
};
/**
 * useKeyRef hook
 *
 * Fires a callback on keyboard events like keyDown, keyPress and keyUp
 *
 * @param {[string|number]} keys List of keys to listen for. Eg: ["a", "b"]
 * @param {Function} callback Callback to fire on keyboard events
 * @param {Options} options Options
 * @returns {CallbackRef} CallbackRef
 * @see {@link https://react-hooks.org/docs/useKeyRef}
 */
function useKeyRef(keys, callback, options) {
    var _a = (0, react_1.useState)(null), targetNode = _a[0], setTargetNode = _a[1];
    var ref = (0, react_1.useCallback)(function (node) {
        setTargetNode(node);
    }, []);
    var keyList = (0, react_1.useMemo)(function () {
        if (Array.isArray(keys)) {
            return keys;
        }
        else {
            return [keys];
        }
    }, [keys]);
    var internalOptions = (0, react_1.useMemo)(function () {
        return __assign(__assign({}, defaultOptions), options);
    }, [options]);
    var when = internalOptions.when, eventTypes = internalOptions.eventTypes;
    var callbackRef = (0, react_1.useRef)(callback);
    (0, react_1.useEffect)(function () {
        callbackRef.current = callback;
    });
    var handle = (0, react_1.useCallback)(function (event) {
        if (keyList.some(function (identifier) {
            return (0, doesIdentifierMatchKeyboardEvent_1.doesIdentifierMatchKeyboardEvent)(event, identifier);
        })) {
            callbackRef.current(event);
        }
    }, [keyList]);
    (0, react_1.useEffect)(function () {
        if (when && targetNode) {
            for (var _i = 0, eventTypes_1 = eventTypes; _i < eventTypes_1.length; _i++) {
                var eventType = eventTypes_1[_i];
                // eslint-disable-next-line prefer-arrow-callback
                targetNode.addEventListener(eventType, handle);
            }
            return function () {
                for (var _i = 0, eventTypes_2 = eventTypes; _i < eventTypes_2.length; _i++) {
                    var eventType = eventTypes_2[_i];
                    targetNode.removeEventListener(eventType, handle);
                }
            };
        }
        return noop_1.noop;
    }, [targetNode, when, eventTypes, keyList, handle]);
    return ref;
}
exports.useKeyRef = useKeyRef;
