"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.useKey = void 0;
var react_1 = require("react");
var doesIdentifierMatchKeyboardEvent_1 = require("../utils/doesIdentifierMatchKeyboardEvent");
var noop_1 = require("../utils/noop");
var defaultOptions = {
    eventTypes: ["keydown"],
    when: true,
};
/**
 * useKey hook
 *
 * Fires a callback on keyboard events like keyDown, keyPress and keyUp
 *
 * @param {TrackedKeyEvents} keys List of keys to listen for. Eg: ["a", "b"]
 * @param {Callback} callback  Callback to fire on keyboard events
 * @param {Options} options Options
 * @see {@link https://react-hooks.org/docs/useKey}
 */
function useKey(keys, callback, options) {
    var keyList = (0, react_1.useMemo)(function () {
        if (Array.isArray(keys)) {
            return keys;
        }
        else {
            return [keys];
        }
    }, [keys]);
    var internalOptions = (0, react_1.useMemo)(function () {
        return __assign(__assign({}, defaultOptions), options);
    }, [options]);
    var when = internalOptions.when, eventTypes = internalOptions.eventTypes;
    var callbackRef = (0, react_1.useRef)(callback);
    var target = internalOptions.target;
    (0, react_1.useEffect)(function () {
        callbackRef.current = callback;
    });
    var handle = (0, react_1.useCallback)(function (event) {
        if (keyList.some(function (identifier) {
            return (0, doesIdentifierMatchKeyboardEvent_1.doesIdentifierMatchKeyboardEvent)(event, identifier);
        })) {
            callbackRef.current(event);
        }
    }, [keyList]);
    (0, react_1.useEffect)(function () {
        if (when && typeof window !== "undefined") {
            // If target is defined by the user
            if (target) {
                var targetNode_1 = target.current;
                if (targetNode_1) {
                    for (var _i = 0, eventTypes_1 = eventTypes; _i < eventTypes_1.length; _i++) {
                        var eventType = eventTypes_1[_i];
                        targetNode_1.addEventListener(eventType, handle);
                    }
                    return function () {
                        for (var _i = 0, eventTypes_3 = eventTypes; _i < eventTypes_3.length; _i++) {
                            var eventType = eventTypes_3[_i];
                            targetNode_1.removeEventListener(eventType, handle);
                        }
                    };
                }
            }
            else {
                for (var _a = 0, eventTypes_2 = eventTypes; _a < eventTypes_2.length; _a++) {
                    var eventType = eventTypes_2[_a];
                    window.addEventListener(eventType, handle);
                }
                return function () {
                    for (var _i = 0, eventTypes_4 = eventTypes; _i < eventTypes_4.length; _i++) {
                        var eventType = eventTypes_4[_i];
                        window.removeEventListener(eventType, handle);
                    }
                };
            }
        }
        return noop_1.noop;
    }, [when, eventTypes, keyList, target, callback, handle]);
}
exports.useKey = useKey;
