"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.blockStyleFn = blockStyleFn;
exports.getSelectionInfo = exports.getSelection = exports.getSelectedLength = exports.getDecoratedValue = exports.getContentInfo = exports.getContent = exports.computeBlockType = void 0;
exports.hasBlockStyle = hasBlockStyle;
exports.hasInlineStyle = hasInlineStyle;
exports.isASCIIChar = isASCIIChar;
exports.moveSelectionToEnd = void 0;
exports.replaceText = replaceText;
exports.resetBlockType = void 0;
var _draftJs = require("draft-js");
var _decorators = _interopRequireDefault(require("../decorators"));
var _types = require("../../types");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const computeBlockType = (char, type) => {
  if (char === "." && type !== _types.ORDERED_LIST) {
    return _types.ORDERED_LIST;
  }
  if (char === "*" && type !== _types.UNORDERED_LIST) {
    return _types.UNORDERED_LIST;
  }
  return "unstyled";
};

/*
Returns default block-level metadata for various block type. Empty object otherwise.
*/
exports.computeBlockType = computeBlockType;
const getDefaultBlockData = (blockType, initialData = {}) => {
  switch (blockType) {
    case _types.ORDERED_LIST:
      return {};
    case _types.UNORDERED_LIST:
      return {};
    default:
      return initialData;
  }
};

/*
Changes the block type of the current block.
*/
const resetBlockType = (value, newType = "unstyled") => {
  const contentState = value.getCurrentContent();
  const selectionState = value.getSelection();
  const key = selectionState.getStartKey();
  const blockMap = contentState.getBlockMap();
  const block = blockMap.get(key);
  const newBlock = block.merge({
    text: "",
    type: newType,
    data: getDefaultBlockData(newType)
  });
  const newContentState = contentState.merge({
    blockMap: blockMap.set(key, newBlock),
    selectionAfter: selectionState.merge({
      anchorOffset: 0,
      focusOffset: 0
    })
  });
  return _draftJs.EditorState.push(value, newContentState, "change-block-type");
};
exports.resetBlockType = resetBlockType;
function blockStyleFn(block) {
  switch (block.getType()) {
    case "unordered-list-item":
      return "text-editor-block-unordered";
    case "ordered-list-item":
      return "text-editor-block-ordered";
    default:
      return "";
  }
}

/*
  Return mutated editorState with decorators added
*/
const getDecoratedValue = value => _draftJs.EditorState.set(value, {
  decorator: _decorators.default
});

/*
  Get the current Content State
*/
exports.getDecoratedValue = getDecoratedValue;
const getContent = value => value.getCurrentContent();

/*
  Get the current selection State
*/
exports.getContent = getContent;
const getSelection = value => value.getSelection();

/*
  Get the current Content and Block information
*/
exports.getSelection = getSelection;
const getContentInfo = value => {
  const content = getContent(value);
  const currentBlock = content.getBlockForKey(getSelection(value).getStartKey());
  const blockType = currentBlock.getType();
  const blockLength = currentBlock.getLength();
  const blockText = currentBlock.getText();
  const blockMap = content.getBlockMap();
  return {
    content,
    currentBlock,
    blockType,
    blockLength,
    blockText,
    blockMap
  };
};

/*
  Get the current Selection information
*/
exports.getContentInfo = getContentInfo;
const getSelectionInfo = value => {
  const selection = getSelection(value);
  const startKey = selection.getStartKey();
  const endKey = selection.getEndKey();
  const startOffset = selection.getStartOffset();
  const endOffset = selection.getEndOffset();
  return {
    selection,
    startKey,
    endKey,
    startOffset,
    endOffset
  };
};

/*
  Move cursor to end of Content
*/
exports.getSelectionInfo = getSelectionInfo;
const moveSelectionToEnd = value => _draftJs.EditorState.forceSelection(value, getContent(value).getSelectionAfter());

/*
  Returns the current Selection length
*/
exports.moveSelectionToEnd = moveSelectionToEnd;
const getSelectedLength = value => {
  const selection = getSelection(value);
  let length = 0;
  if (!selection.isCollapsed()) {
    const {
      startKey,
      endKey,
      startOffset,
      endOffset
    } = getSelectionInfo(value);
    const {
      content,
      blockLength
    } = getContentInfo(value);
    const startLength = blockLength - startOffset;
    const keyAfterEnd = content.getKeyAfter(endKey);
    if (startKey === endKey) {
      length += endOffset - startOffset;
    } else {
      let currentKey = startKey;
      while (currentKey && currentKey !== keyAfterEnd) {
        if (currentKey === startKey) {
          length += startLength + 1;
        } else if (currentKey === endKey) {
          length += endOffset;
        } else {
          length += content.getBlockForKey(currentKey).getLength() + 1;
        }
        currentKey = content.getKeyAfter(currentKey);
      }
    }
  }
  return length;
};
exports.getSelectedLength = getSelectedLength;
function hasBlockStyle(value, type) {
  const {
    blockType
  } = getContentInfo(value);
  return blockType === type;
}
function hasInlineStyle(value, style) {
  return value.getCurrentInlineStyle().has(style);
}
function isASCIIChar(str) {
  return /^\S+$/.test(str);
}
function replaceText(editorState, text, inlineStyle) {
  const contentState = _draftJs.Modifier.replaceText(editorState.getCurrentContent(), editorState.getSelection(), text, inlineStyle);
  return _draftJs.EditorState.push(editorState, contentState, "insert-characters");
}