"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
Object.defineProperty(exports, "Tab", {
  enumerable: true,
  get: function () {
    return _tab.default;
  }
});
exports.Tabs = void 0;
var _react = _interopRequireWildcard(require("react"));
var _propTypes = _interopRequireDefault(require("prop-types"));
var _tab = _interopRequireDefault(require("./tab"));
var _events = _interopRequireDefault(require("../../__internal__/utils/helpers/events"));
var _tags = _interopRequireDefault(require("../../__internal__/utils/helpers/tags/tags"));
var _tabs = _interopRequireDefault(require("./tabs.style"));
var _tabsHeader = _interopRequireDefault(require("./__internal__/tabs-header"));
var _tabTitle = _interopRequireDefault(require("./__internal__/tab-title"));
var _drawer = require("../drawer");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _extends() { _extends = Object.assign ? Object.assign.bind() : function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }
const Tabs = ({
  align = "left",
  className,
  children,
  onTabChange,
  selectedTabId,
  renderHiddenTabs = true,
  position = "top",
  extendedLine = true,
  size,
  borders = "off",
  variant = "default",
  validationStatusOverride,
  headerWidth,
  showValidationsSummary,
  ...rest
}) => {
  if (position !== "left" && headerWidth !== undefined) {
    // eslint-disable-next-line no-console
    console.error("Invalid usage of prop headerWidth in Tabs. The headerWidth can be used only if position is set to left");
  }

  /** The children nodes converted into an Array */
  const filteredChildren = (0, _react.useMemo)(() => _react.Children.toArray(children).filter(child => child), [children]);

  /** Array of the tabIds for the child nodes */
  const tabIds = (0, _react.useMemo)(() => filteredChildren.map(child => child.props.tabId), [filteredChildren]);

  /** Array of refs to the TabTitle nodes */
  const tabRefs = (0, _react.useMemo)(() => Array.from({
    length: filteredChildren.length
  }).map(() => /*#__PURE__*/(0, _react.createRef)()), [filteredChildren.length]);
  const previousSelectedTabId = (0, _react.useRef)(selectedTabId);
  const [selectedTabIdState, setSelectedTabIdState] = (0, _react.useState)(selectedTabId || filteredChildren[0].props.tabId);
  const {
    isInSidebar
  } = (0, _react.useContext)(_drawer.DrawerSidebarContext);
  const [tabsErrors, setTabsErrors] = (0, _react.useState)({});
  const [tabsWarnings, setTabsWarnings] = (0, _react.useState)({});
  const [tabsInfos, setTabsInfos] = (0, _react.useState)({});
  const updateErrors = (0, _react.useCallback)((id, error) => {
    setTabsErrors(state => ({
      ...state,
      [id]: error
    }));
  }, []);
  const updateWarnings = (0, _react.useCallback)((id, warning) => {
    setTabsWarnings(state => ({
      ...state,
      [id]: warning
    }));
  }, []);
  const updateInfos = (0, _react.useCallback)((id, info) => {
    setTabsInfos(state => ({
      ...state,
      [id]: info
    }));
  }, []);

  /** Returns true/false for if the given tab id is selected. */
  const isTabSelected = (0, _react.useCallback)(tabId => tabId === selectedTabIdState, [selectedTabIdState]);

  /** Updates the currently visible tab */
  const updateVisibleTab = (0, _react.useCallback)(tabid => {
    if (!isTabSelected(tabid)) {
      setSelectedTabIdState(tabid);
    }
    if (onTabChange) {
      onTabChange(tabid);
    }
  }, [onTabChange, isTabSelected]);
  const blurPreviousSelectedTab = (0, _react.useCallback)(() => {
    const {
      current
    } = previousSelectedTabId;
    const previousTabIndex = current ? tabIds.indexOf(current) : /* istanbul ignore next */-1;
    /* istanbul ignore else */
    if (previousTabIndex !== -1) {
      const previousTabRef = tabRefs[previousTabIndex];
      previousTabRef.current?.blur();
    }
  }, [tabIds, tabRefs]);
  (0, _react.useEffect)(() => {
    if (previousSelectedTabId.current !== selectedTabId) {
      if (selectedTabId !== selectedTabIdState) {
        setSelectedTabIdState(selectedTabId);
        blurPreviousSelectedTab();
      }
      previousSelectedTabId.current = selectedTabId;
    }
  }, [blurPreviousSelectedTab, previousSelectedTabId, selectedTabId, selectedTabIdState]);

  /** Handles the changing of tabs with the mouse */
  const handleTabClick = ev => {
    if (_events.default.isEventType(ev, "keydown")) {
      return;
    }
    const {
      tabid
    } = ev.target.dataset;
    updateVisibleTab(tabid);
  };

  /** Focuses the tab for the reference specified */
  const focusTab = ref => ref.current?.focus();

  /** Will trigger the tab at the given index. */
  const goToTab = (event, index) => {
    event.preventDefault();
    let newIndex = index;
    if (index < 0) {
      newIndex = tabIds.length - 1;
    } else if (index === tabIds.length) {
      newIndex = 0;
    }
    const nextRef = tabRefs[newIndex];
    focusTab(nextRef);
  };

  /** Handles the keyboard navigation of tabs */
  const handleKeyDown = index => {
    return event => {
      const isTabVertical = isInSidebar || position === "left";
      const isUp = isTabVertical && _events.default.isUpKey(event);
      const isDown = isTabVertical && _events.default.isDownKey(event);
      const isLeft = !isTabVertical && _events.default.isLeftKey(event);
      const isRight = !isTabVertical && _events.default.isRightKey(event);
      if (isUp || isLeft) {
        goToTab(event, index - 1);
      } else if (isDown || isRight) {
        goToTab(event, index + 1);
      }
    };
  };

  /** Build the headers for the tab component */
  const renderTabHeaders = () => {
    const tabTitles = filteredChildren.map((child, index) => {
      const {
        tabId,
        title,
        siblings,
        titlePosition,
        errorMessage,
        warningMessage,
        infoMessage,
        href,
        customLayout,
        titleProps
      } = child.props;
      const refId = `${tabId}-tab`;
      const errors = tabsErrors[tabId];
      const warnings = tabsWarnings[tabId];
      const infos = tabsInfos[tabId];
      const errorsCount = errors && Object.entries(errors).filter(tab => tab[1]).length;
      const warningsCount = warnings && Object.entries(warnings).filter(tab => tab[1]).length;
      const infosCount = infos && Object.entries(infos).filter(tab => tab[1]).length;
      const hasOverride = validationStatusOverride && validationStatusOverride[tabId];
      const errorOverride = hasOverride && validationStatusOverride[tabId].error;
      const warningOverride = hasOverride && validationStatusOverride[tabId].warning;
      const infoOverride = hasOverride && validationStatusOverride[tabId].info;
      const tabHasError = errorOverride !== undefined ? errorOverride : !!errorsCount;
      const tabHasWarning = warningOverride !== undefined ? warningOverride : !!warningsCount && !tabHasError;
      const tabHasInfo = infoOverride !== undefined ? infoOverride : !!infosCount && !tabHasError && !tabHasWarning;
      const getValidationMessage = (message, validations = {}) => {
        const summaryOfMessages = Object.values(validations).filter(value => value && typeof value === "string");
        if (!showValidationsSummary || !summaryOfMessages.length) {
          return message;
        }
        if (summaryOfMessages.length === 1) {
          return summaryOfMessages[0];
        }
        return summaryOfMessages.map(value => `• ${value}`).join("\n");
      };
      return /*#__PURE__*/_react.default.createElement(_tabTitle.default, _extends({}, titleProps, {
        position: isInSidebar ? "left" : position,
        className: child.props.className || "",
        dataTabId: tabId,
        id: refId,
        key: tabId,
        onClick: handleTabClick,
        onKeyDown: handleKeyDown(index),
        ref: tabRefs[index],
        tabIndex: isTabSelected(tabId) ? 0 : -1,
        title: title,
        href: href,
        isTabSelected: isTabSelected(tabId),
        error: tabHasError,
        warning: tabHasWarning,
        info: tabHasInfo,
        size: size || "default",
        borders: borders !== "off",
        siblings: siblings,
        titlePosition: titlePosition,
        errorMessage: getValidationMessage(errorMessage, errors),
        warningMessage: getValidationMessage(warningMessage, warnings),
        infoMessage: getValidationMessage(infoMessage, infos),
        alternateStyling: variant === "alternate",
        noLeftBorder: ["no left side", "no sides"].includes(borders),
        noRightBorder: ["no right side", "no sides"].includes(borders),
        customLayout: customLayout,
        isInSidebar: isInSidebar,
        align: align
      }));
    });
    return /*#__PURE__*/_react.default.createElement(_tabsHeader.default, {
      align: align,
      position: isInSidebar ? "left" : position,
      role: "tablist",
      extendedLine: extendedLine,
      noRightBorder: ["no right side", "no sides"].includes(borders),
      isInSidebar: isInSidebar
    }, tabTitles);
  };

  /** Builds all tabs where non selected tabs have class of hidden */
  const renderTabs = () => {
    if (isInSidebar) return null;
    if (!renderHiddenTabs) {
      const tab = filteredChildren.find(child => isTabSelected(child.props.tabId));
      return tab && /*#__PURE__*/(0, _react.cloneElement)(tab, {
        ...tab.props,
        role: "tabpanel",
        position,
        isTabSelected: isTabSelected(tab.props.tabId),
        key: `${tab.props.tabId}-tab`,
        ariaLabelledby: `${tab.props.tabId}-tab`,
        updateErrors,
        updateWarnings,
        updateInfos
      });
    }
    return filteredChildren.map(child => {
      return /*#__PURE__*/(0, _react.cloneElement)(child, {
        ...child.props,
        role: "tabpanel",
        position,
        isTabSelected: isTabSelected(child.props.tabId),
        key: `${child.props.tabId}-tab`,
        ariaLabelledby: `${child.props.tabId}-tab`,
        updateErrors,
        updateWarnings,
        updateInfos
      });
    });
  };
  return /*#__PURE__*/_react.default.createElement(_tabs.default, _extends({
    className: className,
    position: isInSidebar ? "left" : position
  }, (0, _tags.default)("tabs", rest), {
    isInSidebar: isInSidebar,
    headerWidth: headerWidth
  }, rest), renderTabHeaders(), renderTabs());
};
exports.Tabs = Tabs;