function _extends() { _extends = Object.assign ? Object.assign.bind() : function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }
import React, { useCallback, useEffect, useRef, useState, useContext } from "react";
import PropTypes from "prop-types";
import { ContentState, EditorState, RichUtils, getDefaultKeyBinding, Modifier, Editor } from "draft-js";
import { computeBlockType, getContent, getContentInfo, getDecoratedValue, getSelectedLength, moveSelectionToEnd, resetBlockType, isASCIIChar, replaceText, hasInlineStyle, hasBlockStyle, blockStyleFn } from "./__internal__/utils";
import { StyledEditorWrapper, StyledEditorOutline, StyledEditorContainer } from "./text-editor.style";
import ValidationWrapper from "./__internal__/editor-validation-wrapper";
import Toolbar from "./__internal__/toolbar";
import Label from "../../__internal__/label";
import Events from "../../__internal__/utils/helpers/events";
import guid from "../../__internal__/utils/helpers/guid";
import LabelWrapper from "./__internal__/label-wrapper";
import { BOLD, ITALIC, UNORDERED_LIST, ORDERED_LIST } from "./types";
import { NewValidationContext } from "../carbon-provider/carbon-provider.component";
import { ErrorBorder, StyledHintText } from "../textbox/textbox.style";
import ValidationMessage from "../../__internal__/validation-message";
import useInputAccessibility from "../../hooks/__internal__/useInputAccessibility";
import Box from "../box";
import useCharacterCount from "../../hooks/__internal__/useCharacterCount";
const NUMBERS = ["0", "1", "2", "3", "4", "5", "6", "7", "8", "9"];
const INLINE_STYLES = [BOLD, ITALIC];
export const EditorContext = /*#__PURE__*/React.createContext({});
const TextEditor = /*#__PURE__*/React.forwardRef(({
  characterLimit = 3000,
  labelText,
  onChange,
  value,
  required,
  error,
  warning,
  info,
  toolbarElements,
  rows,
  previews,
  onLinkAdded,
  inputHint,
  isOptional,
  ...rest
}, ref) => {
  const {
    validationRedesignOptIn
  } = useContext(NewValidationContext);
  const [isFocused, setIsFocused] = useState(false);
  const [inlines, setInlines] = useState([]);
  const [activeInlines, setActiveInlines] = useState({});
  const [focusToolbar, setFocusToolbar] = useState(false);
  const editorRef = useRef(null);
  const wrapper = useRef(null);
  const editor = ref || editorRef;
  const contentLength = getContent(value).getPlainText("").length;
  const moveCursor = useRef(contentLength > 0);
  const lastKeyPressed = useRef();
  const inputHintId = useRef(`${guid()}-hint`);
  const {
    current: id
  } = useRef(guid());
  const {
    labelId,
    validationId,
    ariaDescribedBy
  } = useInputAccessibility({
    id,
    validationRedesignOptIn,
    error,
    warning,
    info,
    label: labelText
  });
  const [characterCount, visuallyHiddenHintId] = useCharacterCount(getContent(value).getPlainText(""), characterLimit);
  const combinedAriaDescribedBy = [ariaDescribedBy, inputHint ? inputHintId.current : undefined, visuallyHiddenHintId].filter(Boolean).join(" ");
  if (rows && (typeof rows !== "number" || rows < 2)) {
    // eslint-disable-next-line no-console
    console.warn(`Prop rows must be a number value that is 2 or greater to override the min-height of the \`${TextEditor.displayName}\``);
  }
  const keyBindingFn = ev => {
    if (Events.isTabKey(ev) && !Events.isShiftKey(ev)) {
      setFocusToolbar(true);
    }
    return getDefaultKeyBinding(ev);
  };
  const BLOCK_TYPES = ["unordered-list-item", "ordered-list-item"];
  const handleKeyCommand = command => {
    // bail out if the enter is pressed and limit has been reached
    if (command.includes("split-block") && contentLength === characterLimit) {
      return "handled";
    }

    // if the backspace or enter is pressed get block type and text
    if (command.includes("backspace") || command.includes("split-block")) {
      const {
        blockLength,
        blockType
      } = getContentInfo(value);

      // if a block control is active and there is no text, deactivate it and reset the block
      if (BLOCK_TYPES.includes(blockType) && !blockLength) {
        onChange(resetBlockType(value, "unstyled"));
        return "handled";
      }
    }
    const style = command.toUpperCase();

    // if formatting shortcut used eg. command is "bold" or "italic"
    if (style === BOLD || style === ITALIC) {
      const update = RichUtils.handleKeyCommand(value, command);

      // istanbul ignore else
      if (update) {
        onChange(update);
        setActiveInlines({
          ...activeInlines,
          [style]: !hasInlineStyle(value, style)
        });
        return "handled";
      }
    }
    return "not-handled";
  };
  const handleBeforeInput = (str, newState) => {
    // short circuit if exceeds character limit
    if (contentLength >= characterLimit) {
      return "handled";
    }
    setActiveInlines({});

    // there is a bug in how DraftJS handles the macOS double-space-period feature, this is added to catch this and
    // prevent the editor from crashing until a fix can be added to their codebase
    if (lastKeyPressed.current === " " && !isASCIIChar(str)) {
      lastKeyPressed.current = null;
      onChange(replaceText(newState, " ", newState.getCurrentInlineStyle()));
      return "handled";
    }
    if (str === " ") {
      lastKeyPressed.current = str;
      return "not-handled";
    }
    lastKeyPressed.current = null;
    // short circuit if str does not match expected chars
    if (![".", "*"].includes(str)) {
      return "not-handled";
    }
    const {
      blockType,
      blockLength,
      blockText
    } = getContentInfo(value);
    if (blockLength === 1 && NUMBERS.includes(blockText) && str === "." || blockLength === 0 && str === "*") {
      const newBlockType = computeBlockType(str, blockType);
      const hasNumberList = hasBlockStyle(value, ORDERED_LIST);
      const hasBulletList = hasBlockStyle(value, UNORDERED_LIST);
      if (BLOCK_TYPES.includes(newBlockType) && !hasNumberList && !hasBulletList) {
        onChange(resetBlockType(value, newBlockType));
        setActiveInlines({
          BOLD: hasInlineStyle(value, BOLD),
          ITALIC: hasInlineStyle(value, ITALIC)
        });
        return "handled";
      }
    }
    onChange(value);
    return "not-handled";
  };
  const handlePastedText = pastedText => {
    const selectedTextLength = getSelectedLength(value);
    const newLength = contentLength + pastedText?.length - selectedTextLength;
    // if the pastedText will exceed the limit trim the excess
    if (newLength > characterLimit) {
      const newContentState = Modifier.insertText(getContent(value), value.getSelection(), pastedText.substring(0, characterLimit - contentLength));
      const newState = EditorState.push(value, newContentState, "insert-fragment");
      onChange(newState);
      return "handled";
    }
    setActiveInlines({});
    return "not-handled";
  };
  const getEditorState = () => {
    let editorState = getDecoratedValue(value);

    // should the cursor position be forced to the end of the content
    if (contentLength > 0 && moveCursor.current && isFocused) {
      editorState = moveSelectionToEnd(editorState);
      moveCursor.current = false;
    }
    return editorState;
  };
  const editorState = getEditorState();
  const activeControls = {
    BOLD: activeInlines.BOLD !== undefined ? activeInlines.BOLD : hasInlineStyle(editorState, BOLD),
    ITALIC: activeInlines.ITALIC !== undefined ? activeInlines.ITALIC : hasInlineStyle(editorState, ITALIC),
    "unordered-list-item": hasBlockStyle(editorState, UNORDERED_LIST),
    "ordered-list-item": hasBlockStyle(editorState, ORDERED_LIST)
  };
  const handleEditorFocus = useCallback(focusValue => {
    moveCursor.current = true;
    if (focusValue && typeof editor === "object" && editor.current !== document.activeElement) {
      editor.current?.focus();
      setFocusToolbar(false);
    }
    setIsFocused(focusValue);
  }, [editor]);
  const handleInlineStyleChange = (ev, style) => {
    ev.preventDefault();
    setActiveInlines({
      ...activeInlines,
      [style]: !hasInlineStyle(value, style)
    });
    handleEditorFocus(true);
    setInlines([...inlines, style]);
  };
  const handleBlockStyleChange = (ev, newBlockType) => {
    ev.preventDefault();
    handleEditorFocus(true);
    onChange(RichUtils.toggleBlockType(value, newBlockType));
    const temp = [];
    INLINE_STYLES.forEach(style => {
      if (activeInlines[style] !== undefined) {
        temp.push(style);
      }
    });
    setInlines(temp);
  };
  useEffect(() => {
    // apply the inline styling, having it run in as an effect ensures that styles can be added
    // even when the editor is not focused
    INLINE_STYLES.forEach(style => {
      const preserveStyle = activeInlines[style] !== undefined && activeInlines[style] !== hasInlineStyle(value, style);
      if (preserveStyle && value.getSelection().isCollapsed() || isFocused && inlines.includes(style)) {
        onChange(RichUtils.toggleInlineStyle(value, style));
        setInlines(inlines.filter(inline => inline !== style));
      }
      if (preserveStyle && !value.getSelection().isCollapsed()) {
        setActiveInlines({
          ...activeInlines,
          [style]: undefined
        });
      }
    });
  }, [activeInlines, contentLength, editorState, inlines, isFocused, onChange, value]);
  const handlePreviewClose = (onClose, url) => {
    // istanbul ignore else
    if (url) onClose(url);

    // istanbul ignore else
    if (typeof editor === "object") {
      editor.current?.focus();
    }
  };
  useEffect(() => {
    if (required) {
      const editableElement = wrapper.current?.querySelector("div[contenteditable='true']");
      editableElement?.setAttribute("required", "");
      editableElement?.setAttribute("aria-required", "true");
    }
  }, [required]);
  return /*#__PURE__*/React.createElement(EditorContext.Provider, {
    value: {
      onLinkAdded,
      editMode: true
    }
  }, /*#__PURE__*/React.createElement(StyledEditorWrapper, _extends({
    ref: wrapper
  }, rest), /*#__PURE__*/React.createElement(LabelWrapper, {
    onClick: () => handleEditorFocus(true)
  }, /*#__PURE__*/React.createElement(Label, {
    labelId: labelId,
    isRequired: required,
    optional: isOptional
  }, labelText)), inputHint && /*#__PURE__*/React.createElement(StyledHintText, {
    id: inputHintId.current
  }, inputHint), /*#__PURE__*/React.createElement(Box, {
    position: "relative"
  }, validationRedesignOptIn && /*#__PURE__*/React.createElement(React.Fragment, null, /*#__PURE__*/React.createElement(ValidationMessage, {
    error: error,
    validationId: validationId,
    warning: warning
  }), (error || warning) && /*#__PURE__*/React.createElement(ErrorBorder, {
    warning: !!(!error && warning)
  })), /*#__PURE__*/React.createElement(StyledEditorOutline, {
    isFocused: isFocused,
    hasError: !!error
  }, /*#__PURE__*/React.createElement(StyledEditorContainer, {
    "data-component": "text-editor-container",
    hasError: !!error,
    rows: rows,
    hasPreview: !!React.Children.count(previews)
  }, !validationRedesignOptIn && (error || warning || info) && /*#__PURE__*/React.createElement(ValidationWrapper, {
    error: error,
    warning: warning,
    info: info
  }), /*#__PURE__*/React.createElement(Editor, {
    ref: editor,
    onFocus: () => handleEditorFocus(true),
    onBlur: () => handleEditorFocus(false),
    editorState: editorState,
    onChange: onChange,
    handleBeforeInput: handleBeforeInput,
    handlePastedText: handlePastedText,
    handleKeyCommand: handleKeyCommand,
    ariaLabelledBy: labelId,
    ariaDescribedBy: combinedAriaDescribedBy,
    blockStyleFn: blockStyleFn,
    keyBindingFn: keyBindingFn,
    tabIndex: 0
  }), React.Children.map(previews, preview => {
    if ( /*#__PURE__*/React.isValidElement(preview)) {
      const {
        onClose
      } = preview?.props;
      return /*#__PURE__*/React.cloneElement(preview, {
        as: "div",
        onClose: onClose ? url => handlePreviewClose(onClose, url) : undefined
      });
    }
    return null;
  }), /*#__PURE__*/React.createElement(Toolbar, {
    setBlockStyle: (ev, newBlockType) => handleBlockStyleChange(ev, newBlockType),
    setInlineStyle: (ev, inlineStyle) => handleInlineStyleChange(ev, inlineStyle),
    activeControls: activeControls,
    canFocus: focusToolbar,
    toolbarElements: toolbarElements
  }))), characterCount)));
});
if (process.env.NODE_ENV !== "production") {
  TextEditor.propTypes = {
    "characterLimit": PropTypes.number,
    "error": PropTypes.string,
    "info": PropTypes.string,
    "inputHint": PropTypes.string,
    "isOptional": PropTypes.bool,
    "labelText": PropTypes.string.isRequired,
    "m": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "margin": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "marginBottom": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "marginLeft": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "marginRight": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "marginTop": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "marginX": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "marginY": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "mb": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "ml": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "mr": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "mt": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "mx": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "my": PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.oneOf([null]), PropTypes.number, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string])), PropTypes.number, PropTypes.object, PropTypes.shape({
      "__@toStringTag": PropTypes.string.isRequired,
      "description": PropTypes.string,
      "toString": PropTypes.func.isRequired,
      "valueOf": PropTypes.func.isRequired
    }), PropTypes.string]),
    "onChange": PropTypes.func.isRequired,
    "onLinkAdded": PropTypes.func,
    "previews": PropTypes.node,
    "required": PropTypes.bool,
    "rows": PropTypes.number,
    "toolbarElements": PropTypes.node,
    "value": PropTypes.object.isRequired,
    "warning": PropTypes.string
  };
}
export { TextEditor };
export const TextEditorState = EditorState;
export const TextEditorContentState = ContentState;
export default TextEditor;