function _extends() { _extends = Object.assign ? Object.assign.bind() : function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }
import React, { useEffect, useState, useCallback, useLayoutEffect, useRef, useMemo } from "react";
import PropTypes from "prop-types";
import { flip, offset, size } from "@floating-ui/dom";
import { useVirtualizer, defaultRangeExtractor } from "@tanstack/react-virtual";
import findLastIndex from "lodash/findLastIndex";
import usePrevious from "../../../hooks/__internal__/usePrevious";
import useScrollBlock from "../../../hooks/__internal__/useScrollBlock";
import useModalManager from "../../../hooks/__internal__/useModalManager";
import { StyledSelectList, StyledSelectLoaderContainer, StyledSelectListTable, StyledSelectListTableHeader, StyledSelectListTableBody, StyledSelectListContainer, StyledScrollableContainer } from "./select-list.style";
import Popover from "../../../__internal__/popover";
import OptionRow from "../option-row/option-row.component";
import getNextChildByText from "../utils/get-next-child-by-text";
import getNextIndexByKey from "../utils/get-next-index-by-key";
import isNavigationKey from "../utils/is-navigation-key";
import ListActionButton from "../list-action-button";
import { isEqual } from "lodash";
import Loader from "../../loader";
import Option from "../option";
import SelectListContext from "../__internal__/select-list-context";
const TABLE_HEADER_HEIGHT = 48;
const SCROLL_OPTIONS = {
  behavior: "auto",
  align: "end"
};
const SelectList = /*#__PURE__*/React.forwardRef(({
  listMaxHeight = 180,
  listActionButton,
  id,
  labelId,
  children,
  onSelect,
  onSelectListClose,
  filterText,
  anchorElement,
  highlightedValue,
  onListAction,
  isLoading,
  onListScrollBottom,
  multiColumn,
  tableHeader,
  loaderDataRole,
  listPlacement = "bottom",
  flipEnabled = true,
  isOpen,
  multiselectValues,
  enableVirtualScroll,
  virtualScrollOverscan = 5,
  ...listProps
}, listContainerRef) => {
  const [currentOptionsListIndex, setCurrentOptionsListIndex] = useState(-1);
  const [scrollbarWidth, setScrollbarWidth] = useState(0);
  const lastFilter = useRef("");
  const listRef = useRef(null);
  const tableRef = useRef(null);
  const listActionButtonRef = useRef(null);
  const {
    blockScroll,
    allowScroll
  } = useScrollBlock();
  const actionButtonHeight = useRef(0);
  const wasOpen = usePrevious(isOpen);

  // ensure scroll-position goes back to the top whenever the list is (re)-opened. (On Safari, without this it remains at the bottom if it had been scrolled
  // to the bottom before closing.)
  useEffect(() => {
    if (isOpen && !wasOpen) {
       listContainerRef?.current?.scrollTo?.(0, 0);
    }
  });
  const overscan = enableVirtualScroll ? virtualScrollOverscan : React.Children.count(children);

  // need to use a custom rangeExtractor that ensure the currently-selected option, if any, always appears as an option returned from the virtualiser.
  // This ensures that the aria-activedescendant value is always valid even when the currently-selected item is out of the visible range.
  const rangeExtractor = range => {
    const toRender = defaultRangeExtractor(range);
    if (currentOptionsListIndex !== -1 && !toRender.includes(currentOptionsListIndex)) {
      toRender.push(currentOptionsListIndex);
    }
    return toRender;
  };
  const virtualizer = useVirtualizer({
    count: React.Children.count(children),
    getScrollElement: () => listContainerRef.current,
    estimateSize: () => 40,
    // value doesn't really seem to matter since we're dynamically measuring, but 40px is the height of a single-line option
    overscan,
    paddingStart: multiColumn ? TABLE_HEADER_HEIGHT : 0,
    scrollPaddingEnd: actionButtonHeight.current,
    rangeExtractor
  });
  const items = virtualizer.getVirtualItems();
  const childrenList = useMemo(() => React.Children.toArray(children), [children]);

  // check if object values are equal
  function shallowEqual(objA, objB) {
    if(objA && objB && typeof objA === "object" && typeof objB === "object") {
        const keysA = Object.keys(objA);
        return keysA.every(key => objA[key] === objB[key]);
    }
    return isEqual(objA, objB);
    
    
  }
  const getIndexOfMatch = useCallback(valueToMatch => {
    return childrenList.findIndex(child => {
      if (child.props.value && typeof valueToMatch === "object") {
        return shallowEqual(child.props.value, valueToMatch);
      }
      return /*#__PURE__*/React.isValidElement(child) && child.props.value === valueToMatch;
    });
  }, [childrenList]);

  // getVirtualItems returns an empty array of items if the select list is currently closed - which is correct visually but
  // we need to ensure that any currently-selected option is still in the DOM to avoid any accessibility issues.
  // The isOpen prop will ensure that no options are visible regardless of what is in the items array.
  if (items.length === 0) {
    const currentIndex = highlightedValue ? getIndexOfMatch(highlightedValue) : currentOptionsListIndex;
    if (currentIndex > -1) {
      // only index property is required with the item not visible so the following type assertion, even though incorrect,
      // should be OK
      items.push({
        index: currentIndex
      });
    }
  }
  const totalSize = virtualizer.getTotalSize();
  const listHeight = totalSize === 0 ? 1 : totalSize;
  useEffect(() => {
    if (isOpen) {
      blockScroll();
    }
    return () => {
      if (isOpen) {
        allowScroll();
      }
    };
  }, [allowScroll, blockScroll, isOpen]);
  useLayoutEffect(() => {
    if (multiColumn) {
      setScrollbarWidth(tableRef.current ? tableRef.current.offsetWidth - tableRef.current.clientWidth : /* istanbul ignore next */0);
    }
  }, [multiColumn]);
  const anchorRef = useMemo(() => ({
    current: anchorElement || null
  }), [anchorElement]);
  const handleSelect = useCallback(optionData => {
    onSelect({
      ...optionData,
      selectionType: "click",
      selectionConfirmed: true
    });
  }, [onSelect]);
  const childElementRefs = useRef(Array.from({
    length: React.Children.count(children)
  }));
  const optionChildrenList = useMemo(() => childrenList.filter(child => /*#__PURE__*/React.isValidElement(child) && (child.type === Option || child.type === OptionRow)), [childrenList]);
  const {
    measureElement
  } = virtualizer;
  const measureCallback = element => {
    // need a guard to prevent crash with too many rerenders when closing the list
    if (isOpen) {
      measureElement(element);
    }
  };

  // the rangeExtractor above can cause an undefined value to be appended to the return items.
  // Easiest way to stop that crashing is just to filter it out.
  const renderedChildren = items.filter(item => item !== undefined).map(({
    index,
    start
  }) => {
    const child = childrenList[index];
    if (! /*#__PURE__*/React.isValidElement(child)) {
      return child;
    }
    const optionChildIndex = optionChildrenList.indexOf(child);
    const isOption = optionChildIndex > -1;
    const newProps = {
      index,
      onSelect: handleSelect,
      hidden: isLoading && childrenList.length === 1,
      // these need to be inline styles rather than implemented in styled-components to avoid it generating thousands of classes
      style: {
        transform: `translateY(${start}px)`
      },
      "aria-setsize": isOption ? optionChildrenList.length : undefined,
      "aria-posinset": isOption ? optionChildIndex + 1 : undefined,
      ref: optionElement => {
        // needed to dynamically compute the size
        measureCallback(optionElement);
        // add the DOM element to the array of refs
        childElementRefs.current[index] = optionElement;
      },
      "data-index": index
    };
    return /*#__PURE__*/React.cloneElement(child, newProps);
  });
  const lastOptionIndex = findLastIndex(childrenList, child => /*#__PURE__*/React.isValidElement(child) && (child.type === Option || child.type === OptionRow));
  const getNextHighlightableItemIndex = useCallback((key, indexOfHighlighted) => {
    const lastIndex = lastOptionIndex;
    if (lastIndex === -1) {
      return -1;
    }
    let nextIndex = getNextIndexByKey(key, indexOfHighlighted, lastIndex, isLoading);
    const nextElement = childrenList[nextIndex];
    if ( /*#__PURE__*/React.isValidElement(nextElement) && nextElement.type !== Option && nextElement.type !== OptionRow || nextElement.props.disabled) {
      nextIndex = getNextHighlightableItemIndex(key, nextIndex);
    }
    return nextIndex;
  }, [childrenList, lastOptionIndex, isLoading]);
  const highlightNextItem = useCallback(key => {
    let currentIndex = currentOptionsListIndex;
    if (highlightedValue) {
      const indexOfHighlighted = getIndexOfMatch(highlightedValue);
      currentIndex = indexOfHighlighted;
    }
    const nextIndex = getNextHighlightableItemIndex(key, currentIndex);
    if (nextIndex === -1 || currentIndex === nextIndex) {
      return;
    }
    const {
      text,
      value
    } = childrenList[nextIndex].props;
    onSelect({
      text,
      value,
      selectionType: "navigationKey",
      id: childElementRefs.current[nextIndex]?.id,
      selectionConfirmed: false
    });
  }, [childrenList, currentOptionsListIndex, getIndexOfMatch, getNextHighlightableItemIndex, highlightedValue, onSelect]);
  const handleActionButtonTab = useCallback((event, isActionButtonFocused) => {
    if (isActionButtonFocused) {
      onSelect({
        selectionType: "tab",
        selectionConfirmed: false
      });
    } else {
      event.preventDefault();
      listActionButtonRef.current?.focus();
    }
  }, [onSelect]);
  const focusOnAnchor = useCallback(() => {
    if (anchorElement) {
      anchorElement.getElementsByTagName("input")[0].focus();
    }
  }, [anchorElement]);
  const handleGlobalKeydown = useCallback(event => {
    if (!isOpen) {
      return;
    }
    const {
      key
    } = event;
    const isActionButtonFocused = document.activeElement === listActionButtonRef.current;
    if (key === "Tab" && listActionButton) {
      handleActionButtonTab(event, isActionButtonFocused);
    } else if (key === "Tab") {
      onSelectListClose();
    } else if (key === "Enter" && !isActionButtonFocused) {
      event.preventDefault();
      const currentOption = childrenList[currentOptionsListIndex];
      if (! /*#__PURE__*/React.isValidElement(currentOption)) {
        onSelectListClose();

        // need to call onSelect here with empty text/value to clear the input when
        // no matches found in FilterableSelect
        onSelect({
          id: undefined,
          text: "",
          value: "",
          selectionType: "enterKey",
          selectionConfirmed: false
        });
        return;
      }
      if (currentOption.props.disabled) {
        return;
      }
      const {
        text,
        value
      } = currentOption.props;
      onSelect({
        id: childElementRefs.current[currentOptionsListIndex]?.id,
        text,
        value,
        selectionType: "enterKey",
        selectionConfirmed: true
      });
    } else if (isNavigationKey(key)) {
      focusOnAnchor();
      highlightNextItem(key);
    }
  }, [childrenList, listActionButton, handleActionButtonTab, onSelectListClose, currentOptionsListIndex, onSelect, highlightNextItem, focusOnAnchor, isOpen]);
  const handleEscapeKey = useCallback(event => {
    if (event.key === "Escape") {
      onSelectListClose();
    }
  }, [onSelectListClose]);
  useModalManager({
    open: !!isOpen,
    closeModal: handleEscapeKey,
    modalRef: listRef,
    triggerRefocusOnClose: false
  });
  const handleListScroll = useCallback(event => {
    const element = event.target;

    /* istanbul ignore else */
    if (isOpen && onListScrollBottom && element.scrollHeight - element.scrollTop === element.clientHeight) {
      onListScrollBottom();
    }
  }, [onListScrollBottom, isOpen]);
  useEffect(() => {
    const keyboardEvent = "keydown";
    const listElement = listContainerRef.current;
    window.addEventListener(keyboardEvent, handleGlobalKeydown);
    listElement?.addEventListener("scroll", handleListScroll);
    return function cleanup() {
      window.removeEventListener(keyboardEvent, handleGlobalKeydown);
      listElement?.removeEventListener("scroll", handleListScroll);
    };
  }, [handleGlobalKeydown, handleListScroll, listContainerRef]);
  useEffect(() => {
    if (!filterText || filterText === lastFilter.current) {
      lastFilter.current = filterText || "";
      return;
    }
    lastFilter.current = filterText;
    setCurrentOptionsListIndex(previousIndex => {
      const match = getNextChildByText(filterText, childrenList, previousIndex);
      if (!match) {
        return -1;
      }
      const indexOfMatch = getIndexOfMatch(match.props.value);
      virtualizer.scrollToIndex(indexOfMatch, SCROLL_OPTIONS);
      return indexOfMatch;
    });
  }, [childrenList, filterText, getIndexOfMatch, virtualizer]);
  useEffect(() => {
    // remove the current selected option if the value is cleared
    // this prevents it from remaining highlighted when the list is re-opened
    if ((!highlightedValue || Object.keys(highlightedValue).length === 0) && !isOpen) {
      setCurrentOptionsListIndex(-1);
      return;
    }
    const indexOfMatch = getIndexOfMatch(highlightedValue);
    if (indexOfMatch === -1) {
      return;
    }
    setCurrentOptionsListIndex(indexOfMatch);
  }, [getIndexOfMatch, highlightedValue, isOpen]);

  // ensure that the currently-selected option is always visible immediately after
  // it has been changed
  useEffect(() => {
    if (currentOptionsListIndex > -1) {
      virtualizer.scrollToIndex(currentOptionsListIndex, SCROLL_OPTIONS);
    }
  }, [currentOptionsListIndex, virtualizer]);
  useEffect(() => {
    if (isLoading && currentOptionsListIndex === lastOptionIndex && lastOptionIndex > -1) {
      virtualizer.scrollToIndex(lastOptionIndex, {
        ...SCROLL_OPTIONS,
        align: "start"
      });
    }
  }, [children, currentOptionsListIndex, isLoading, lastOptionIndex, listContainerRef, virtualizer]);
  const popoverMiddleware = useMemo(() => [offset(3), size({
    apply({
      rects,
      elements
    }) {
      Object.assign(elements.floating.style, {
        width: `${rects.reference.width}px`
      });
    }
  }), ...(flipEnabled ? [flip({
    fallbackStrategy: "initialPlacement"
  })] : [])], [flipEnabled]);
  const loader = isLoading ? /*#__PURE__*/React.createElement(StyledSelectLoaderContainer, {
    key: "loader"
  }, /*#__PURE__*/React.createElement(Loader, {
    "data-role": loaderDataRole
  })) : undefined;
  let selectListContent = renderedChildren;
  const listBoxProps = {
    role: "listbox",
    id,
    "aria-labelledby": labelId,
    "aria-multiselectable": multiselectValues ? true : undefined
  };
  useLayoutEffect(() => {
    if (listActionButton && isOpen) {
      actionButtonHeight.current = listActionButtonRef.current?.parentElement?.offsetHeight || 0;
    }
  }, [listActionButton, isOpen]);
  if (multiColumn) {
    selectListContent = /*#__PURE__*/React.createElement(StyledSelectListTable, null, /*#__PURE__*/React.createElement(StyledSelectListTableHeader, {
      scrollbarWidth: scrollbarWidth
    }, tableHeader), /*#__PURE__*/React.createElement(StyledSelectListTableBody, _extends({}, listBoxProps, {
      "aria-labelledby": labelId,
      ref: tableRef,
      listHeight: listHeight - TABLE_HEADER_HEIGHT
    }), renderedChildren));
  }
  return /*#__PURE__*/React.createElement(SelectListContext.Provider, {
    value: {
      currentOptionsListIndex,
      multiselectValues
    }
  }, /*#__PURE__*/React.createElement(Popover, {
    placement: listPlacement,
    disablePortal: true,
    reference: anchorRef,
    middleware: popoverMiddleware,
    isOpen: isOpen,
    disableBackgroundUI: true,
    animationFrame: true
  }, /*#__PURE__*/React.createElement(StyledSelectListContainer, _extends({
    "data-element": "select-list-wrapper",
    isLoading: isLoading
  }, listProps), /*#__PURE__*/React.createElement(StyledScrollableContainer, {
    ref: listContainerRef,
    maxHeight: listMaxHeight,
    "data-component": "select-list-scrollable-container",
    hasActionButton: !!listActionButton
  }, /*#__PURE__*/React.createElement(StyledSelectList, _extends({
    as: multiColumn ? "div" : "ul",
    "data-element": "select-list"
  }, multiColumn ? {} : listBoxProps, {
    ref: listRef,
    tabIndex: -1,
    listHeight: multiColumn ? undefined : listHeight
  }), selectListContent), loader), listActionButton && /*#__PURE__*/React.createElement(ListActionButton, {
    ref: listActionButtonRef,
    listActionButton: listActionButton,
    onListAction: onListAction
  }))));
});
if (process.env.NODE_ENV !== "production") {
  SelectList.propTypes = {
    "anchorElement": function (props, propName) {
      if (props[propName] == null) {
        return null;
      } else if (typeof props[propName] !== 'object' || props[propName].nodeType !== 1) {
        return new Error("Expected prop '" + propName + "' to be of type Element");
      }
    },
    "children": PropTypes.node,
    "enableVirtualScroll": PropTypes.bool,
    "filterText": PropTypes.string,
    "flipEnabled": PropTypes.bool,
    "highlightedValue": PropTypes.oneOfType([PropTypes.object, PropTypes.string]),
    "id": PropTypes.string,
    "isLoading": PropTypes.bool,
    "isOpen": PropTypes.bool,
    "labelId": PropTypes.string,
    "listActionButton": PropTypes.node,
    "listMaxHeight": PropTypes.number,
    "listPlacement": PropTypes.oneOf(["bottom", "left", "right", "top"]),
    "loaderDataRole": PropTypes.string,
    "multiColumn": PropTypes.bool,
    "multiselectValues": PropTypes.arrayOf(PropTypes.string),
    "onListAction": PropTypes.func,
    "onListScrollBottom": PropTypes.func,
    "onMouseDown": PropTypes.func,
    "onSelect": PropTypes.func.isRequired,
    "onSelectListClose": PropTypes.func.isRequired,
    "tableHeader": PropTypes.node,
    "virtualScrollOverscan": PropTypes.number
  };
}
export default SelectList;