import React, { useState, useRef, useEffect, useMemo, useCallback } from "react";
import PropTypes from "prop-types";
import { isFragment } from "react-is";
import invariant from "invariant";
import throttle from "lodash/throttle";
import { defaultFocusableSelectors } from "../../__internal__/focus-trap/focus-trap-utils";
import Event from "../../__internal__/utils/helpers/events";
import { StyledAnchorNavigation, StyledNavigation, StyledContent } from "./anchor-navigation.style";
import AnchorNavigationItem from "./anchor-navigation-item/anchor-navigation-item.component";
const SECTION_VISIBILITY_OFFSET = 200;
const SCROLL_THROTTLE = 100;
const AnchorNavigation = ({
  children,
  stickyNavigation
}) => {
  !isFragment(stickyNavigation) ? process.env.NODE_ENV !== "production" ? invariant(false, "`stickyNavigation` prop in `AnchorNavigation` should be a React Fragment.") : invariant(false) : void 0;
  const hasCorrectItemStructure = useMemo(() => {
    const incorrectChild = React.Children.toArray(stickyNavigation.props.children).find(child => {
      return ! /*#__PURE__*/React.isValidElement(child) || child.type.displayName !== AnchorNavigationItem.displayName;
    });
    return !incorrectChild;
  }, [stickyNavigation]);
  !hasCorrectItemStructure ? process.env.NODE_ENV !== "production" ? invariant(false, `\`stickyNavigation\` prop in \`AnchorNavigation\` should be a React Fragment that only contains children of type \`${AnchorNavigationItem.displayName}\``) : invariant(false) : void 0;
  const [selectedIndex, setSelectedIndex] = useState(0);
  const sectionRefs = useRef(React.Children.map(stickyNavigation.props.children, child => child.props.target));
  const contentRef = useRef(null);
  const navigationRef = useRef(null);
  const isUserScroll = useRef(true);
  const isUserScrollTimer = useRef();
  const setSelectedAnchorBasedOnScroll = useCallback(() => {
    // istanbul ignore if
    // function is called only after component is rendered, so ref cannot hold a null value
    if (navigationRef.current === null) return;
    const offsetsWithIndexes = sectionRefs.current.map(({
      current
    }, index) => [index, current?.getBoundingClientRect().top]).filter(offsetWithIndex => offsetWithIndex[1] !== undefined);
    const {
      top: navTopOffset
    } = navigationRef.current.getBoundingClientRect();
    const indexOfSmallestNegativeTopOffset = offsetsWithIndexes.reduce((currentTopIndex, offsetWithIndex) => {
      const [index, offset] = offsetWithIndex;
      if (offset - SECTION_VISIBILITY_OFFSET > navTopOffset) return currentTopIndex;
      return offset > offsetsWithIndexes[currentTopIndex][1] ? index : currentTopIndex;
    }, offsetsWithIndexes[0][0]);
    setSelectedIndex(indexOfSmallestNegativeTopOffset);
  }, []);
  const scrollHandler = useMemo(() => throttle(() => {
    if (isUserScroll.current) {
      setSelectedAnchorBasedOnScroll();
    } else {
      if (isUserScrollTimer.current !== undefined) {
        window.clearTimeout(isUserScrollTimer.current);
      }
      isUserScrollTimer.current = setTimeout( /* istanbul ignore next */() => {
        isUserScroll.current = true;
      }, SCROLL_THROTTLE + 50);
    }
  }, SCROLL_THROTTLE), [setSelectedAnchorBasedOnScroll]);
  useEffect(() => {
    window.addEventListener("scroll", scrollHandler, true);
    return () => window.removeEventListener("scroll", scrollHandler, true);
  }, [scrollHandler]);
  const focusSection = section => {
    if (!section.matches(defaultFocusableSelectors)) {
      section.setAttribute("tabindex", "-1");
    }
    section.focus({
      preventScroll: true
    });
  };
  const scrollToSection = index => {
    const sectionToScroll = sectionRefs.current[index].current;

    // istanbul ignore if
    // function is called only after component is rendered, so ref cannot hold a null value
    if (sectionToScroll === null) return;

    // ensure section has the appropriate element to remove the default focus styles.
    // Can ignore else branch because there's no harm in setting this to "true" twice (it can't hold any other value),
    // but it's probably more efficient not to.
    // istanbul ignore else
    if (!sectionToScroll.dataset.carbonAnchornavRef) {
      sectionToScroll.dataset.carbonAnchornavRef = "true";
    }
    focusSection(sectionToScroll);

    // workaround due to preventScroll focus method option on firefox not working consistently
    window.setTimeout(() => {
      isUserScroll.current = false;
      sectionToScroll.scrollIntoView({
        block: "start",
        inline: "nearest",
        behavior: "smooth"
      });
      setSelectedIndex(index);
    }, 10);
  };
  const handleClick = (event, index) => {
    event.preventDefault();
    scrollToSection(index);
  };
  const handleKeyDown = (event, index) => {
    if (Event.isEnterKey(event)) {
      scrollToSection(index);
    }
  };
  return /*#__PURE__*/React.createElement(StyledAnchorNavigation, {
    ref: contentRef,
    "data-component": "anchor-navigation"
  }, /*#__PURE__*/React.createElement(StyledNavigation, {
    ref: navigationRef,
    "data-element": "anchor-sticky-navigation"
  }, React.Children.map(stickyNavigation.props.children, (child, index) => /*#__PURE__*/React.cloneElement(child, {
    href: child.props.href || "#",
    // need to pass an href to ensure the link is tabbable by default
    isSelected: index === selectedIndex,
    onClick: event => handleClick(event, index),
    onKeyDown: event => handleKeyDown(event, index)
  }))), /*#__PURE__*/React.createElement(StyledContent, null, children));
};
if (process.env.NODE_ENV !== "production") {
  AnchorNavigation.propTypes = {
    "children": PropTypes.node,
    "stickyNavigation": PropTypes.node
  };
}
AnchorNavigation.displayName = "AnchorNavigation";
export default AnchorNavigation;